// Dynamic import for ES Module
let makeWASocket, DisconnectReason, useMultiFileAuthState, downloadMediaMessage, fetchLatestBaileysVersion;
let baileysLoaded = false;

// Load Baileys module dynamically
async function loadBaileys() {
  if (baileysLoaded) return;
  
  try {
    const baileys = await import('@whiskeysockets/baileys');
    makeWASocket = baileys.default;
    DisconnectReason = baileys.DisconnectReason;
    useMultiFileAuthState = baileys.useMultiFileAuthState;
    downloadMediaMessage = baileys.downloadMediaMessage;
    fetchLatestBaileysVersion = baileys.fetchLatestBaileysVersion;
    baileysLoaded = true;
    console.log('✅ Baileys module loaded successfully');
  } catch (error) {
    console.error('❌ Error loading Baileys module:', error);
    throw error;
  }
}

const pino = require('pino');
const qrcode = require('qrcode');
const fs = require('fs');
const path = require('path');
const mime = require('mime-types');
const { pool } = require('../config/database');
const webpush = require('web-push'); 
let aiService = null;

// Import aiService unconditionally. OpenAI activation/deactivation logic
// is handled internally by aiService based on process.env.ENABLE_OPENAI.
  try {
    aiService = require('./aiService'); // aiService.js already exports an instance
  } catch (e) {
    console.error('Error loading aiService:', e.message);
    // In case of critical error, create a stub to avoid TypeError.
    aiService = { processMessage: async () => ({ shouldRespond: false, success: false, response: 'AI service unavailable.' }) };
  }


// Configure web-push with VAPID keys from .env
if (process.env.VAPID_PUBLIC_KEY && process.env.VAPID_PRIVATE_KEY) {
    webpush.setVapidDetails(
        'mailto:seu-email@exemplo.com.br', // Replace with your email
        process.env.VAPID_PUBLIC_KEY,
        process.env.VAPID_PRIVATE_KEY
    );
    console.log('✅ Web Push configured successfully.');
} else {
    console.warn('⚠️ VAPID keys not found. Push notifications are disabled.');
}

class WhatsAppService {
  constructor(io, enableOpenAI) {
    this.enableOpenAI = enableOpenAI;
    this.io = io;
    this.sock = null;
    this.qrCode = null;
    this.isConnected = false;
    this.sessionPath = path.join(__dirname, '../sessions');
    this.reconnectAttempts = 0;
    this.maxReconnectAttempts = 5;
    this.retryTimeout = null;
    this.isInitializing = false;
    this.shouldReconnect = true;
    this.qrGenerationCount = 0;
    this.maxQrGenerations = 3;
  }
  

  /**
   * Remove a push subscription from the database.
   * @param {string} subscriptionJson The complete JSON string of the subscription to be removed.
   */
  async deleteSubscription(subscriptionJson) {
    try {
      console.log(`[PUSH] Removing invalid subscription from database...`);
      await pool.execute('DELETE FROM push_subscriptions WHERE subscription = ?', [subscriptionJson]);
      console.log(`[PUSH] ✅ Subscription removed successfully.`);
    } catch (dbError) {
      console.error(`[PUSH] ❌ Error trying to remove subscription from database:`, dbError);
    }
  }

  /**
   * Sends push notifications to users associated with a conversation.
   * @param {object} conversation The conversation object that just received a message.
   */
  async sendPushNotification(conversation) {
    console.log('--- [PUSH] Starting notification sending process ---');

    // Check if VAPID keys are configured before proceeding
    if (!process.env.VAPID_PUBLIC_KEY) {
      console.log('[PUSH] ⚠️ VAPID_PUBLIC_KEY not configured. Notifications disabled.');
      return;
    }

    try {
      // Determine which users should receive the notification
      let userIds = [];
      if (conversation.assigned_user_id) {
        userIds.push(conversation.assigned_user_id);
        console.log(`[PUSH] Conversation assigned to user ID: ${conversation.assigned_user_id}`);
      } else if (conversation.assigned_store) {
        console.log(`[PUSH] Fetching users from store: "${conversation.assigned_store}"`);
        const [users] = await pool.execute('SELECT id FROM users WHERE store = ?', [conversation.assigned_store]);
        userIds = users.map(u => u.id);
      } else {
        console.log('[PUSH] ⚠️ Conversation without assigned store, notifying all users');
        const [users] = await pool.execute('SELECT id FROM users WHERE role = "user"');
        userIds = users.map(u => u.id);
      }

      if (userIds.length === 0) {
        console.log('[PUSH] ⚠️ No users found to notify.');
        return;
      }
      console.log(`[PUSH] Users to be notified (IDs): ${userIds.join(', ')}`);

      // Fetch all subscriptions (devices) of found users from database
      const [subscriptions] = await pool.query(
        'SELECT subscription FROM push_subscriptions WHERE user_id IN (?)',
        [userIds]
      );

      if (subscriptions.length === 0) {
        console.log('[PUSH] ⚠️ No push subscriptions found for these users.');
        return;
      }
      console.log(`[PUSH] Found ${subscriptions.length} subscriptions to notify.`);

      // Prepare notification content (payload)
      const payload = JSON.stringify({
        title: `New Message from ${conversation.phone_number}`,
        body: conversation.last_message.substring(0, 100),
        url: `/user` // URL where user will be redirected when clicking
      });

      // Map each subscription to a notification sending promise
      const sendPromises = subscriptions.map(record => {
        const parsedSubscription = JSON.parse(record.subscription);
        
        return webpush.sendNotification(parsedSubscription, payload)
          .catch(error => {
            // If sending fails, .catch is triggered
            if (error.statusCode === 410 || error.statusCode === 404) {
              // If error is 410 (Gone) or 404 (Not Found), subscription is invalid.
              console.warn(`[PUSH] Invalid subscription (code ${error.statusCode}). Triggering removal.`);
              // Call our new function to delete subscription from database.
              this.deleteSubscription(record.subscription); 
            } else {
              // For any other type of error, just log it.
              console.error('[PUSH] ❌ An error occurred sending notification:', error.body || error);
            }
          });
      });

      // Wait for all sending attempts (whether successful or failed) to finish.
      await Promise.all(sendPromises);
      console.log(`[PUSH] Notification sending process completed for ${subscriptions.length} device(s).`);

    } catch (error) {
      // Catch any general error that may occur in the process (e.g., database connection failure)
      console.error('❌ GENERAL error in push notification sending process:', error);
    }
  }
//================================================================================
// END OF CODE BLOCK
//================================================================================

  async initialize() {
    if (this.isInitializing) {
      console.log('⚠️ Initialization already in progress, ignoring...');
      return;
    }

    try {
      this.isInitializing = true;
      console.log('🔄 Starting WhatsApp Service...');
      
      // Load Baileys module first
      await loadBaileys();
      
      // Clear previous timeout if exists
      if (this.retryTimeout) {
        clearTimeout(this.retryTimeout);
        this.retryTimeout = null;
      }

      // Create sessions directory if it doesn't exist
      if (!fs.existsSync(this.sessionPath)) {
        fs.mkdirSync(this.sessionPath, { recursive: true });
        console.log('📁 Sessions directory created');
      }

      // Fetch latest Baileys version
      const { version, isLatest } = await fetchLatestBaileysVersion();
      console.log(`📱 Using Baileys version ${version.join('.')}, is latest: ${isLatest}`);

      // Configure authentication
      const { state, saveCreds } = await useMultiFileAuthState(this.sessionPath);

      // Close previous socket if exists
      if (this.sock) {
        try {
          this.sock.ev.removeAllListeners();
          this.sock.end();
        } catch (err) {
          console.log('Error closing previous socket:', err.message);
        }
        this.sock = null;
      }

      // Create socket
      this.sock = makeWASocket({
        version,
        auth: state,
        logger: pino({ level: 'error' }), // Reduce logs even more
        browser: ["AI Chatbot", "Chrome", "1.0.0"],
        keepAliveIntervalMs: 60000,
        connectTimeoutMs: 60000,
        defaultQueryTimeoutMs: 60000,
        markOnlineOnConnect: false, // Avoid marking as online immediately
        syncFullHistory: false,
        fireInitQueries: false, // Don't make unnecessary queries
        generateHighQualityLinkPreview: false,
        printQRInTerminal: false,
        shouldSyncHistoryMessage: () => false,
        getMessage: async (key) => {
          return { conversation: "" };
        }
      });

      this.setupEventHandlers();
      this.sock.ev.on('creds.update', saveCreds);

      console.log('✅ WhatsApp Service configured');

    } catch (error) {
      console.error('❌ Error initializing WhatsApp:', error);
      this.handleError(error);
    } finally {
      this.isInitializing = false;
    }
  }

  setupEventHandlers() {
    if (!this.sock) return;

    // Handler for connection updates
    this.sock.ev.on('connection.update', async (update) => {
      try {
        await this.handleConnectionUpdate(update);
      } catch (error) {
        console.error('Error in connection handler:', error);
      }
    });

    // Handler for messages
    this.sock.ev.on('messages.upsert', async (m) => {
      try {
        await this.handleMessage(m);
      } catch (error) {
        console.error('Error processing message:', error);
      }
    });

    // Handler for errors
    this.sock.ev.on('connection.close', (reason) => {
      console.log('🔴 Connection closed:', reason);
    });
  }

  async handleConnectionUpdate(update) {
    const { connection, lastDisconnect, qr } = update;

    // Generate QR Code with limit
    if (qr) {
      if (this.qrGenerationCount >= this.maxQrGenerations) {
        console.log('⚠️ QR code limit reached, forcing new session...');
        await this.clearSession();
        setTimeout(() => {
          if (this.shouldReconnect) {
            this.qrGenerationCount = 0;
            this.initialize();
          }
        }, 5000);
        return;
      }

      try {
        this.qrGenerationCount++;
        console.log(`📱 Generating QR Code (${this.qrGenerationCount}/${this.maxQrGenerations})...`);
        
        this.qrCode = await qrcode.toDataURL(qr, {
          width: 300,
          margin: 2,
          color: {
            dark: '#000000',
            light: '#FFFFFF'
          }
        });
        
        console.log('✅ QR Code generated successfully!');
        this.io.emit('qr-code', this.qrCode);
      } catch (error) {
        console.error('❌ Error generating QR Code:', error);
        this.io.emit('qr-code', null);
      }
    }

    // Connection status
    if (connection === 'close') {
      await this.handleDisconnection(lastDisconnect);
    } else if (connection === 'open') {
      await this.handleConnection();
    } else if (connection === 'connecting') {
      console.log('🔄 Connecting to WhatsApp...');
      this.io.emit('connection-status', { connected: false, connecting: true });
    }
  }

  async handleConnection() {
    console.log('✅ Connected to WhatsApp successfully!');
    this.isConnected = true;
    this.qrCode = null;
    this.reconnectAttempts = 0;
    this.qrGenerationCount = 0;
    this.isInitializing = false;
    
    // Emit status to clients
    this.io.emit('connection-status', { connected: true });
    this.io.emit('qr-code', null);

    // Wait a bit before marking as ready
    setTimeout(() => {
      console.log('📱 WhatsApp is ready for use!');
    }, 2000);
  }

  async handleDisconnection(lastDisconnect) {
    const reason = lastDisconnect?.error?.output?.statusCode;
    console.log('🔌 Connection closed. Code:', reason);
    
    this.isConnected = false;
    this.isInitializing = false;
    
    // Emit disconnection status
    this.io.emit('connection-status', { 
      connected: false, 
      error: this.getDisconnectReason(reason) 
    });

    if (!this.shouldReconnect) {
      console.log('🛑 Reconnection disabled');
      return;
    }

    const shouldReconnect = reason !== DisconnectReason.loggedOut;
    
    if (reason === DisconnectReason.loggedOut) {
      console.log('💤 User logged out. Clearing session...');
      await this.clearSession();
    } else if (reason === DisconnectReason.connectionClosed) {
      console.log('🔌 Connection lost. Attempting to reconnect...');
      await this.scheduleReconnection();
    } else if (reason === DisconnectReason.connectionLost) {
      console.log('📡 Connection lost. Attempting to reconnect...');
      await this.scheduleReconnection();
    } else if (reason === DisconnectReason.restartRequired) {
      console.log('🔄 Restart required...');
      await this.scheduleReconnection();
    } else if (reason === DisconnectReason.timedOut) {
      console.log('⏱️ Connection timeout. Attempting to reconnect...');
      await this.scheduleReconnection();
    } else if (shouldReconnect) {
      console.log(`🔄 Unknown error (${reason}). Attempting to reconnect...`);
      await this.scheduleReconnection();
    }
  }

  getDisconnectReason(reason) {
    const reasons = {
      [DisconnectReason.badSession]: 'Invalid session',
      [DisconnectReason.connectionClosed]: 'Connection closed',
      [DisconnectReason.connectionLost]: 'Connection lost',
      [DisconnectReason.connectionReplaced]: 'Connection replaced',
      [DisconnectReason.loggedOut]: 'User disconnected',
      [DisconnectReason.multideviceMismatch]: 'Multidevice error',
      [DisconnectReason.forbidden]: 'Access denied',
      [DisconnectReason.restartRequired]: 'Restart required',
      [DisconnectReason.timedOut]: 'Connection timeout'
    };
    
    return reasons[reason] || `Unknown error (${reason})`;
  }

  async scheduleReconnection() {
    if (this.isInitializing) {
      console.log('⚠️ Already initializing, ignoring reconnection...');
      return;
    }

    if (this.reconnectAttempts >= this.maxReconnectAttempts) {
      console.log('❌ Maximum reconnection attempts reached.');
      this.io.emit('connection-status', { 
        connected: false, 
        error: 'Maximum reconnection attempts reached. Click "Reconnect" to try again.' 
      });
      return;
    }

    this.reconnectAttempts++;
    const delay = Math.min(10000 * this.reconnectAttempts, 60000); // Max 60s
    
    console.log(`🔄 Scheduling reconnection ${this.reconnectAttempts}/${this.maxReconnectAttempts} in ${delay/1000}s...`);
    
    this.retryTimeout = setTimeout(async () => {
      if (this.shouldReconnect && !this.isInitializing) {
        try {
          await this.initialize();
        } catch (error) {
          console.error('Error in reconnection:', error);
          await this.scheduleReconnection();
        }
      }
    }, delay);
  }

  async clearSession() {
    try {
      console.log('🧹 Clearing session...');
      
      // Stop reconnections
      this.shouldReconnect = false;
      
      // Close socket if active
      if (this.sock) {
        try {
          this.sock.ev.removeAllListeners();
          this.sock.end();
        } catch (err) {
          console.log('Error closing socket:', err.message);
        }
        this.sock = null;
      }

      // Clear timeout
      if (this.retryTimeout) {
        clearTimeout(this.retryTimeout);
        this.retryTimeout = null;
      }

      // Wait a bit
      await new Promise(resolve => setTimeout(resolve, 2000));

      // Remove session files
      if (fs.existsSync(this.sessionPath)) {
        const files = fs.readdirSync(this.sessionPath);
        for (const file of files) {
          try {
            const filePath = path.join(this.sessionPath, file);
            if (fs.statSync(filePath).isFile()) {
              fs.unlinkSync(filePath);
            }
          } catch (err) {
            console.log(`Error removing file ${file}:`, err.message);
          }
        }
        console.log('🗑️ Session cleared successfully');
      }

      // Reset state
      this.qrCode = null;
      this.isConnected = false;
      this.reconnectAttempts = 0;
      this.qrGenerationCount = 0;
      this.isInitializing = false;
      
      // Reactivate reconnection
      this.shouldReconnect = true;
      
      // Emit reset to clients
      this.io.emit('qr-code', null);
      this.io.emit('connection-status', { connected: false });
      
    } catch (error) {
      console.error('Error clearing session:', error);
    }
  }

  // services/whatsappService.js

  async handleMessage(m) {
    try {
      const message = m.messages[0];
      if (!message || message.key.fromMe || !message.message) return;

      const phoneNumber = message.key.remoteJid?.replace("@s.whatsapp.net", "");
      if (!phoneNumber) return;

      // Extract contact name from WhatsApp (pushName)
      const pushName = message.pushName || null;
      console.log(`📱 WhatsApp name (pushName): ${pushName}`);

      // Extract message text and media information
      let messageData = this.extractMessageData(message);
      messageData.pushName = pushName; // Add pushName to message data

      // If it's a media message, download and save
      if (messageData.mediaType) {
        console.log(`Received media from ${phoneNumber}. Type: ${messageData.mediaType}`);
        const mediaUrl = await this.downloadAndSaveMedia(message, messageData.mediaType);
        if (mediaUrl) {
          messageData.mediaUrl = mediaUrl;
        } else {
          console.error("Could not download or save media, ignoring message.");
          return;
        }
      }

      if (!messageData.text && !messageData.mediaUrl) {
        console.log("Empty or unsupported message, ignoring.");
        return;
      }

      console.log(`📨 Message received from ${phoneNumber}: ${messageData.text ? messageData.text.substring(0, 50) + '...' : '[Media]'}`);

      // Save message and capture the ID and complete conversation object
      const { conversationId, fullConversation } = await this.saveMessage(phoneNumber, messageData, message);

      // Check if conversation was saved successfully before continuing
      if (!conversationId || !fullConversation) {
          console.error('Could not save conversation, aborting additional processing.');
          return;
      }

      // Reaction will be sent by FAQ or AI (don't send default reaction here to avoid duplication)

      // Check if it's first message of the day and send welcome messages
      const sentWelcome = await this.sendWelcomeMessages(phoneNumber, message.key, messageData.pushName);

      // Process automatic reply (FAQ or AI) ALWAYS, regardless of welcome messages
      if (messageData.text) {
        // If welcome messages were sent, wait a bit more before processing
        if (sentWelcome) {
          setTimeout(async () => {
            await this.processAutoReply(phoneNumber, messageData.text, message, messageData.pushName);
          }, 10000); // 10s after last welcome message
        } else {
          // If welcome messages were not sent, process normally
          await this.processAutoReply(phoneNumber, messageData.text, message, messageData.pushName);
        }
      }

      // Send push notification if message is not from bot
      if (!message.key.fromMe) {
        await this.sendPushNotification(fullConversation);
      }

      // Emit to connected clients with COMPLETE object including messageId
      this.io.emit('new-message', {
        messageId: fullConversation.last_message_id || conversationId + '-' + Date.now(), // Unique ID
        conversationId: conversationId,
        phone_number: phoneNumber,
        message_text: messageData.text,
        media_url: messageData.mediaUrl,
        media_type: messageData.mediaType,
        timestamp: new Date(),
        is_from_bot: false,
        message_type: messageData.mediaUrl ? 'media' : 'text'
      });

    } catch (error) {
      console.error('Error processing message:', error);
    }
  }

  async downloadAndSaveMedia(message, mediaType) {
    try {
      const stream = await downloadMediaMessage(
        message,
        'buffer',
        { },
        { 
          logger: pino({ level: 'error' })
        }
      );

      const uploadDir = path.join(__dirname, '..', 'uploads');
      if (!fs.existsSync(uploadDir)) {
        fs.mkdirSync(uploadDir, { recursive: true });
      }

      let extension = '';
      switch (mediaType) {
        case 'image':
          extension = '.jpeg';
          break;
        case 'video':
          extension = '.mp4';
          break;
        case 'audio':
          extension = '.mp3';
          break;
        case 'document':
          // Try to get extension from original file name, if available
          extension = message.message?.documentMessage?.fileName ? path.extname(message.message.documentMessage.fileName) : '.bin';
          break;
        default:
          extension = '.bin';
      }

      const fileName = `${Date.now()}-${Math.random().toString(36).substring(7)}${extension}`;
      const filePath = path.join(uploadDir, fileName);
      fs.writeFileSync(filePath, stream);
      console.log(`✅ Media saved at: ${filePath}`);
      return `/uploads/${fileName}`;
    } catch (error) {
      console.error('❌ Error downloading and saving media:', error);
      return null;
    }
  }

  extractMessageData(message) {
    const msg = message.message;
    let messageData = { text: null, mediaUrl: null, mediaType: null };
    
    if (msg.conversation) {
      messageData.text = msg.conversation;
    } else if (msg.extendedTextMessage?.text) {
      messageData.text = msg.extendedTextMessage.text;
    } else if (msg.imageMessage) {
      messageData.text = msg.imageMessage.caption || null;
      messageData.mediaType = 'image';
    } else if (msg.videoMessage) {
      messageData.text = msg.videoMessage.caption || null;
      messageData.mediaType = 'video';
    } else if (msg.audioMessage) {
      messageData.mediaType = 'audio';
    } else if (msg.documentMessage) {
      messageData.text = msg.documentMessage.caption || null;
      messageData.mediaType = 'document';
    } else if (msg.stickerMessage) {
      messageData.mediaType = 'sticker';
    } else if (msg.locationMessage) {
      messageData.text = `Location: Latitude ${msg.locationMessage.degreesLatitude}, Longitude ${msg.locationMessage.degreesLongitude}`;
      messageData.mediaType = 'location';
    } else if (msg.contactMessage) {
      messageData.text = `Contact: ${msg.contactMessage.displayName}`;
      messageData.mediaType = 'contact';
    } else if (msg.reactionMessage) {
      messageData.text = `Reaction: ${msg.reactionMessage.text}`;
      messageData.mediaType = 'reaction';
    }
    
    return messageData;
  }

  async saveMessage(phoneNumber, messageData, message) {
    let connection;
    try {
      connection = await pool.getConnection();

      // Check if conversation exists
      let [conversations] = await connection.execute(
        'SELECT id FROM conversations WHERE phone_number = ?',
        [phoneNumber]
      );

      let conversationId;
      const lastMessageText = messageData.text || `[${messageData.mediaType || 'Media'}]`;
      
      if (conversations.length === 0) {
        // Create new conversation
        const [result] = await connection.execute(
          'INSERT INTO conversations (phone_number, last_message, last_message_time, status) VALUES (?, ?, NOW(), ?)',
          [phoneNumber, lastMessageText, 'waiting']
        );
        conversationId = result.insertId;
      } else {
        conversationId = conversations[0].id;
        // Update last message
        await connection.execute(
          'UPDATE conversations SET last_message = ?, last_message_time = NOW(), status = ? WHERE id = ?',
          [lastMessageText, 'waiting', conversationId]
        );
      }

      // --- FIX APPLIED HERE ---
      // The INSERT query was corrected to no longer use the 'media_type' column, which doesn't exist.
      // And the 'message_type' column now receives the correct value from messageData.mediaType
      await connection.execute(
        'INSERT INTO messages (conversation_id, phone_number, message_text, media_url, message_type, timestamp, is_from_bot) VALUES (?, ?, ?, ?, ?, NOW(), ?)',
        [
          conversationId, 
          phoneNumber, 
          messageData.text, 
          messageData.mediaUrl, 
          messageData.mediaType || 'text', 
          false
        ]
      );

      // Fetch updated conversation data to return
      const [updatedConversations] = await connection.execute(
          'SELECT * FROM conversations WHERE id = ?',
          [conversationId]
      );
      const fullConversation = updatedConversations[0];

      return { conversationId, fullConversation };

    } catch (error) {
      console.error('Error saving message:', error);
      return { conversationId: null, fullConversation: null };
    } finally {
      if (connection) connection.release();
    }
  }
  async processAutoReply(phoneNumber, messageText, message, pushName = null) {
  let connection;
  try {
    connection = await pool.getConnection();

    // 1. Check if bot is active
    const botActive = await this.getBotSetting(connection, "bot_active", "false");
    if (botActive !== "true") {
      console.log("🤖 Bot disabled, not processing auto-reply.");
      return;
    }

    // 2. Check if conversation is being attended by a human
    const [conversations] = await connection.execute(
      'SELECT status, assigned_user_id, last_message_time FROM conversations WHERE phone_number = ? LIMIT 1',
      [phoneNumber]
    );
    
    // If conversation has status 'attended' (being attended by human)
    if (conversations.length > 0 && conversations[0].status === 'attended' && conversations[0].assigned_user_id) {
      const lastMessageTime = new Date(conversations[0].last_message_time);
      const now = new Date();
      const inactiveMinutes = (now - lastMessageTime) / 1000 / 60;
      
      // If attended by human and has less than 30 minutes of inactivity, don't reply
      if (inactiveMinutes < 30) {
        console.log(`👤 Conversation being attended (ID: ${conversations[0].assigned_user_id}). AI disabled. Inactivity: ${inactiveMinutes.toFixed(1)} min`);
        return;
      } else {
        console.log(`⏰ Conversation inactive for ${inactiveMinutes.toFixed(1)} min. Reactivating AI...`);
        // Change status back to 'waiting' to reactivate AI
        await connection.execute(
          'UPDATE conversations SET status = \'waiting\', assigned_user_id = NULL WHERE phone_number = ?',
          [phoneNumber]
        );
      }
    }

    // 3. Try to find a response in FAQ with custom reaction
    // Supports multiple questions separated by comma
    const [faqs] = await connection.execute(`
      SELECT answer, emoji, question FROM faqs WHERE active = TRUE
    `);
    
    // Search for match in any of the questions
    let matchedFaq = null;
    const normalizedMessage = messageText.toLowerCase().trim();
    
    for (const faq of faqs) {
      // Split questions by comma and check each one
      const questions = faq.question.split(',').map(q => q.trim().toLowerCase());
      
      for (const question of questions) {
        // Check if message contains the question or vice versa
        if (normalizedMessage.includes(question) || question.includes(normalizedMessage)) {
          matchedFaq = faq;
          break;
        }
      }
      
      if (matchedFaq) break;
    }

    if (matchedFaq) {
      console.log(`💡 Found FAQ response for: "${messageText.substring(0, 30)}..."`);
      
      let faqAnswer = matchedFaq.answer;
      const faqReaction = matchedFaq.emoji || '❤️';
      
      // Process placeholders in FAQ response
      try {
        // Fetch contact name - try with and without country prefix
        const cleanPhone = phoneNumber.replace(/[^0-9]/g, '');
        const [contact] = await connection.execute(
          'SELECT name FROM contacts WHERE REPLACE(REPLACE(REPLACE(phone, \'+\', \'\'), \'-\', \'\'), \' \', \'\') LIKE ? LIMIT 1',
          [`%${cleanPhone}%`]
        );
        
        console.log(`🔍 Searching contact for: ${phoneNumber} (cleaned: ${cleanPhone})`);
        console.log(`📄 Search result:`, contact);
        
        let contactName = pushName || 'Customer'; // Use pushName as default
        if (contact && contact.length > 0 && contact[0] && contact[0].name) {
          contactName = contact[0].name;
          console.log(`✅ Name found in database: ${contactName}`);
        } else if (pushName) {
          console.log(`📱 Using WhatsApp name: ${pushName}`);
        } else {
          console.log(`⚠️ Contact not found, using 'Customer'`);
        }
        
        const now = new Date();
        const dataAtual = now.toLocaleDateString('pt-BR');
        const horaAtual = now.toLocaleTimeString('pt-BR', { hour: '2-digit', minute: '2-digit' });
        
        faqAnswer = faqAnswer
          .replace(/{{nome_cliente}}/g, contactName)
          .replace(/{{data_atual}}/g, dataAtual)
          .replace(/{{hora_atual}}/g, horaAtual);
      } catch (error) {
        console.error('Error processing FAQ placeholders:', error);
      }
      
      // Release connection BEFORE async operations with timeout
      connection.release();
      connection = null;
      
      // 1. Send reaction after 3 seconds
      setTimeout(async () => {
        try {
          console.log(`😊 Sending reaction ${faqReaction} to ${phoneNumber}`);
          await this.sendReaction(message.key, faqReaction);
        } catch (error) {
          console.error('Error sending FAQ reaction:', error);
        }
      }, 3000);

      // 2. Show "typing" after 4 seconds (and keep for 3 seconds)
      setTimeout(async () => {
        try {
          console.log(`✏️ Showing typing status for ${phoneNumber}`);
          await this.sendTyping(phoneNumber, true, 3000);
        } catch (error) {
          console.error('Error sending typing status:', error);
        }
      }, 4000);

      // 3. Send response after 7 seconds
      setTimeout(async () => {
        try {
          console.log(`📤 Sending FAQ response to ${phoneNumber}`);
          await this.sendMessage(phoneNumber, faqAnswer);
        } catch (error) {
          console.error('Error sending FAQ message:', error);
        }
      }, 7000);

      return; // End processing if found in FAQ
    }

    // 4. If not found in FAQ, trigger AI (DeepSeek)
    console.log("🧠 FAQ not found. Trying to process with AI (DeepSeek)...");
    
    // Release connection BEFORE calling AI
    connection.release();
    connection = null;
    
    // FIX: Remove verification that blocks processing
    // Now always tries to process, and aiService decides internally if it responds
    const aiResult = await aiService.processMessage(phoneNumber, messageText);

    // Check if AI returned a valid response
    if (aiResult.shouldRespond && aiResult.success) {
      console.log(`🧠 AI activated: ${aiResult.provider || 'DeepSeek'} - ${aiResult.model || 'default model'}`);
      
      // 1. AI reaction after 3 seconds
      setTimeout(async () => {
        try {
          await this.sendReaction(message.key, aiResult.emoji || '❤️');
        } catch (error) {
          console.error('Error sending AI reaction:', error);
        }
      }, 3000);

      // 2. Typing status after 4 seconds
      setTimeout(async () => {
        try {
          const typingDuration = Math.min(aiResult.response.length * 50, 8000);
          await this.sendTyping(phoneNumber, true, typingDuration);
        } catch (error) {
          console.error('Error sending typing status (AI):', error);
        }
      }, 4000);

      // 3. Send AI response with processed placeholders
      const delay = aiResult.responseTime ? (aiResult.responseTime * 1000) + 5000 : 8000;
      setTimeout(async () => {
        try {
          // Process placeholders in AI response
          let aiResponse = aiResult.response;
          try {
            const connection = await pool.getConnection();
            
            // Fetch contact name - try with and without country prefix
            const cleanPhone = phoneNumber.replace(/[^0-9]/g, '');
            const [contact] = await connection.execute(
              'SELECT name FROM contacts WHERE REPLACE(REPLACE(REPLACE(phone, \'+\', \'\'), \'-\', \'\'), \' \', \'\') LIKE ? LIMIT 1',
              [`%${cleanPhone}%`]
            );
            connection.release();
            
            console.log(`🔍 [AI] Searching contact for: ${phoneNumber} (cleaned: ${cleanPhone})`);
            console.log(`📄 [AI] Search result:`, contact);
            
            let contactName = pushName || 'Customer'; // Use pushName as default
            if (contact && contact.length > 0 && contact[0] && contact[0].name) {
              contactName = contact[0].name;
              console.log(`✅ [AI] Name found in database: ${contactName}`);
            } else if (pushName) {
              console.log(`📱 [AI] Using WhatsApp name: ${pushName}`);
            } else {
              console.log(`⚠️ [AI] Contact not found, using 'Customer'`);
            }
            
            const now = new Date();
            const dataAtual = now.toLocaleDateString('pt-BR');
            const horaAtual = now.toLocaleTimeString('pt-BR', { hour: '2-digit', minute: '2-digit' });
            
            aiResponse = aiResponse
              .replace(/{{nome_cliente}}/g, contactName)
              .replace(/{{data_atual}}/g, dataAtual)
              .replace(/{{hora_atual}}/g, horaAtual);
          } catch (error) {
            console.error('Error processing AI placeholders:', error);
          }
          
          await this.sendMessage(phoneNumber, aiResponse);
        } catch (error) {
          console.error('Error sending AI message:', error);
        }
      }, delay);

      return;
    } else {
      // If AI should not respond (outside hours, no config, etc)
      console.log(`🔕 AI did not respond. Reason: ${aiResult.error || 'No response configured'}`);
    }

    console.log("🔕 No auto-reply rule (FAQ/AI) was triggered.");

  } catch (error) {
    console.error("❌ Error in processAutoReply:", error);
    
    if (error.message && error.message.includes('closed state')) {
      console.log("🔄 Trying to reconnect to database...");
    }
  } finally {
    if (connection) {
      try {
        connection.release();
      } catch (releaseError) {
        console.error("Error releasing connection:", releaseError);
      }
    }
  }
}

// Helper method (if it doesn't exist yet or is outdated)
async getBotSetting(connection, key, defaultValue = null) {
  try {
    if (!connection || connection.connection._closing) {
      console.warn(`⚠️ Invalid connection when fetching setting ${key}. Returning default value.`);
      return defaultValue;
    }
    
    const [settings] = await connection.execute(
      'SELECT setting_value FROM bot_settings WHERE setting_key = ?',
      [key]
    );
    return settings.length > 0 ? settings[0].setting_value : defaultValue;
  } catch (error) {
    console.error(`Error fetching bot setting ${key}:`, error);
    return defaultValue;
  }
}

// ADD this improved helper method if it doesn't exist yet or replace the existing one:
async getBotSetting(connection, key, defaultValue = null) {
  try {
    // Check if connection is valid before using
    if (!connection || connection.connection._closing) {
      console.warn(`⚠️ Invalid connection when fetching setting ${key}. Returning default value.`);
      return defaultValue;
    }
    
    const [settings] = await connection.execute(
      'SELECT setting_value FROM bot_settings WHERE setting_key = ?',
      [key]
    );
    return settings.length > 0 ? settings[0].setting_value : defaultValue;
  } catch (error) {
    console.error(`Error fetching bot setting ${key}:`, error);
    return defaultValue;
  }
}

// IMPORTANT: Make sure the sendTyping method is correct.
// If necessary, replace with this improved version:
async sendTyping(phoneNumber, isTyping, duration = 5000) {
  try {
    if (!this.sock || !this.isConnected) {
      console.log('❌ WhatsApp not connected. Cannot send typing status.');
      return;
    }
    const jid = phoneNumber.includes('@') ? phoneNumber : `${phoneNumber}@s.whatsapp.net`;

    if (isTyping) {
      await this.sock.sendPresenceUpdate('composing', jid);
      console.log(`✍️ Sending 'typing' status to ${phoneNumber} for ${duration}ms`);

      // Automatically stop "typing" after specified duration
      setTimeout(async () => {
        try {
          await this.sock.sendPresenceUpdate('paused', jid);
          console.log(`✍️ Stopping 'typing' status for ${phoneNumber}`);
        } catch (error) {
          console.error('Error stopping typing status:', error);
        }
      }, duration);

    } else {
      await this.sock.sendPresenceUpdate('paused', jid);
      console.log(`✍️ Stopping 'typing' status for ${phoneNumber}`);
    }
  } catch (error) {
    console.error('Error sending presence status:', error);
  }
}

  async sendMessage(phoneNumber, content, options = {}) {
    try {
      if (!this.sock || !this.isConnected) {
        console.log('❌ WhatsApp not connected. Message not sent.');
        return false;
      }

      const jid = phoneNumber.includes('@') ? phoneNumber : phoneNumber + '@s.whatsapp.net';
      let messageOptions = {};
      let messageText = '';
      let messageType = 'text';
      let mediaUrl = null;
      let mediaType = null;
      
      // Check if it's simple text or media
      if (typeof content === 'string') {
        messageOptions.text = content;
        messageText = content;
      } else if (content.mediaUrl) {
        // It's a media message
        const filePath = path.join(__dirname, '..', content.mediaUrl);
        
        if (fs.existsSync(filePath)) {
          const mimeType = mime.lookup(filePath) || 'application/octet-stream';
          const fileBuffer = fs.readFileSync(filePath);
          
          if (mimeType.startsWith('image/')) {
            messageOptions.image = fileBuffer;
            messageOptions.caption = content.caption || '';
            messageType = 'media';
            mediaType = 'image';
          } else if (mimeType.startsWith('video/')) {
            messageOptions.video = fileBuffer;
            messageOptions.caption = content.caption || '';
            messageType = 'media';
            mediaType = 'video';
          } else if (mimeType.startsWith('audio/')) {
            messageOptions.audio = fileBuffer;
            messageType = 'media';
            mediaType = 'audio';
          } else {
            messageOptions.document = fileBuffer;
            messageOptions.fileName = path.basename(filePath);
            messageOptions.mimetype = mimeType;
            messageOptions.caption = content.caption || '';
            messageType = 'media';
            mediaType = 'document';
          }
          
          messageText = content.caption || '';
          mediaUrl = content.mediaUrl;
        } else {
          console.error('Media file not found:', filePath);
          return false;
        }
      }

      await this.sock.sendMessage(jid, messageOptions);
      
      console.log(`📤 Message sent to ${phoneNumber}: ${messageText ? messageText.substring(0, 50) + '...' : '[Media]'}`);
      
      // Save sent message with user information
      // The 'new-message' emit is already done inside saveOutgoingMessage
      await this.saveOutgoingMessage(phoneNumber, messageText, mediaUrl, mediaType, messageType, options.senderUserId, options.conversationId);
      
      return true;
      
    } catch (error) {
      console.error('Error sending message:', error);
      return false;
    }
  }

  async saveOutgoingMessage(phoneNumber, messageText, mediaUrl, mediaType, messageTypeDb, senderUserId, conversationId = null) {
    let connection;
    try {
      connection = await pool.getConnection();

      // If conversationId was not provided, fetch it
      if (!conversationId) {
        const [conversations] = await connection.execute(
          'SELECT id FROM conversations WHERE phone_number = ?',
          [phoneNumber]
        );
        if (conversations.length === 0) {
          connection.release();
          return;
        }
        conversationId = conversations[0].id;
      }
      const lastMessageText = messageText || `[${mediaType || 'Media'}]`;

      // --- FIX APPLIED HERE ---
      // The INSERT query was corrected to no longer use the 'media_type' column.
      await connection.execute(
        'INSERT INTO messages (conversation_id, phone_number, message_text, media_url, message_type, timestamp, is_from_bot, sender_user_id) VALUES (?, ?, ?, ?, ?, NOW(), ?, ?)',
        [
          conversationId, 
          phoneNumber, 
          messageText, 
          mediaUrl, 
          messageTypeDb, // Using the correct 'message_type' column
          senderUserId ? false : true, // If has senderUserId, it's not from bot
          senderUserId || null
        ]
      );

      // Update conversation last message
      await connection.execute(
        'UPDATE conversations SET last_message = ?, last_message_time = NOW() WHERE id = ?',
        [lastMessageText, conversationId]
      );
      
      // Fetch newly saved message to emit with ID
      const [newMessage] = await connection.execute(
        'SELECT * FROM messages WHERE conversation_id = ? ORDER BY id DESC LIMIT 1',
        [conversationId]
      );
      
      // Emit new-message with message ID
      if (newMessage.length > 0) {
        this.io.emit('new-message', {
          messageId: newMessage[0].id,
          conversationId: conversationId,
          phone_number: phoneNumber,
          message_text: messageText,
          media_url: mediaUrl,
          message_type: messageTypeDb,
          timestamp: new Date(),
          is_from_bot: senderUserId ? false : true,
          sender_user_id: senderUserId,
          userId: senderUserId // For frontend to identify
        });
      }

    } catch (error) {
      console.error('Error saving sent message:', error);
    } finally {
      if (connection) connection.release();
    }
  }

  async reactToMessage(messageKey, emoji) {
    try {
      if (this.sock && this.isConnected && messageKey) {
        await this.sock.sendMessage(messageKey.remoteJid, {
          react: {
            text: emoji,
            key: messageKey
          }
        });
        console.log(`😀 Reaction sent: ${emoji}`);
      }
    } catch (error) {
      console.error('Error reacting to message:', error);
    }
  }

  async handleError(error) {
    console.error('❌ Error in WhatsApp Service:', error);
    this.isInitializing = false;
    
    if (this.shouldReconnect && this.reconnectAttempts < this.maxReconnectAttempts) {
      await this.scheduleReconnection();
    }
  }

  // Method to manually disconnect
  async disconnect() {
    try {
      console.log('🔌 Disconnecting WhatsApp...');
      this.shouldReconnect = false;
      
      if (this.sock) {
        await this.sock.logout();
      }
      await this.clearSession();
    } catch (error) {
      console.error('Error disconnecting:', error);
      await this.clearSession(); // Force cleanup even with error
    }
  }

  // Method to force reconnection
  async forceReconnect() {
    try {
      console.log('🔄 Forcing WhatsApp reconnection...');
      this.reconnectAttempts = 0; // Reset counter
      this.qrGenerationCount = 0; // Reset QR counter
      await this.clearSession();
      
      // Wait a bit before reinitializing
      setTimeout(() => {
        if (this.shouldReconnect) {
          this.initialize();
        }
      }, 3000);
    } catch (error) {
      console.error('Error reconnecting:', error);
    }
  }

  getQRCode() {
    return this.qrCode;
  }

  getConnectionStatus() {
    return {
      connected: this.isConnected,
      qrCode: this.qrCode,
      reconnectAttempts: this.reconnectAttempts,
      maxReconnectAttempts: this.maxReconnectAttempts,
      isInitializing: this.isInitializing,
      qrGenerationCount: this.qrGenerationCount
    };
  }

  // Method to check if connected
  isWhatsAppConnected() {
    return this.isConnected && this.sock && !this.isInitializing;
  }

  // Send reaction to a message
  async sendReaction(messageKey, emoji) {
    try {
      if (!this.sock || !this.isConnected) {
        console.log('❌ WhatsApp not connected. Reaction not sent.');
        return false;
      }

      await this.sock.sendMessage(messageKey.remoteJid, {
        react: {
          text: emoji,
          key: messageKey
        }
      });

      console.log(`✅ Reaction ${emoji} sent`);
      return true;
    } catch (error) {
      console.error('Error sending reaction:', error);
      return false;
    }
  }

  // Send typing status
  async sendTyping(phoneNumber, isTyping = true) {
    try {
      if (!this.sock || !this.isConnected) {
        console.log('❌ WhatsApp not connected. Typing status not sent.');
        return false;
      }

      const jid = phoneNumber.includes('@') ? phoneNumber : phoneNumber + '@s.whatsapp.net';
      
      await this.sock.sendPresenceUpdate(isTyping ? 'composing' : 'paused', jid);
      
      console.log(`✅ Typing status ${isTyping ? 'activated' : 'deactivated'}`);
      return true;
    } catch (error) {
      console.error('Error sending typing status:', error);
      return false;
    }
  }

  // Send welcome messages (only once per day)
  async sendWelcomeMessages(phoneNumber, messageKey, pushName = null) {
    let connection;
    try {
      connection = await pool.getConnection();

      // Check if welcome messages were already sent today (using specific table)
      const [lastWelcome] = await connection.execute(`
        SELECT id FROM welcome_sent 
        WHERE phone_number = ? AND DATE(sent_at) = CURDATE()
        LIMIT 1
      `, [phoneNumber]);

      if (lastWelcome.length > 0) {
        console.log(`👋 Welcome messages already sent today for ${phoneNumber}`);
        return false; // Returns false to indicate it didn't send
      }

      // Fetch configured welcome messages
      const [welcomeMessages] = await connection.execute(`
        SELECT * FROM welcome_messages 
        WHERE active = TRUE 
        ORDER BY order_position ASC
      `);

      if (welcomeMessages.length === 0) {
        console.log('👋 No welcome messages configured');
        return;
      }

      // Send initial reaction after 3 seconds
      setTimeout(async () => {
        await this.sendReaction(messageKey, '👋');
      }, 3000);

      // Send typing status after 4 seconds
      setTimeout(async () => {
        await this.sendTyping(phoneNumber, true);
      }, 4000);

      // Fetch contact name - try with and without country prefix
      const cleanPhone = phoneNumber.replace(/[^0-9]/g, '');
      let contact = await connection.execute(
        'SELECT name FROM contacts WHERE REPLACE(REPLACE(REPLACE(phone, \'+\', \'\'), \'-\', \'\'), \' \', \'\') LIKE ? LIMIT 1',
        [`%${cleanPhone}%`]
      ).catch(() => [[]]);
      
      console.log(`🔍 [Welcome] Searching contact for: ${phoneNumber} (cleaned: ${cleanPhone})`);
      console.log(`📄 [Welcome] Search result:`, contact);
      
      let contactName = pushName || 'Customer'; // Use pushName as default
      if (contact && contact[0] && contact[0].length > 0 && contact[0][0] && contact[0][0].name) {
        contactName = contact[0][0].name;
        console.log(`✅ [Welcome] Name found in database: ${contactName}`);
      } else if (pushName) {
        console.log(`📱 [Welcome] Using WhatsApp name: ${pushName}`);
      } else {
        console.log(`⚠️ [Welcome] Contact not found, using 'Customer'`);
      }

      // Send each message with 3 second interval
      for (let i = 0; i < welcomeMessages.length; i++) {
        const msg = welcomeMessages[i];
        const delay = 7000 + (i * 3000); // 7s for first, then +3s for each

        setTimeout(async () => {
          try {
            // Process placeholders
            const now = new Date();
            const dataAtual = now.toLocaleDateString('pt-BR');
            const horaAtual = now.toLocaleTimeString('pt-BR', { hour: '2-digit', minute: '2-digit' });
            
            let messageText = msg.message_text
              .replace(/{{nome_cliente}}/g, contactName)
              .replace(/{{data_atual}}/g, dataAtual)
              .replace(/{{hora_atual}}/g, horaAtual);
            
            // Send text only (no media)
            await this.sendMessage(phoneNumber, messageText);
            console.log(`👋 Welcome message ${i + 1} sent to ${phoneNumber}`);
          } catch (error) {
            console.error(`Error sending welcome message ${i + 1}:`, error);
          }
        }, delay);
      }

      // Register that welcome messages were sent today
      await connection.execute(`
        INSERT INTO welcome_sent (phone_number, sent_at) VALUES (?, NOW())
      `, [phoneNumber]);
      
      return true; // Returns true to indicate it sent

    } catch (error) {
      console.error('Error processing welcome messages:', error);
    } finally {
      if (connection) connection.release();
    }
  }
}

module.exports = WhatsAppService;