const { pool } = require('../config/database');

class MessageQueueService {
    constructor(whatsappService, io) {
        this.whatsappService = whatsappService;
        this.io = io;
        this.isProcessing = false;
        this.processingInterval = null;
        this.connectionCheckInterval = null;
        this.isConnected = false;
        
        // Settings
        this.config = {
            processIntervalMs: 5000, // 5 seconds
            connectionCheckIntervalMs: 10000, // 10 seconds
            maxRetries: 5,
            retryDelays: [0, 30000, 120000, 480000, 1920000] // 0s, 30s, 2min, 8min, 32min
        };
        
        this.initialize();
    }
    
    async initialize() {
        console.log('🔄 Initializing MessageQueueService...');
        
        // Create necessary tables
        await this.createTables();
        
        // Initialize connection status
        await this.initializeConnectionStatus();
        
        // Start workers
        this.startProcessingWorker();
        this.startConnectionMonitor();
        
        console.log('✅ MessageQueueService initialized!');
    }
    
    async createTables() {
        let connection;
        try {
            connection = await pool.getConnection();
            
            // Message queue table
            await connection.execute(`
                CREATE TABLE IF NOT EXISTS message_queue (
                    id INT PRIMARY KEY AUTO_INCREMENT,
                    conversation_id INT NOT NULL,
                    phone_number VARCHAR(20) NOT NULL,
                    message_text TEXT,
                    message_type ENUM('text', 'image', 'video', 'audio', 'document') DEFAULT 'text',
                    media_url VARCHAR(500),
                    sender_user_id INT,
                    status ENUM('pending', 'processing', 'sent', 'failed') DEFAULT 'pending',
                    attempts INT DEFAULT 0,
                    max_attempts INT DEFAULT 5,
                    priority INT DEFAULT 1,
                    scheduled_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    last_attempt_at TIMESTAMP NULL,
                    error_message TEXT NULL,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    
                    INDEX idx_status_priority (status, priority, scheduled_at),
                    INDEX idx_phone_number (phone_number),
                    INDEX idx_conversation_id (conversation_id),
                    FOREIGN KEY (conversation_id) REFERENCES conversations(id) ON DELETE CASCADE,
                    FOREIGN KEY (sender_user_id) REFERENCES users(id) ON DELETE SET NULL
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            `);
            
            // Connection status table
            await connection.execute(`
                CREATE TABLE IF NOT EXISTS whatsapp_connection_status (
                    id INT PRIMARY KEY AUTO_INCREMENT,
                    is_connected BOOLEAN DEFAULT FALSE,
                    last_check_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    error_message TEXT NULL,
                    reconnect_attempts INT DEFAULT 0,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            `);
            
            console.log('✅ Message queue tables created/verified');
            
        } catch (error) {
            console.error('❌ Error creating tables:', error);
            throw error;
        } finally {
            if (connection) connection.release();
        }
    }
    
    async initializeConnectionStatus() {
        let connection;
        try {
            connection = await pool.getConnection();
            
            // Check if a record already exists
            const [existing] = await connection.execute(
                'SELECT id FROM whatsapp_connection_status LIMIT 1'
            );
            
            if (existing.length === 0) {
                await connection.execute(
                    'INSERT INTO whatsapp_connection_status (is_connected) VALUES (FALSE)'
                );
            }
            
        } catch (error) {
            console.error('❌ Error initializing connection status:', error);
        } finally {
            if (connection) connection.release();
        }
    }
    
    // Add message to queue
    async addToQueue(conversationId, phoneNumber, messageText, messageType = 'text', mediaUrl = null, senderUserId = null, priority = 1) {
        let connection;
        try {
            connection = await pool.getConnection();
            
            const [result] = await connection.execute(`
                INSERT INTO message_queue 
                (conversation_id, phone_number, message_text, message_type, media_url, sender_user_id, priority)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            `, [conversationId, phoneNumber, messageText, messageType, mediaUrl, senderUserId, priority]);
            
            const queueId = result.insertId;
            
            // Try to send immediately if connected
            if (this.isConnected) {
                await this.processMessage(queueId);
            }
            
            // Notify interface about queued message
            this.io.emit('message-queued', {
                queueId,
                conversationId,
                phoneNumber,
                status: 'pending'
            });
            
            return queueId;
            
        } catch (error) {
            console.error('❌ Error adding message to queue:', error);
            throw error;
        } finally {
            if (connection) connection.release();
        }
    }
    
    // Process a specific message
    async processMessage(queueId) {
        let connection;
        try {
            connection = await pool.getConnection();
            
            // Fetch message from queue
            const [messages] = await connection.execute(
                'SELECT * FROM message_queue WHERE id = ? AND status IN ("pending", "failed")',
                [queueId]
            );
            
            if (messages.length === 0) {
                return false;
            }
            
            const message = messages[0];
            
            // Check if max attempts exceeded
            if (message.attempts >= message.max_attempts) {
                await connection.execute(
                    'UPDATE message_queue SET status = "failed", error_message = ? WHERE id = ?',
                    ['Maximum number of attempts exceeded', queueId]
                );
                return false;
            }
            
            // Check delay between attempts
            if (message.last_attempt_at) {
                const lastAttempt = new Date(message.last_attempt_at);
                const now = new Date();
                const timeSinceLastAttempt = now - lastAttempt;
                const requiredDelay = this.config.retryDelays[message.attempts] || this.config.retryDelays[this.config.retryDelays.length - 1];
                
                if (timeSinceLastAttempt < requiredDelay) {
                    return false; // Not time to retry yet
                }
            }
            
            // Mark as processing
            await connection.execute(
                'UPDATE message_queue SET status = "processing", attempts = attempts + 1, last_attempt_at = NOW() WHERE id = ?',
                [queueId]
            );
            
            try {
                // Try to send message
                let success = false;
                
                if (message.message_type === 'text') {
                    success = await this.whatsappService.sendMessage(
                        message.phone_number, 
                        message.message_text,
                        { senderUserId: message.sender_user_id }
                    );
                } else if (message.media_url) {
                    success = await this.whatsappService.sendMessage(
                        message.phone_number,
                        {
                            mediaUrl: message.media_url,
                            caption: message.message_text || ''
                        },
                        { senderUserId: message.sender_user_id }
                    );
                }
                
                if (success) {
                    // Mark as sent
                    await connection.execute(
                        'UPDATE message_queue SET status = "sent", error_message = NULL WHERE id = ?',
                        [queueId]
                    );
                    
                    // Notify interface
                    this.io.emit('message-sent', {
                        queueId,
                        conversationId: message.conversation_id,
                        phoneNumber: message.phone_number,
                        status: 'sent'
                    });
                    
                    return true;
                } else {
                    throw new Error('Message sending failed');
                }
                
            } catch (sendError) {
                console.error(`❌ Error sending message ${queueId}:`, sendError);
                
                // Mark as pending for retry or failed if max attempts exceeded
                const newStatus = message.attempts >= (message.max_attempts - 1) ? 'failed' : 'pending';
                await connection.execute(
                    'UPDATE message_queue SET status = ?, error_message = ? WHERE id = ?',
                    [newStatus, sendError.message, queueId]
                );
                
                // Notify interface about failure
                this.io.emit('message-failed', {
                    queueId,
                    conversationId: message.conversation_id,
                    phoneNumber: message.phone_number,
                    status: newStatus,
                    error: sendError.message,
                    attempts: message.attempts + 1,
                    maxAttempts: message.max_attempts
                });
                
                return false;
            }
            
        } catch (error) {
            console.error('❌ Error processing message:', error);
            return false;
        } finally {
            if (connection) connection.release();
        }
    }
    
    // Worker to process queue
    startProcessingWorker() {
        this.processingInterval = setInterval(async () => {
            if (this.isProcessing) return;
            
            this.isProcessing = true;
            
            try {
                await this.processQueue();
            } catch (error) {
                console.error('❌ Error in processing worker:', error);
            } finally {
                this.isProcessing = false;
            }
        }, this.config.processIntervalMs);
        
        console.log('🔄 Processing worker started');
    }
    
    // Process entire queue
    async processQueue() {
        if (!this.isConnected) {
            return; // Don't process if not connected
        }
        
        let connection;
        try {
            connection = await pool.getConnection();
            
            // Fetch pending messages ordered by priority and date
            const [pendingMessages] = await connection.execute(`
                SELECT id FROM message_queue 
                WHERE status IN ('pending', 'failed') 
                AND attempts < max_attempts
                ORDER BY priority DESC, scheduled_at ASC 
                LIMIT 10
            `);
            
            // Process messages one by one
            for (const msg of pendingMessages) {
                await this.processMessage(msg.id);
                
                // Small delay between messages to avoid overload
                await new Promise(resolve => setTimeout(resolve, 1000));
            }
            
        } catch (error) {
            console.error('❌ Error processing queue:', error);
        } finally {
            if (connection) connection.release();
        }
    }
    
    // Connection monitor
    startConnectionMonitor() {
        this.connectionCheckInterval = setInterval(async () => {
            await this.checkConnection();
        }, this.config.connectionCheckIntervalMs);
        
        console.log('🔄 Connection monitor started');
    }
    
    // Check connection status
    async checkConnection() {
        let connection;
        try {
            connection = await pool.getConnection();
            
            // Check if WhatsApp is connected
            let isConnected = false;
            let errorMessage = null;
            
            try {
                // Try to check WhatsApp status
                if (this.whatsappService && typeof this.whatsappService.isConnected === 'function') {
                    isConnected = await this.whatsappService.isConnected();
                } else if (this.whatsappService && this.whatsappService.client) {
                    // Fallback: check if client exists and is ready
                    isConnected = this.whatsappService.client.info ? true : false;
                } else {
                    isConnected = false;
                    errorMessage = 'WhatsApp service not available';
                }
            } catch (error) {
                isConnected = false;
                errorMessage = error.message;
            }
            
            // Update status in database
            await connection.execute(`
                UPDATE whatsapp_connection_status 
                SET is_connected = ?, last_check_at = NOW(), error_message = ?, updated_at = NOW()
                WHERE id = 1
            `, [isConnected, errorMessage]);
            
            // If status changed, notify
            if (this.isConnected !== isConnected) {
                const previousStatus = this.isConnected;
                this.isConnected = isConnected;
                
                console.log(`🔄 WhatsApp connection status: ${isConnected ? 'CONNECTED' : 'DISCONNECTED'}`);
                
                // Notify interface
                this.io.emit('whatsapp-connection-status', {
                    isConnected,
                    errorMessage,
                    timestamp: new Date()
                });
                
                // If reconnected, process queue immediately
                if (!previousStatus && isConnected) {
                    console.log('🔄 Reconnection detected, processing queue...');
                    setTimeout(() => this.processQueue(), 1000);
                }
            }
            
        } catch (error) {
            console.error('❌ Error checking connection:', error);
        } finally {
            if (connection) connection.release();
        }
    }
    
    // Get queue statistics
    async getQueueStats() {
        let connection;
        try {
            connection = await pool.getConnection();
            
            const [stats] = await connection.execute(`
                SELECT 
                    status,
                    COUNT(*) as count
                FROM message_queue 
                GROUP BY status
            `);
            
            const [connectionStatus] = await connection.execute(
                'SELECT * FROM whatsapp_connection_status ORDER BY id DESC LIMIT 1'
            );
            
            return {
                queue: stats.reduce((acc, stat) => {
                    acc[stat.status] = stat.count;
                    return acc;
                }, {}),
                connection: connectionStatus[0] || null
            };
            
        } catch (error) {
            console.error('❌ Error getting statistics:', error);
            return null;
        } finally {
            if (connection) connection.release();
        }
    }
    
    // Retry failed message
    async retryMessage(queueId) {
        let connection;
        try {
            connection = await pool.getConnection();
            
            // Reset status and attempts
            await connection.execute(`
                UPDATE message_queue 
                SET status = 'pending', attempts = 0, error_message = NULL, last_attempt_at = NULL
                WHERE id = ? AND status = 'failed'
            `, [queueId]);
            
            // Try to process immediately
            if (this.isConnected) {
                await this.processMessage(queueId);
            }
            
            return true;
            
        } catch (error) {
            console.error('❌ Error retrying message:', error);
            return false;
        } finally {
            if (connection) connection.release();
        }
    }
    
    // Cancel message in queue
    async cancelMessage(queueId) {
        let connection;
        try {
            connection = await pool.getConnection();
            
            await connection.execute(
                'DELETE FROM message_queue WHERE id = ? AND status IN ("pending", "failed")',
                [queueId]
            );
            
            // Notify interface
            this.io.emit('message-cancelled', { queueId });
            
            return true;
            
        } catch (error) {
            console.error('❌ Error cancelling message:', error);
            return false;
        } finally {
            if (connection) connection.release();
        }
    }
    
    // Clean up old messages from queue
    async cleanupQueue(daysOld = 7) {
        let connection;
        try {
            connection = await pool.getConnection();
            
            const [result] = await connection.execute(`
                DELETE FROM message_queue 
                WHERE status IN ('sent', 'failed') 
                AND created_at < DATE_SUB(NOW(), INTERVAL ? DAY)
            `, [daysOld]);
            
            console.log(`🧹 Queue cleanup: ${result.affectedRows} messages removed`);
            
            return result.affectedRows;
            
        } catch (error) {
            console.error('❌ Error cleaning queue:', error);
            return 0;
        } finally {
            if (connection) connection.release();
        }
    }
    
    // Stop service
    stop() {
        if (this.processingInterval) {
            clearInterval(this.processingInterval);
            this.processingInterval = null;
        }
        
        if (this.connectionCheckInterval) {
            clearInterval(this.connectionCheckInterval);
            this.connectionCheckInterval = null;
        }
        
        console.log('🛑 MessageQueueService stopped');
    }
}

module.exports = MessageQueueService;
