const axios = require("axios");
const crypto = require("crypto");
const { pool } = require("../config/database");

let OpenAI = null;

// OpenAI disabled - only DeepSeek will be used
process.env.ENABLE_OPENAI = 'false';

class AIService {
  constructor(enableOpenAI) {
    this.enableOpenAI = enableOpenAI;
    this.openai = null;
    if (this.enableOpenAI && OpenAI) {
      this.openai = new OpenAI({
        apiKey: process.env.OPENAI_API_KEY,
      });
    }
    this.activeConfig = null;
    this.conversationHistory = new Map();
  }

  /**
   * Decrypt sensitive data
   */
  decryptData(encryptedText) {
    try {
      const algorithm = "aes-256-cbc";
      const secretKey = process.env.ENCRYPTION_KEY || "default-key-change-in-production-32";
      
      // Ensure key has exactly 32 bytes for AES-256
      const key = crypto.createHash('sha256').update(String(secretKey)).digest();
      
      const textParts = encryptedText.split(":");
      const iv = Buffer.from(textParts.shift(), "hex");
      const encrypted = Buffer.from(textParts.join(":"), "hex");
      
      // Use createDecipheriv instead of createDecipher (deprecated)
      const decipher = crypto.createDecipheriv(algorithm, key, iv);
      let decrypted = decipher.update(encrypted);
      decrypted = Buffer.concat([decrypted, decipher.final()]);
      
      return decrypted.toString('utf8');
    } catch (error) {
      console.error("Error decrypting:", error);
      return null;
    }
  }

  /**
   * Load active AI configuration
   */
  async loadActiveConfig() {
    let connection;
    try {
      connection = await pool.getConnection();
      
      const [configs] = await connection.execute(
        "SELECT * FROM ai_configurations WHERE active = TRUE LIMIT 1"
      );

      if (configs.length === 0) {
        console.log("🤖 No active AI configuration found.");
        this.activeConfig = null;
        return null;
      }

      this.activeConfig = configs[0];
      console.log(`🤖 Active configuration loaded: ${this.activeConfig.persona_name} (${this.activeConfig.provider})`);
      return this.activeConfig;

    } catch (error) {
      console.error("Error loading AI configuration:", error);
      return null;
    } finally {
      if (connection) connection.release();
    }
  }

  /**
   * Check if within business hours
   */
  isWithinBusinessHours(config) {
    if (!config.business_hours_start || !config.business_hours_end) {
      return true;
    }

    const now = new Date();
    const currentDay = ["sunday", "monday", "tuesday", "wednesday", "thursday", "friday", "saturday"][now.getDay()];
    
    const businessDays = config.business_days ? config.business_days.split(",") : [];
    if (businessDays.length > 0 && !businessDays.includes(currentDay)) {
      console.log(`🤖 Outside business days. Today: ${currentDay}`);
      return false;
    }

    const currentTime = now.getHours() * 60 + now.getMinutes();
    const [startHour, startMin] = config.business_hours_start.split(":").map(Number);
    const [endHour, endMin] = config.business_hours_end.split(":").map(Number);
    
    const startTime = startHour * 60 + startMin;
    const endTime = endHour * 60 + endMin;

    const isWithin = currentTime >= startTime && currentTime <= endTime;
    
    if (!isWithin) {
      console.log(`🤖 Outside business hours. Hours: ${config.business_hours_start}-${config.business_hours_end}`);
    }

    return isWithin;
  }

  /**
   * Get user conversation history
   */
  async getConversationHistory(phoneNumber, limit = 10) {
    let connection;
    try {
      connection = await pool.getConnection();
      
      const [messages] = await connection.execute(
        `SELECT message_text, is_from_bot, timestamp 
         FROM messages 
         WHERE phone_number = ? 
         AND message_text IS NOT NULL 
         ORDER BY timestamp DESC 
         LIMIT ?`,
        [phoneNumber, limit]
      );

      return messages.reverse();

    } catch (error) {
      console.error("Error fetching history:", error);
      return [];
    } finally {
      if (connection) connection.release();
    }
  }

  /**
   * Get contact information
   */
  async getContactInfo(phoneNumber) {
    let connection;
    try {
      connection = await pool.getConnection();
      
      const [contacts] = await connection.execute(
        "SELECT name, email, tags, notes FROM contacts WHERE phone = ? LIMIT 1",
        [phoneNumber]
      );

      return contacts.length > 0 ? contacts[0] : null;

    } catch (error) {
      console.error("Error fetching contact:", error);
      return null;
    } finally {
      if (connection) connection.release();
    }
  }

  /**
   * Process message with DeepSeek
   */
  async processWithDeepSeek(config, userMessage, phoneNumber) {
    try {
      const apiKey = this.decryptData(config.api_key);
      
      if (!apiKey) {
        throw new Error("API key could not be decrypted");
      }

      const conversationHistory = await this.getConversationHistory(phoneNumber);
      const contactInfo = await this.getContactInfo(phoneNumber);

      const messages = [];

      let systemPrompt = config.system_prompt || "You are a helpful assistant.";
      
      if (contactInfo) {
        systemPrompt += `\n\nCustomer information:\n- Name: ${contactInfo.name || "Not provided"}`;
        if (contactInfo.email) systemPrompt += `\n- Email: ${contactInfo.email}`;
        if (contactInfo.tags) systemPrompt += `\n- Tags: ${contactInfo.tags}`;
        if (contactInfo.notes) systemPrompt += `\n- Notes: ${contactInfo.notes}`;
      }

      messages.push({
        role: "system",
        content: systemPrompt
      });

      conversationHistory.slice(-5).forEach(msg => {
        messages.push({
          role: msg.is_from_bot ? "assistant" : "user",
          content: msg.message_text
        });
      });

      messages.push({
        role: "user",
        content: userMessage
      });

      console.log(`🤖 Sending to DeepSeek (${config.model_name})...`);

      const startTime = Date.now();

      const response = await axios.post("https://api.deepseek.com/v1/chat/completions", {
        model: config.model_name,
        messages: messages,
        temperature: parseFloat(config.temperature) || 0.5,
        max_tokens: parseInt(config.max_tokens) || 70
      }, {
        headers: {
          "Authorization": `Bearer ${apiKey}`,
          "Content-Type": "application/json"
        },
        timeout: 30000
      });

      const responseTime = ((Date.now() - startTime) / 1000).toFixed(2);
      console.log(`✅ Response received from DeepSeek in ${responseTime}s`);

      return {
        success: true,
        response: response.data.choices[0].message.content,
        usage: response.data.usage,
        responseTime: parseFloat(responseTime),
        shouldRespond: true,
        provider: 'DeepSeek',
        model: config.model_name,
        emoji: '🤖'
      };

    } catch (error) {
      console.error("❌ Error processing with DeepSeek:", error.response?.data || error.message);
      return {
        success: false,
        shouldRespond: false,
        error: error.response?.data?.error?.message || error.message
      };
    }
  }

  /**
   * Process message with GPT (OpenAI) - DISABLED
   */
  async processWithGPT(config, userMessage, phoneNumber) {
    console.log('🤖 OpenAI is disabled. Use DeepSeek instead.');
    return { 
      success: false, 
      shouldRespond: false,
      response: 'GPT functionality disabled.' 
    };
  }

  /**
   * Save AI usage log to database
   */
  async saveAILog(phoneNumber, userMessage, aiResponse, responseTime, success) {
    let connection;
    try {
      connection = await pool.getConnection();
      
      await connection.execute(
        `INSERT INTO ai_usage_logs 
         (phone_number, user_message, ai_response, response_time, success, created_at) 
         VALUES (?, ?, ?, ?, ?, NOW())`,
        [phoneNumber, userMessage, aiResponse, responseTime, success]
      );

    } catch (error) {
      console.error("Error saving AI log:", error);
    } finally {
      if (connection) connection.release();
    }
  }

  /**
   * Main method: process message with AI
   * FIX: Now processes even with OpenAI disabled if there's a DeepSeek config
   */
  async processMessage(phoneNumber, userMessage) {
    try {
      // Load active configuration if needed
      if (!this.activeConfig) {
        await this.loadActiveConfig();
      }

      if (!this.activeConfig) {
        console.log("🤖 No active AI configuration. Returning without response.");
        return { 
          success: false, 
          shouldRespond: false,
          response: "Sorry, there is no active AI configuration at the moment." 
        };
      }

      // Check business hours
      if (!this.isWithinBusinessHours(this.activeConfig)) {
        console.log("🤖 Outside AI business hours. Returning without response.");
        return { 
          success: true,
          shouldRespond: false,
          response: this.activeConfig.out_of_hours_message || "Our AI assistant is offline at the moment. Please try again during business hours." 
        };
      }

      let aiResponse = { success: false, shouldRespond: false, response: "Unknown AI error." };

      // FIX: Process DeepSeek regardless of OpenAI status
      if (this.activeConfig.provider === "deepseek") {
        aiResponse = await this.processWithDeepSeek(this.activeConfig, userMessage, phoneNumber);
      } else if (this.activeConfig.provider === "openai") {
        // OpenAI disabled
        console.log('🤖 OpenAI is disabled. Configure DeepSeek as provider.');
        aiResponse = { 
          success: false, 
          shouldRespond: false,
          response: "OpenAI disabled. Use DeepSeek." 
        };
      } else {
        console.warn(`🤖 Unknown AI provider: ${this.activeConfig.provider}`);
        aiResponse = { 
          success: false, 
          shouldRespond: false,
          response: "AI provider not configured or unknown." 
        };
      }

      // Save AI usage log
      if (aiResponse.success) {
        await this.saveAILog(
          phoneNumber, 
          userMessage, 
          aiResponse.response, 
          aiResponse.responseTime, 
          aiResponse.success
        );
      }

      return aiResponse;

    } catch (error) {
      console.error("❌ General error processing message with AI:", error);
      return { 
        success: false, 
        shouldRespond: false,
        response: "Sorry, an error occurred while processing your request with AI." 
      };
    }
  }
}

module.exports = new AIService(process.env.ENABLE_OPENAI === 'true');