const dns = require('dns');
dns.setDefaultResultOrder('ipv4first');

const express = require('express');
const http = require('http');
const socketIo = require('socket.io');
const cors = require('cors');
const path = require('path');
const jwt = require('jsonwebtoken');
const helmet = require('helmet');
const compression = require('compression');
require('dotenv').config();

const { initDatabase } = require('./config/database');
const { initializeTranslations } = require('./config/init-translations');
const WhatsAppService = require('./services/whatsappService');
const { apiLimiter, authLimiter, sanitizeInput } = require('./middleware/security');
const { generateCSRFToken, validateCSRFToken } = require('./middleware/csrf');

// Validate required environment variables
if (!process.env.JWT_SECRET) {
  console.error('❌ ERROR: JWT_SECRET environment variable is required!');
  console.error('   Please set JWT_SECRET in your .env file');
  process.exit(1);
}

const app = express();
const server = http.createServer(app);

const io = socketIo(server, {
  cors: {
    origin: [
      "http://localhost:7000",
      "https://yourdomain.com",
      "http://yourdomain.com"
    ],
    methods: ["GET", "POST"],
    allowedHeaders: ["Content-Type", "Authorization"],
    credentials: true
  },
  allowEIO3: true,
  transports: ['websocket', 'polling']
});

// Security Middleware - Helmet.js (HTTP Security Headers)
app.use(helmet({
  contentSecurityPolicy: {
    directives: {
      defaultSrc: ["'self'"],
      styleSrc: ["'self'", "'unsafe-inline'", "https://cdnjs.cloudflare.com", "https://unpkg.com"],
      scriptSrc: ["'self'", "'unsafe-inline'", "https://cdn.jsdelivr.net", "https://cdnjs.cloudflare.com", "https://unpkg.com"],
      scriptSrcAttr: ["'unsafe-inline'"], // Allow inline event handlers (onclick, etc.)
      imgSrc: ["'self'", "data:", "https:", "http:"],
      connectSrc: ["'self'", "ws:", "wss:", "https://cdn.jsdelivr.net", "https://cdnjs.cloudflare.com", "https://unpkg.com"], // Allow CDN connections for source maps
      fontSrc: ["'self'", "https://cdnjs.cloudflare.com", "https://unpkg.com"],
    },
  },
  crossOriginEmbedderPolicy: false, // Disable for Socket.IO compatibility
}));

// Compression middleware (gzip)
app.use(compression());

// CORS Configuration
app.use(cors({
  origin: [
    "http://localhost:7000",
    "https://yourdomain.com",
    "http://yourdomain.com"
  ],
  methods: ["GET", "POST", "PUT", "DELETE"],
  allowedHeaders: ["Content-Type", "Authorization"],
  credentials: true
}));

// Rate Limiting - Apply to all API routes
app.use('/api', apiLimiter);

// CSRF Token Generation (for HTML forms)
app.use(generateCSRFToken);

// Serve static files FIRST (before body parsers)
app.use(express.static(path.join(__dirname, 'public')));
app.use('/uploads', express.static(path.join(__dirname, 'uploads')));

// Middlewares
// Check if it's not multipart/form-data before parsing JSON
app.use((req, res, next) => {
  // Skip body parsers for static files and upload routes
  if (req.path.startsWith('/css/') ||
      req.path.startsWith('/js/') ||
      req.path.startsWith('/images/') ||
      req.path.startsWith('/uploads/') ||
      req.path.includes('/import') || 
      req.path.includes('/upload')) {
    return next();
  }
  
  const contentType = req.headers['content-type'] || '';
  if (!contentType.includes('multipart/form-data')) {
    express.json({ limit: '50mb' })(req, res, next);
  } else {
    next();
  }
});
app.use((req, res, next) => {
  // Skip urlencoded for static files and upload routes
  if (req.path.startsWith('/css/') ||
      req.path.startsWith('/js/') ||
      req.path.startsWith('/images/') ||
      req.path.startsWith('/uploads/') ||
      req.path.includes('/import') || 
      req.path.includes('/upload')) {
    return next();
  }
  express.urlencoded({ extended: true, limit: '50mb' })(req, res, next);
});

// CSRF Protection (for HTML forms, after body parsing)
// Skip for static files
app.use((req, res, next) => {
  if (req.path.startsWith('/css/') ||
      req.path.startsWith('/js/') ||
      req.path.startsWith('/images/') ||
      req.path.startsWith('/uploads/')) {
    return next();
  }
  validateCSRFToken(req, res, next);
});

// Input Sanitization - Sanitize all inputs to prevent XSS
// Skip for file uploads and static files
app.use((req, res, next) => {
  if (req.path.includes('/import') || 
      req.path.includes('/upload') || 
      req.path.startsWith('/uploads/') ||
      req.path.startsWith('/css/') ||
      req.path.startsWith('/js/') ||
      req.path.startsWith('/images/')) {
    return next();
  }
  sanitizeInput(req, res, next);
});

// Middleware to validate JWT token
function requireAuth(req, res, next) {
  const authHeader = req.headers.authorization;

  if (!authHeader || !authHeader.startsWith('Bearer ')) {
    return res.status(401).json({ error: 'Access denied. Token not provided.' });
  }

  const token = authHeader.split(' ')[1];

  try {
    // JWT_SECRET is now required (validated at startup)
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    req.user = decoded;
    next();
  } catch (error) {
    // Log error in production only (without sensitive details)
    if (process.env.NODE_ENV === 'production') {
      console.error('Token validation failed');
    } else {
      console.error('Error validating token:', error.message);
    }
    return res.status(401).json({ error: 'Invalid or expired token.' });
  }
}

// Middleware to validate if user is admin
function requireAdmin(req, res, next) {
  if (req.user.role !== 'admin') {
    return res.status(403).json({ error: 'Access denied. Administrators only.' });
  }
  next();
}

// API Routes
// Authentication routes with strict rate limiting
app.use('/api/auth', authLimiter, require('./routes/auth'));

// Public route for profile settings (used on login screen)
app.use('/api/public', require('./routes/public'));


// Routes protected by authentication
app.use('/api/admin', requireAuth, requireAdmin, require('./routes/admin'));
app.use('/api/chat', requireAuth, require('./routes/chat'));
app.use('/api/payments', requireAuth, require('./routes/payments'));
app.use('/api/translations', requireAuth, require('./routes/translations'));
app.use('/api/pwa-notifications', requireAuth, require('./routes/pwa-notifications'));

// Contact routes (import route uses multer which needs to process before body parser)
app.use('/api', requireAuth, require('./routes/contacts'));

// FAQ route (with media upload)
app.use('/api/faqs', require('./routes/faq'));

// AI configuration route
app.use('/api/ai-config', requireAuth, requireAdmin, require('./routes/ai-config'));

// Routes to serve HTML pages - NO SERVER AUTHENTICATION
app.get('/', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'index.html'));
});

app.get('/login', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'index.html'));
});

app.get('/admin', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'admin', 'index.html'));
});

app.get('/user', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'user', 'index.html'));
});

app.get('/mobile', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'mobile.html'));
});

app.get('/faq-admin', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'faq-admin.html'));
});

app.get('/mass-admin', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'mass-admin.html'));
});

app.get('/payments-admin', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'payments-admin.html'));
});

app.get('/translations-admin', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'translations-admin.html'));
});

app.get('/ai-config-admin', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'ai-config-admin.html'));
});


// Ignore common browser/development tool requests
app.get('/.well-known/*', (req, res) => {
  res.status(404).end();
});

app.get('/favicon.ico', (req, res) => {
  res.status(204).end();
});

// Import error handler
const { errorHandler, notFoundHandler } = require('./middleware/errorHandler');

// 404 Handler - Must be before error handler
app.use(notFoundHandler);

// Error handling middleware - Must be last
app.use(errorHandler);

// Initialize WhatsApp Service
const whatsappService = new WhatsAppService(io);

// Make whatsappService available to routes
app.set('whatsappService', whatsappService);
app.set('io', io);

// Socket.IO
io.on('connection', (socket) => {
  console.log('Client connected:', socket.id);

  socket.on('join-room', (room) => {
    socket.join(room);
    console.log(`Client ${socket.id} joined room: ${room}`);
  });

  // New: allow joining multiple rooms (for departments)
  socket.on('join-department', (department) => {
    const departmentRoom = `SETOR:${department}`;
    socket.join(departmentRoom);
    console.log(`Client ${socket.id} joined department: ${departmentRoom}`);
  });

  socket.on('error', (error) => {
    console.error('Socket error:', error);
  });

  socket.on('disconnect', (reason) => {
    console.log('Client disconnected:', socket.id, 'Reason:', reason);
  });
});

// Unhandled error handling
process.on('unhandledRejection', (err) => {
  console.error('Unhandled promise rejection:', err);
});

process.on('uncaughtException', (err) => {
  console.error('Uncaught exception:', err);
  process.exit(1);
});

// Initialize server
const PORT = process.env.PORT || 7000;

async function startServer() {
  try {
    console.log('Initializing database...');
    await initDatabase();
    console.log('Database initialized!');

    console.log('Initializing translations...');
    await initializeTranslations();
    console.log('Translations initialized!');

    console.log('Initializing WhatsApp Service...');
    await whatsappService.initialize();
    console.log('WhatsApp Service initialized!');

    server.listen(PORT, '0.0.0.0', () => {
      console.log(`Server running on port ${PORT}`);
      console.log(`Access: http://localhost:${PORT}`);
      console.log(`Mode: ${process.env.NODE_ENV || 'development'}`);
    });
  } catch (error) {
    console.error('Error starting server:', error);
    process.exit(1);
  }
}

startServer();

