const express = require('express');
const multer = require('multer');
const fs = require('fs');
const path = require('path');
const { pool } = require('../config/database');
const { authenticateToken, requireAdmin } = require('../middleware/auth');

const router = express.Router();
router.use(authenticateToken);

// Configure multer for translation file upload
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const uploadDir = 'translations/';
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: function (req, file, cb) {
    cb(null, file.originalname);
  }
});

const upload = multer({ 
  storage: storage,
  fileFilter: function (req, file, cb) {
    if (file.mimetype === 'application/json' || path.extname(file.originalname) === '.json') {
      cb(null, true);
    } else {
      cb(new Error('Only JSON files are allowed'), false);
    }
  }
});

// ===== AVAILABLE LANGUAGES =====

// Endpoint to fetch all translations (for i18n.js)
router.get('/', async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const [rows] = await connection.execute(
      'SELECT * FROM translations ORDER BY language_code, translation_key'
    );
    
    res.json(rows);
  } catch (error) {
    console.error('Error fetching translations:', error);
    res.status(500).json({ error: 'Error fetching translations' });
  } finally {
    if (connection) connection.release();
  }
});

// Endpoint to add/update translations in batch
router.post('/', async (req, res) => {
  let connection;
  try {
    const { language_code, translations } = req.body;
    
    if (!language_code || !translations) {
      return res.status(400).json({ error: 'Incomplete data' });
    }
    
    connection = await pool.getConnection();
    
    // Fetch language name
    const [langInfo] = await connection.execute(
      'SELECT language_name FROM translations WHERE language_code = ? LIMIT 1',
      [language_code]
    );
    
    const language_name = langInfo.length > 0 ? langInfo[0].language_name : language_code.toUpperCase();
    
    // Insert or update each translation
    for (const [key, value] of Object.entries(translations)) {
      await connection.execute(
        `INSERT INTO translations (language_code, language_name, translation_key, translation_value) 
         VALUES (?, ?, ?, ?) 
         ON DUPLICATE KEY UPDATE translation_value = ?`,
        [language_code, language_name, key, value, value]
      );
    }
    
    res.json({ success: true, message: 'Translations saved successfully' });
  } catch (error) {
    console.error('Error saving translations:', error);
    res.status(500).json({ error: 'Error saving translations' });
  } finally {
    if (connection) connection.release();
  }
});

// List available languages
router.get('/languages', async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const [languages] = await connection.execute(`
      SELECT DISTINCT language_code, language_name, 
             COUNT(*) as translation_count,
             MAX(updated_at) as last_updated
      FROM translations 
      GROUP BY language_code, language_name
      ORDER BY language_name ASC
    `);
    
    res.json(languages);
  } catch (error) {
    console.error('Error fetching languages:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Create new language
router.post('/languages', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { language_code, language_name } = req.body;
    
    if (!language_code || !language_name) {
      return res.status(400).json({ error: 'Language code and name are required' });
    }

    // Validate language code (ISO 639-1 format or similar)
    if (!/^[a-z]{2}(-[A-Z]{2})?$/.test(language_code)) {
      return res.status(400).json({ 
        error: 'Language code must be in ISO format (e.g., pt, en, es, pt-BR)' 
      });
    }

    connection = await pool.getConnection();
    
    // Check if already exists
    const [existing] = await connection.execute(
      'SELECT language_code FROM translations WHERE language_code = ? LIMIT 1',
      [language_code]
    );

    if (existing.length > 0) {
      return res.status(400).json({ error: 'Language already exists' });
    }

    // Fetch default English strings to create base
    const [englishStrings] = await connection.execute(
      'SELECT translation_key FROM translations WHERE language_code = ? GROUP BY translation_key',
      ['en']
    );

    // If there are no English strings, create some basic ones
    const baseStrings = englishStrings.length > 0 ? 
      englishStrings.map(s => s.translation_key) : 
      getDefaultTranslationKeys();

    // Insert base strings for the new language
    for (const key of baseStrings) {
      await connection.execute(`
        INSERT INTO translations (language_code, language_name, translation_key, translation_value) 
        VALUES (?, ?, ?, ?)
      `, [language_code, language_name, key, '']);
    }

    res.json({ 
      success: true, 
      message: `Language ${language_name} created with ${baseStrings.length} base strings` 
    });
  } catch (error) {
    console.error('Error creating language:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Delete language
router.delete('/languages/:language_code', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { language_code } = req.params;
    
    // Do not allow deleting English (default language)
    if (language_code === 'en') {
      return res.status(400).json({ error: 'Cannot delete the default language' });
    }

    connection = await pool.getConnection();
    
    // Check if language exists
    const [existing] = await connection.execute(
      'SELECT COUNT(*) as count FROM translations WHERE language_code = ?',
      [language_code]
    );

    if (existing[0].count === 0) {
      return res.status(404).json({ error: 'Language not found' });
    }

    // Delete all translations for this language
    await connection.execute(
      'DELETE FROM translations WHERE language_code = ?',
      [language_code]
    );

    res.json({ 
      success: true, 
      message: `Language ${language_code} deleted successfully` 
    });
  } catch (error) {
    console.error('Error deleting language:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== TRANSLATIONS =====

// List translations for a language
router.get('/strings/:language_code', async (req, res) => {
  let connection;
  try {
    const { language_code } = req.params;
    const { search, page = 1, limit = 50 } = req.query;
    const offset = (page - 1) * limit;

    connection = await pool.getConnection();
    
    let query = `
      SELECT translation_key, translation_value, updated_at
      FROM translations 
      WHERE language_code = ?
    `;
    let params = [language_code];

    if (search) {
      query += ' AND (translation_key LIKE ? OR translation_value LIKE ?)';
      params.push(`%${search}%`, `%${search}%`);
    }

    query += ' ORDER BY translation_key ASC LIMIT ? OFFSET ?';
    params.push(parseInt(limit), parseInt(offset));

    const [translations] = await connection.execute(query, params);

    // Count total
    let countQuery = 'SELECT COUNT(*) as total FROM translations WHERE language_code = ?';
    let countParams = [language_code];

    if (search) {
      countQuery += ' AND (translation_key LIKE ? OR translation_value LIKE ?)';
      countParams.push(`%${search}%`, `%${search}%`);
    }

    const [countResult] = await connection.execute(countQuery, countParams);

    res.json({
      translations,
      total: countResult[0].total,
      page: parseInt(page),
      limit: parseInt(limit),
      totalPages: Math.ceil(countResult[0].total / limit)
    });
  } catch (error) {
    console.error('Error fetching translations:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Create new translation string
router.post('/strings/:language_code', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { language_code } = req.params;
    const { translation_key, translation_value } = req.body;

    if (!translation_key) {
      return res.status(400).json({ error: 'translation_key is required' });
    }

    connection = await pool.getConnection();
    
    // Check if string already exists
    const [existing] = await connection.execute(
      'SELECT id FROM translations WHERE language_code = ? AND translation_key = ?',
      [language_code, translation_key]
    );

    if (existing.length > 0) {
      return res.status(400).json({ error: 'Translation string already exists. Use PUT to update.' });
    }

    // Fetch language name
    const [langInfo] = await connection.execute(
      'SELECT language_name FROM translations WHERE language_code = ? LIMIT 1',
      [language_code]
    );

    if (langInfo.length === 0) {
      return res.status(404).json({ error: 'Language not found' });
    }

    const language_name = langInfo[0].language_name;

    // Insert new string
    await connection.execute(`
      INSERT INTO translations (language_code, language_name, translation_key, translation_value) 
      VALUES (?, ?, ?, ?)
    `, [language_code, language_name, translation_key, translation_value || '']);

    // Add the same key to all other languages
    const [allLanguages] = await connection.execute(
      'SELECT DISTINCT language_code, language_name FROM translations WHERE language_code != ?',
      [language_code]
    );

    for (const lang of allLanguages) {
      await connection.execute(`
        INSERT IGNORE INTO translations (language_code, language_name, translation_key, translation_value) 
        VALUES (?, ?, ?, ?)
      `, [lang.language_code, lang.language_name, translation_key, '']);
    }

    res.json({ success: true, message: 'Translation string created successfully' });
  } catch (error) {
    console.error('Error creating translation string:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Update individual translation
router.put('/strings/:language_code/:translation_key', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { language_code, translation_key } = req.params;
    const { translation_value } = req.body;

    connection = await pool.getConnection();
    
    const [result] = await connection.execute(`
      UPDATE translations 
      SET translation_value = ?, updated_at = CURRENT_TIMESTAMP 
      WHERE language_code = ? AND translation_key = ?
    `, [translation_value, language_code, decodeURIComponent(translation_key)]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ error: 'Translation not found' });
    }

    res.json({ success: true, message: 'Translation updated successfully' });
  } catch (error) {
    console.error('Error updating translation:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Delete translation string
router.delete('/strings/:language_code/:translation_key', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { language_code, translation_key } = req.params;

    connection = await pool.getConnection();
    
    const [result] = await connection.execute(
      'DELETE FROM translations WHERE language_code = ? AND translation_key = ?',
      [language_code, decodeURIComponent(translation_key)]
    );

    if (result.affectedRows === 0) {
      return res.status(404).json({ error: 'Translation not found' });
    }

    res.json({ success: true, message: 'String deleted successfully' });
  } catch (error) {
    console.error('Error deleting string:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== FILE DOWNLOADS =====

// Download English translation file (base)
router.get('/download/base', requireAdmin, async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    
    // Fetch all base strings (Portuguese or system)
    const [baseStrings] = await connection.execute(`
      SELECT DISTINCT translation_key 
      FROM translations 
      ORDER BY translation_key ASC
    `);

    // If there are no strings, create some default ones
    const strings = baseStrings.length > 0 ? 
      baseStrings.map(s => s.translation_key) : 
      getDefaultTranslationKeys();

    // Create JSON object with empty strings for translation
    const translationObject = {};
    strings.forEach(key => {
      translationObject[key] = '';
    });

    // Set headers for download
    res.setHeader('Content-Type', 'application/json');
    res.setHeader('Content-Disposition', 'attachment; filename="en.json"');
    
    res.json(translationObject);
  } catch (error) {
    console.error('Error generating base file:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Download translation file for a specific language
router.get('/download/:language_code', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { language_code } = req.params;

    connection = await pool.getConnection();
    
    const [translations] = await connection.execute(
      'SELECT translation_key, translation_value FROM translations WHERE language_code = ? ORDER BY translation_key ASC',
      [language_code]
    );

    if (translations.length === 0) {
      return res.status(404).json({ error: 'Language not found' });
    }

    // Create JSON object
    const translationObject = {};
    translations.forEach(t => {
      translationObject[t.translation_key] = t.translation_value;
    });

    // Set headers for download
    res.setHeader('Content-Type', 'application/json');
    res.setHeader('Content-Disposition', `attachment; filename="${language_code}.json"`);
    
    res.json(translationObject);
  } catch (error) {
    console.error('Error downloading translation:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== FILE UPLOADS =====

// Upload translation file
router.post('/upload', requireAdmin, upload.single('translationFile'), async (req, res) => {
  let connection;
  try {
    if (!req.file) {
      return res.status(400).json({ error: 'Translation file is required' });
    }

    const filePath = req.file.path;
    const fileName = req.file.originalname;
    
    // Extract language code from file name (e.g., pt.json, es.json, pt-BR.json)
    const languageCode = path.basename(fileName, '.json');
    
    // Validate language code
    if (!/^[a-z]{2}(-[A-Z]{2})?$/.test(languageCode)) {
      fs.unlinkSync(filePath); // Clean file
      return res.status(400).json({ 
        error: 'File name must follow language code pattern (e.g., pt.json, en.json, es.json)' 
      });
    }

    // Read and validate JSON file
    let translationData;
    try {
      const fileContent = fs.readFileSync(filePath, 'utf8');
      translationData = JSON.parse(fileContent);
    } catch (parseError) {
      fs.unlinkSync(filePath); // Clean file
      return res.status(400).json({ error: 'Invalid JSON file' });
    }

    // Validate JSON structure
    if (typeof translationData !== 'object' || Array.isArray(translationData)) {
      fs.unlinkSync(filePath); // Clean file
      return res.status(400).json({ error: 'File must contain a JSON object with translation keys and values' });
    }

    connection = await pool.getConnection();
    await connection.beginTransaction();

    // Determine language name
    let languageName = getLanguageName(languageCode);
    
    // Check if language already exists to get correct name
    const [existingLang] = await connection.execute(
      'SELECT language_name FROM translations WHERE language_code = ? LIMIT 1',
      [languageCode]
    );
    
    if (existingLang.length > 0) {
      languageName = existingLang[0].language_name;
    }

    let imported = 0;
    let updated = 0;
    let errors = [];

    // Process each translation
    for (const [key, value] of Object.entries(translationData)) {
      try {
        // Check if already exists
        const [existing] = await connection.execute(
          'SELECT id FROM translations WHERE language_code = ? AND translation_key = ?',
          [languageCode, key]
        );

        if (existing.length > 0) {
          // Update existing
          await connection.execute(`
            UPDATE translations 
            SET translation_value = ?, updated_at = CURRENT_TIMESTAMP 
            WHERE language_code = ? AND translation_key = ?
          `, [value, languageCode, key]);
          updated++;
        } else {
          // Insert new
          await connection.execute(`
            INSERT INTO translations (language_code, language_name, translation_key, translation_value) 
            VALUES (?, ?, ?, ?)
          `, [languageCode, languageName, key, value]);
          imported++;
        }
      } catch (error) {
        errors.push(`Error in key "${key}": ${error.message}`);
      }
    }

    await connection.commit();
    
    // Clean temporary file
    fs.unlinkSync(filePath);

    res.json({
      success: true,
      message: 'Translation imported successfully',
      language_code: languageCode,
      language_name: languageName,
      imported,
      updated,
      errors: errors.slice(0, 10), // Limit errors shown
      totalErrors: errors.length
    });
  } catch (error) {
    if (connection) await connection.rollback();
    console.error('Error importing translation:', error);
    
    // Clean file in case of error
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path);
    }
    
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== APPLY TRANSLATIONS =====

// Apply active translation in system
router.post('/apply/:language_code', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { language_code } = req.params;

    connection = await pool.getConnection();
    
    // Check if language exists
    const [translations] = await connection.execute(
      'SELECT COUNT(*) as count FROM translations WHERE language_code = ?',
      [language_code]
    );

    if (translations[0].count === 0) {
      return res.status(404).json({ error: 'Language not found' });
    }

    // Save active language in settings
    await connection.execute(`
      INSERT INTO bot_settings (setting_key, setting_value) 
      VALUES ('active_language', ?) 
      ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)
    `, [language_code]);

    res.json({ 
      success: true, 
      message: `Language ${language_code} applied as active in system` 
    });
  } catch (error) {
    console.error('Error applying translation:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Get current active language
router.get('/active', async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    
    const [setting] = await connection.execute(
      'SELECT setting_value FROM bot_settings WHERE setting_key = ?',
      ['active_language']
    );

    const activeLanguage = setting.length > 0 ? setting[0].setting_value : 'en';

    // Fetch active language information
    const [langInfo] = await connection.execute(
      'SELECT language_code, language_name, COUNT(*) as translation_count FROM translations WHERE language_code = ? GROUP BY language_code, language_name',
      [activeLanguage]
    );

    res.json({
      active_language: activeLanguage,
      language_info: langInfo.length > 0 ? langInfo[0] : null
    });
  } catch (error) {
    console.error('Error fetching active language:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== HELPER FUNCTIONS =====

function getDefaultTranslationKeys() {
  return [
    'welcome_message',
    'bot_name',
    'business_hours',
    'after_hours_message',
    'help_message',
    'error_message',
    'thank_you_message',
    'goodbye_message',
    'menu_option_1',
    'menu_option_2',
    'menu_option_3',
    'contact_info',
    'store_address',
    'payment_methods',
    'shipping_info',
    'return_policy',
    'faq_title',
    'support_message',
    'loading_message',
    'success_message',
    'cancel_message',
    'confirm_message',
    'invalid_option',
    'try_again_message',
    'main_menu',
    'back_to_menu'
  ];
}

function getLanguageName(languageCode) {
  const languageNames = {
    'pt': 'Português',
    'pt-BR': 'Português (Brasil)',
    'en': 'English',
    'es': 'Español',
    'fr': 'Français',
    'it': 'Italiano',
    'de': 'Deutsch',
    'ru': 'Русский',
    'zh': '中文',
    'ja': '日本語',
    'ko': '한국어',
    'ar': 'العربية'
  };
  
  return languageNames[languageCode] || languageCode.toUpperCase();
}

module.exports = router;

