const express = require('express');
const { pool } = require('../config/database');
const { authenticateToken } = require('../middleware/auth');

const router = express.Router();

// Configure simple PWA notifications
router.post('/register', authenticateToken, async (req, res) => {
  let connection;
  try {
    const { subscription } = req.body;
    
    if (!subscription || !subscription.endpoint) {
      return res.status(400).json({ error: 'Invalid subscription' });
    }

    connection = await pool.getConnection();
    
    // Check if subscription already exists for this user
    const [existing] = await connection.execute(
      'SELECT id FROM pwa_subscriptions WHERE user_id = ? AND endpoint = ?',
      [req.user.id, subscription.endpoint]
    );

    if (existing.length === 0) {
      // Create new subscription
      await connection.execute(`
        INSERT INTO pwa_subscriptions (user_id, endpoint, subscription_data, active)
        VALUES (?, ?, ?, TRUE)
      `, [
        req.user.id,
        subscription.endpoint,
        JSON.stringify(subscription)
      ]);
    }

    res.json({ 
      success: true, 
      message: 'Notifications activated successfully'
    });
  } catch (error) {
    console.error('Error registering PWA subscription:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Send new WhatsApp message notification
router.post('/new-message', authenticateToken, async (req, res) => {
  let connection;
  try {
    const { userId, senderName, message } = req.body;
    
    connection = await pool.getConnection();
    
    // Fetch user's active subscriptions
    const [subscriptions] = await connection.execute(
      'SELECT subscription_data FROM pwa_subscriptions WHERE user_id = ? AND active = TRUE',
      [userId]
    );

    if (subscriptions.length === 0) {
      return res.json({ message: 'User does not have notifications enabled' });
    }

    // Prepare notification
    const notificationPayload = {
      title: `New message - ${senderName}`,
      body: message.length > 100 ? message.substring(0, 100) + '...' : message,
      icon: '/images/icon-192x192.png',
      badge: '/images/badge-72x72.png',
      tag: 'new-message',
      data: {
        type: 'whatsapp-message',
        sender: senderName,
        url: '/user'
      }
    };

    // Simulate sending (here you would implement real sending via service worker)
    console.log('PWA Notification:', notificationPayload);

    res.json({
      success: true,
      message: 'Notification sent'
    });
  } catch (error) {
    console.error('Error sending PWA notification:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

module.exports = router;
