const express = require('express');
const { pool } = require('../config/database');
const { authenticateToken } = require('../middleware/auth');

const router = express.Router();
router.use(authenticateToken);

// Get message queue statistics
router.get('/stats', async (req, res) => {
    let connection;
    try {
        connection = await pool.getConnection();
        
        // Queue statistics
        const [queueStats] = await connection.execute(`
            SELECT 
                status,
                COUNT(*) as count,
                AVG(attempts) as avg_attempts
            FROM message_queue 
            GROUP BY status
        `);
        
        // Connection status
        const [connectionStatus] = await connection.execute(
            'SELECT * FROM whatsapp_connection_status ORDER BY id DESC LIMIT 1'
        );
        
        // Pending messages by priority
        const [priorityStats] = await connection.execute(`
            SELECT 
                priority,
                COUNT(*) as count
            FROM message_queue 
            WHERE status IN ('pending', 'failed')
            GROUP BY priority
            ORDER BY priority DESC
        `);
        
        // Recent failed messages
        const [recentFailures] = await connection.execute(`
            SELECT 
                mq.*,
                c.contact_name,
                u.username as sender_name
            FROM message_queue mq
            LEFT JOIN conversations c ON mq.conversation_id = c.id
            LEFT JOIN users u ON mq.sender_user_id = u.id
            WHERE mq.status = 'failed'
            ORDER BY mq.updated_at DESC
            LIMIT 10
        `);
        
        res.json({
            queue: queueStats.reduce((acc, stat) => {
                acc[stat.status] = {
                    count: stat.count,
                    avgAttempts: parseFloat(stat.avg_attempts) || 0
                };
                return acc;
            }, {}),
            connection: connectionStatus[0] || null,
            priority: priorityStats,
            recentFailures
        });
        
    } catch (error) {
        console.error('Error getting queue statistics:', error);
        res.status(500).json({ error: 'Internal server error' });
    } finally {
        if (connection) connection.release();
    }
});

// List messages in queue
router.get('/messages', async (req, res) => {
    let connection;
    try {
        const { status, limit = 50, offset = 0 } = req.query;
        
        connection = await pool.getConnection();
        
        let query = `
            SELECT 
                mq.*,
                c.contact_name,
                u.username as sender_name
            FROM message_queue mq
            LEFT JOIN conversations c ON mq.conversation_id = c.id
            LEFT JOIN users u ON mq.sender_user_id = u.id
            WHERE 1=1
        `;
        
        const params = [];
        
        if (status) {
            query += ' AND mq.status = ?';
            params.push(status);
        }
        
        query += ' ORDER BY mq.priority DESC, mq.created_at DESC LIMIT ? OFFSET ?';
        params.push(parseInt(limit), parseInt(offset));
        
        const [messages] = await connection.execute(query, params);
        
        res.json(messages);
        
    } catch (error) {
        console.error('Error listing queue messages:', error);
        res.status(500).json({ error: 'Internal server error' });
    } finally {
        if (connection) connection.release();
    }
});

// Retry failed message
router.post('/messages/:id/retry', async (req, res) => {
    let connection;
    try {
        const { id } = req.params;
        
        connection = await pool.getConnection();
        
        // Check if message exists and is failed
        const [messages] = await connection.execute(
            'SELECT * FROM message_queue WHERE id = ? AND status = "failed"',
            [id]
        );
        
        if (messages.length === 0) {
            return res.status(404).json({ error: 'Message not found or not failed' });
        }
        
        // Reset status for retry
        await connection.execute(`
            UPDATE message_queue 
            SET status = 'pending', attempts = 0, error_message = NULL, last_attempt_at = NULL, updated_at = NOW()
            WHERE id = ?
        `, [id]);
        
        // Get queue service to process immediately
        const messageQueueService = req.app.get('messageQueueService');
        if (messageQueueService) {
            // Try to process immediately if connected
            setTimeout(() => messageQueueService.processMessage(id), 1000);
        }
        
        res.json({ message: 'Message marked for retry' });
        
    } catch (error) {
        console.error('Error retrying message:', error);
        res.status(500).json({ error: 'Internal server error' });
    } finally {
        if (connection) connection.release();
    }
});

// Cancel message in queue
router.delete('/messages/:id', async (req, res) => {
    let connection;
    try {
        const { id } = req.params;
        
        connection = await pool.getConnection();
        
        // Check if message can be cancelled
        const [result] = await connection.execute(
            'DELETE FROM message_queue WHERE id = ? AND status IN ("pending", "failed")',
            [id]
        );
        
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Message not found or cannot be cancelled' });
        }
        
        // Notify via Socket.IO
        const io = req.app.get('io');
        if (io) {
            io.emit('message-cancelled', { queueId: parseInt(id) });
        }
        
        res.json({ message: 'Message cancelled successfully' });
        
    } catch (error) {
        console.error('Error cancelling message:', error);
        res.status(500).json({ error: 'Internal server error' });
    } finally {
        if (connection) connection.release();
    }
});

// Change message priority
router.patch('/messages/:id/priority', async (req, res) => {
    let connection;
    try {
        const { id } = req.params;
        const { priority } = req.body;
        
        if (!priority || priority < 1 || priority > 5) {
            return res.status(400).json({ error: 'Priority must be between 1 and 5' });
        }
        
        connection = await pool.getConnection();
        
        const [result] = await connection.execute(
            'UPDATE message_queue SET priority = ?, updated_at = NOW() WHERE id = ? AND status IN ("pending", "failed")',
            [priority, id]
        );
        
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Message not found or cannot be changed' });
        }
        
        res.json({ message: 'Priority changed successfully' });
        
    } catch (error) {
        console.error('Error changing priority:', error);
        res.status(500).json({ error: 'Internal server error' });
    } finally {
        if (connection) connection.release();
    }
});

// Clean up old messages from queue
router.post('/cleanup', async (req, res) => {
    let connection;
    try {
        const { days = 7 } = req.body;
        
        if (req.user.role !== 'admin') {
            return res.status(403).json({ error: 'Access denied. Administrators only.' });
        }
        
        connection = await pool.getConnection();
        
        const [result] = await connection.execute(`
            DELETE FROM message_queue 
            WHERE status IN ('sent', 'failed') 
            AND created_at < DATE_SUB(NOW(), INTERVAL ? DAY)
        `, [days]);
        
        res.json({ 
            message: 'Cleanup completed',
            removedCount: result.affectedRows
        });
        
    } catch (error) {
        console.error('Error cleaning queue:', error);
        res.status(500).json({ error: 'Internal server error' });
    } finally {
        if (connection) connection.release();
    }
});

// Force queue processing
router.post('/process', async (req, res) => {
    try {
        if (req.user.role !== 'admin') {
            return res.status(403).json({ error: 'Access denied. Administrators only.' });
        }
        
        const messageQueueService = req.app.get('messageQueueService');
        if (!messageQueueService) {
            return res.status(503).json({ error: 'Queue service not available' });
        }
        
        // Force processing
        setTimeout(() => messageQueueService.processQueue(), 1000);
        
        res.json({ message: 'Queue processing started' });
        
    } catch (error) {
        console.error('Error forcing processing:', error);
        res.status(500).json({ error: 'Internal server error' });
    }
});

// Get history of a specific conversation in queue
router.get('/conversations/:id/history', async (req, res) => {
    let connection;
    try {
        const { id } = req.params;
        
        connection = await pool.getConnection();
        
        const [messages] = await connection.execute(`
            SELECT 
                mq.*,
                u.username as sender_name
            FROM message_queue mq
            LEFT JOIN users u ON mq.sender_user_id = u.id
            WHERE mq.conversation_id = ?
            ORDER BY mq.created_at DESC
        `, [id]);
        
        res.json(messages);
        
    } catch (error) {
        console.error('Error getting conversation history:', error);
        res.status(500).json({ error: 'Internal server error' });
    } finally {
        if (connection) connection.release();
    }
});

// Configure queue settings
router.post('/config', async (req, res) => {
    let connection;
    try {
        if (req.user.role !== 'admin') {
            return res.status(403).json({ error: 'Access denied. Administrators only.' });
        }
        
        const { maxRetries, processInterval, connectionCheckInterval } = req.body;
        
        connection = await pool.getConnection();
        
        // Save settings in bot_settings
        if (maxRetries !== undefined) {
            await connection.execute(
                'INSERT INTO bot_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)',
                ['queue_max_retries', maxRetries.toString()]
            );
        }
        
        if (processInterval !== undefined) {
            await connection.execute(
                'INSERT INTO bot_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)',
                ['queue_process_interval', processInterval.toString()]
            );
        }
        
        if (connectionCheckInterval !== undefined) {
            await connection.execute(
                'INSERT INTO bot_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)',
                ['queue_connection_check_interval', connectionCheckInterval.toString()]
            );
        }
        
        res.json({ message: 'Settings saved successfully' });
        
    } catch (error) {
        console.error('Error saving settings:', error);
        res.status(500).json({ error: 'Internal server error' });
    } finally {
        if (connection) connection.release();
    }
});

// Get queue settings
router.get('/config', async (req, res) => {
    let connection;
    try {
        connection = await pool.getConnection();
        
        const [settings] = await connection.execute(`
            SELECT setting_key, setting_value 
            FROM bot_settings 
            WHERE setting_key LIKE 'queue_%'
        `);
        
        const config = settings.reduce((acc, setting) => {
            acc[setting.setting_key] = setting.setting_value;
            return acc;
        }, {});
        
        res.json(config);
        
    } catch (error) {
        console.error('Error getting settings:', error);
        res.status(500).json({ error: 'Internal server error' });
    } finally {
        if (connection) connection.release();
    }
});

module.exports = router;
