const express = require('express');
const { pool } = require('../config/database');
const { authenticateToken, requireAdmin } = require('../middleware/auth');
const router = express.Router();

// Middleware for authentication
router.use(authenticateToken);

// ===== WELCOME MESSAGES =====

// List welcome messages
router.get('/welcome-messages', async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const [messages] = await connection.execute(`
      SELECT * FROM welcome_messages 
      WHERE active = TRUE 
      ORDER BY order_position ASC
    `);
    res.json(messages);
  } catch (error) {
    console.error('Error fetching welcome messages:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Create/Update welcome message (Admin)
router.post('/welcome-messages', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { messages } = req.body;
    
    if (!Array.isArray(messages) || messages.length === 0) {
      return res.status(400).json({ error: 'Messages are required' });
    }

    connection = await pool.getConnection();
    await connection.beginTransaction();

    // Deactivate all existing messages
    await connection.execute('UPDATE welcome_messages SET active = FALSE');

    // Insert new messages (without media)
    for (let i = 0; i < messages.length; i++) {
      const message = messages[i];
      
      await connection.execute(`
        INSERT INTO welcome_messages (message_text, order_position, active) 
        VALUES (?, ?, TRUE)
      `, [message.text, message.line]);
    }

    await connection.commit();
    res.json({ success: true, message: 'Welcome messages updated' });
  } catch (error) {
    if (connection) await connection.rollback();
    console.error('Error saving welcome messages:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== PLACEHOLDERS =====

// List available placeholders
router.get('/placeholders', async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const [placeholders] = await connection.execute(`
      SELECT * FROM message_placeholders 
      WHERE active = TRUE 
      ORDER BY placeholder_key ASC
    `);
    res.json(placeholders);
  } catch (error) {
    console.error('Error fetching placeholders:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Create/Update placeholder (Admin)
router.post('/placeholders', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { placeholder_key, placeholder_value, description } = req.body;
    
    if (!placeholder_key || !placeholder_value) {
      return res.status(400).json({ error: 'Placeholder key and value are required' });
    }

    connection = await pool.getConnection();
    await connection.execute(`
      INSERT INTO message_placeholders (placeholder_key, placeholder_value, description) 
      VALUES (?, ?, ?) 
      ON DUPLICATE KEY UPDATE 
      placeholder_value = VALUES(placeholder_value),
      description = VALUES(description)
    `, [placeholder_key, placeholder_value, description]);

    res.json({ success: true, message: 'Placeholder saved successfully' });
  } catch (error) {
    console.error('Error saving placeholder:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== IMPROVED FAQs =====

// List FAQs with advanced settings
router.get('/faqs', async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const [faqs] = await connection.execute(`
      SELECT * FROM faqs 
      WHERE active = TRUE 
      ORDER BY order_position ASC, created_at DESC
    `);
    res.json(faqs);
  } catch (error) {
    console.error('Error fetching FAQs:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Create FAQ (Admin)
router.post('/faqs', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { 
      question, 
      answer, 
      emoji, 
      reaction_time = 3, 
      response_time = 7,
      schedule_hours = '08:00-18:00',
      schedule_days = 'monday,tuesday,wednesday,thursday,friday,saturday',
      order_position = 1
    } = req.body;
    
    if (!question || !answer) {
      return res.status(400).json({ error: 'Question and answer are required' });
    }

    connection = await pool.getConnection();
    const [result] = await connection.execute(`
      INSERT INTO faqs (
        question, answer, emoji, reaction_time, response_time, 
        schedule_hours, schedule_days, order_position, active
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, TRUE)
    `, [question, answer, emoji, reaction_time, response_time, schedule_hours, schedule_days, order_position]);

    res.json({ 
      success: true, 
      message: 'FAQ created successfully',
      id: result.insertId 
    });
  } catch (error) {
    console.error('Error creating FAQ:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Update FAQ (Admin)
router.put('/faqs/:id', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const { 
      question, 
      answer, 
      emoji, 
      reaction_time, 
      response_time,
      schedule_hours,
      schedule_days,
      order_position,
      active
    } = req.body;

    connection = await pool.getConnection();
    await connection.execute(`
      UPDATE faqs SET 
        question = ?, answer = ?, emoji = ?, reaction_time = ?, 
        response_time = ?, schedule_hours = ?, schedule_days = ?, 
        order_position = ?, active = ?
      WHERE id = ?
    `, [question, answer, emoji, reaction_time, response_time, schedule_hours, schedule_days, order_position, active, id]);

    res.json({ success: true, message: 'FAQ updated successfully' });
  } catch (error) {
    console.error('Error updating FAQ:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Delete FAQ (Admin)
router.delete('/faqs/:id', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { id } = req.params;

    connection = await pool.getConnection();
    await connection.execute('UPDATE faqs SET active = FALSE WHERE id = ?', [id]);

    res.json({ success: true, message: 'FAQ removed successfully' });
  } catch (error) {
    console.error('Error removing FAQ:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== MESSAGE PROCESSING WITH FAQ =====

// Process message and check FAQs
router.post('/process-message', async (req, res) => {
  let connection;
  try {
    const { message, phone_number } = req.body;
    
    if (!message || !phone_number) {
      return res.status(400).json({ error: 'Message and phone are required' });
    }

    connection = await pool.getConnection();
    
    // Search for matching FAQ
    const [faqs] = await connection.execute(`
      SELECT * FROM faqs 
      WHERE active = TRUE 
      AND LOWER(?) REGEXP LOWER(REPLACE(question, ' ', '|'))
      ORDER BY order_position ASC
      LIMIT 1
    `, [message]);

    if (faqs.length > 0) {
      const faq = faqs[0];
      
      // Check business hours
      const now = new Date();
      const currentDay = now.toLocaleDateString('en-US', { weekday: 'lowercase' });
      const currentTime = now.toTimeString().slice(0, 5);
      
      const workingDays = faq.schedule_days.split(',');
      const [startTime, endTime] = faq.schedule_hours.split('-');
      
      const isWorkingDay = workingDays.includes(currentDay);
      const isWorkingHour = currentTime >= startTime && currentTime <= endTime;
      
      if (isWorkingDay && isWorkingHour) {
        // Process placeholders in answer
        let processedAnswer = faq.answer;
        
        const [placeholders] = await connection.execute(`
          SELECT placeholder_key, placeholder_value FROM message_placeholders WHERE active = TRUE
        `);
        
        placeholders.forEach(placeholder => {
          processedAnswer = processedAnswer.replace(
            new RegExp(placeholder.placeholder_key, 'g'), 
            placeholder.placeholder_value
          );
        });
        
        res.json({
          found: true,
          faq: {
            ...faq,
            answer: processedAnswer
          },
          reaction_time: faq.reaction_time * 1000, // convert to ms
          response_time: faq.response_time * 1000  // convert to ms
        });
      } else {
        res.json({
          found: false,
          reason: 'outside_working_hours',
          message: 'Outside business hours'
        });
      }
    } else {
      res.json({
        found: false,
        reason: 'no_match',
        message: 'No FAQ found'
      });
    }
  } catch (error) {
    console.error('Error processing message:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Function to process placeholders in text
function processPlaceholders(text, placeholders, customData = {}) {
  let processedText = text;
  
  // Placeholders from database
  placeholders.forEach(placeholder => {
    processedText = processedText.replace(
      new RegExp(placeholder.placeholder_key, 'g'), 
      placeholder.placeholder_value
    );
  });
  
  // Dynamic placeholders
  const now = new Date();
  processedText = processedText.replace(/\{\{data_atual\}\}/g, now.toLocaleDateString('pt-BR'));
  processedText = processedText.replace(/\{\{hora_atual\}\}/g, now.toLocaleTimeString('pt-BR'));
  
  // Custom placeholders
  Object.keys(customData).forEach(key => {
    processedText = processedText.replace(
      new RegExp(`\\{\\{${key}\\}\\}`, 'g'), 
      customData[key]
    );
  });
  
  return processedText;
}

module.exports = router;
