const express = require('express');
const router = express.Router();
const { pool } = require('../config/database');
const multer = require('multer');
const csv = require('csv-parser');
const fs = require('fs');
const path = require('path');

// Configure multer for CSV file upload
const upload = multer({ dest: 'uploads/' });

// REMOVE: Don't use authenticateToken here, as it's already being applied in server.js
// router.use(authenticateToken);

// ===================================================================
//  CONTACT ROUTES
// ===================================================================

// List all contacts (accessible by all authenticated users)
router.get('/contacts', async (req, res) => {
    let connection;
    try {
        connection = await pool.getConnection();
        
        const query = `
            SELECT c.*, cg.group_name as group_name 
            FROM contacts c 
            LEFT JOIN contact_groups cg ON c.group_id = cg.id 
            ORDER BY c.name ASC
        `;
        
        const [contacts] = await connection.execute(query);
        res.json(contacts);
    } catch (error) {
        console.error('Error fetching contacts:', error);
        res.status(500).json({ error: 'Error fetching contacts' });
    } finally {
        if (connection) connection.release();
    }
});

// Fetch contact by ID
router.get('/contacts/:id', async (req, res) => {
    let connection;
    try {
        connection = await pool.getConnection();
        const { id } = req.params;
        
        const query = `
            SELECT c.*, cg.group_name as group_name 
            FROM contacts c 
            LEFT JOIN contact_groups cg ON c.group_id = cg.id 
            WHERE c.id = ?
        `;
        
        const [contacts] = await connection.execute(query, [id]);
        
        if (contacts.length === 0) {
            return res.status(404).json({ error: 'Contact not found' });
        }
        
        res.json(contacts[0]);
    } catch (error) {
        console.error('Error fetching contact:', error);
        res.status(500).json({ error: 'Error fetching contact' });
    } finally {
        if (connection) connection.release();
    }
});

// Create new contact (admin only)
router.post('/contacts', async (req, res) => {
    let connection;
    try {
        
        connection = await pool.getConnection();
        const { name, phone_number, email, group_id, tags, notes } = req.body;
        
        // Basic validations
        if (!name || !phone_number) {
            return res.status(400).json({ error: 'Name and phone are required' });
        }
        
        // Validate phone format (must contain only digits, without +)
        const cleanPhone = phone_number.replace(/\D/g, '');
        if (cleanPhone.length < 8 || cleanPhone.length > 15) {
            return res.status(400).json({ error: 'Phone must contain between 8 and 15 digits' });
        }
        
        // Add + to phone to save in database
        const formattedPhone = '+' + cleanPhone;
        
        // Check if phone already exists
        const [existingContact] = await connection.execute(
            'SELECT id FROM contacts WHERE phone_number = ?',
            [formattedPhone]
        );
        
        if (existingContact.length > 0) {
            return res.status(400).json({ error: 'A contact with this phone already exists' });
        }
        
        const query = `
            INSERT INTO contacts (name, phone_number, email, group_id, tags, notes, created_at)
            VALUES (?, ?, ?, ?, ?, ?, NOW())
        `;
        
        const [result] = await connection.execute(query, [
            name,
            formattedPhone,
            email || null,
            group_id || null,
            tags || null,
            notes || null
        ]);
        
        res.status(201).json({
            id: result.insertId,
            message: 'Contact created successfully'
        });
    } catch (error) {
        console.error('Error creating contact:', error);
        res.status(500).json({ error: 'Error creating contact' });
    } finally {
        if (connection) connection.release();
    }
});

// Update contact (admin only)
router.put('/contacts/:id', async (req, res) => {
    let connection;
    try {
        // Role verification for contact editing is handled by a global middleware.
        
        connection = await pool.getConnection();
        const { id } = req.params;
        const { name, phone_number, email, group_id, tags, notes } = req.body;
        
        // Basic validations
        if (!name || !phone_number) {
            return res.status(400).json({ error: 'Name and phone are required' });
        }
        
        // Validate phone format (must contain only digits, without +)
        const cleanPhone = phone_number.replace(/\D/g, '');
        if (cleanPhone.length < 8 || cleanPhone.length > 15) {
            return res.status(400).json({ error: 'Phone must contain between 8 and 15 digits' });
        }
        
        // Add + to phone to save in database
        const formattedPhone = '+' + cleanPhone;
        
        // Check if contact exists
        const [existingContact] = await connection.execute(
            'SELECT id FROM contacts WHERE id = ?',
            [id]
        );
        
        if (existingContact.length === 0) {
            return res.status(404).json({ error: 'Contact not found' });
        }
        
        // Check if phone already exists in another contact
        const [phoneCheck] = await connection.execute(
            'SELECT id FROM contacts WHERE phone_number = ? AND id != ?',
            [formattedPhone, id]
        );
        
        if (phoneCheck.length > 0) {
            return res.status(400).json({ error: 'Another contact with this phone already exists' });
        }
        
        const query = `
            UPDATE contacts 
            SET name = ?, phone_number = ?, email = ?, group_id = ?, tags = ?, notes = ?
            WHERE id = ?
        `;
        
        await connection.execute(query, [
            name,
            formattedPhone,
            email || null,
            group_id || null,
            tags || null,
            notes || null,
            id
        ]);
        
        res.json({ message: 'Contact updated successfully' });
    } catch (error) {
        console.error('Error updating contact:', error);
        res.status(500).json({ error: 'Error updating contact' });
    } finally {
        if (connection) connection.release();
    }
});

// Delete contact (admin only)
router.delete('/contacts/:id', async (req, res) => {
    let connection;
    try {
        // Role verification for contact deletion is handled by a global middleware.
        
        connection = await pool.getConnection();
        const { id } = req.params;
        
        // Check if contact exists
        const [existingContact] = await connection.execute(
            'SELECT id FROM contacts WHERE id = ?',
            [id]
        );
        
        if (existingContact.length === 0) {
            return res.status(404).json({ error: 'Contact not found' });
        }
        
        await connection.execute('DELETE FROM contacts WHERE id = ?', [id]);
        
        res.json({ message: 'Contact deleted successfully' });
    } catch (error) {
        console.error('Error deleting contact:', error);
        res.status(500).json({ error: 'Error deleting contact' });
    } finally {
        if (connection) connection.release();
    }
});

// ===================================================================
//  CONTACT GROUP ROUTES
// ===================================================================

// List all groups
router.get('/contact-groups', async (req, res) => {
    let connection;
    try {
        connection = await pool.getConnection();
        
        const query = `
            SELECT cg.id, cg.group_name as name, cg.description, cg.created_at, cg.updated_at,
                   COUNT(c.id) as contact_count
            FROM contact_groups cg
            LEFT JOIN contacts c ON cg.id = c.group_id
            GROUP BY cg.id, cg.group_name, cg.description, cg.created_at, cg.updated_at
            ORDER BY cg.group_name ASC
        `;
        
        const [groups] = await connection.execute(query);
        res.json(groups);
    } catch (error) {
        console.error('Error fetching groups:', error);
        res.status(500).json({ error: 'Error fetching groups' });
    } finally {
        if (connection) connection.release();
    }
});

// Fetch group by ID
router.get('/contact-groups/:id', async (req, res) => {
    let connection;
    try {
        connection = await pool.getConnection();
        const { id } = req.params;
        
        const query = `
            SELECT cg.id, cg.group_name as name, cg.description, cg.created_at, cg.updated_at,
                   COUNT(c.id) as contact_count
            FROM contact_groups cg
            LEFT JOIN contacts c ON cg.id = c.group_id
            WHERE cg.id = ?
            GROUP BY cg.id, cg.group_name, cg.description, cg.created_at, cg.updated_at
        `;
        
        const [groups] = await connection.execute(query, [id]);
        
        if (groups.length === 0) {
            return res.status(404).json({ error: 'Group not found' });
        }
        
        res.json(groups[0]);
    } catch (error) {
        console.error('Error fetching group:', error);
        res.status(500).json({ error: 'Error fetching group' });
    } finally {
        if (connection) connection.release();
    }
});

// Create new group (admin only)
router.post('/contact-groups', async (req, res) => {
    let connection;
    try {
        // Check if is admin
        if (req.user.role !== 'admin') {
            return res.status(403).json({ error: 'Access denied. Only administrators can create groups.' });
        }
        
        connection = await pool.getConnection();
        const { group_name, description } = req.body;
        
        // Basic validations
        if (!group_name) {
            return res.status(400).json({ error: 'Group name is required' });
        }
        
        // Check if group already exists
        const [existingGroup] = await connection.execute(
            'SELECT id FROM contact_groups WHERE group_name = ?',
            [group_name]
        );
        
        if (existingGroup.length > 0) {
            return res.status(400).json({ error: 'A group with this name already exists' });
        }
        
        const query = `
            INSERT INTO contact_groups (group_name, description, created_at, updated_at)
            VALUES (?, ?, NOW(), NOW())
        `;
        
        const [result] = await connection.execute(query, [
            group_name,
            description || null
        ]);
        
        res.status(201).json({
            id: result.insertId,
            message: 'Group created successfully'
        });
    } catch (error) {
        console.error('Error creating group:', error);
        res.status(500).json({ error: 'Error creating group' });
    } finally {
        if (connection) connection.release();
    }
});

// Update group (admin only)
router.put('/contact-groups/:id', async (req, res) => {
    let connection;
    try {
        // Check if is admin
        if (req.user.role !== 'admin') {
            return res.status(403).json({ error: 'Access denied. Only administrators can edit groups.' });
        }
        
        connection = await pool.getConnection();
        const { id } = req.params;
        const { group_name, description } = req.body;
        
        // Basic validations
        if (!group_name) {
            return res.status(400).json({ error: 'Group name is required' });
        }
        
        // Check if group exists
        const [existingGroup] = await connection.execute(
            'SELECT id FROM contact_groups WHERE id = ?',
            [id]
        );
        
        if (existingGroup.length === 0) {
            return res.status(404).json({ error: 'Group not found' });
        }
        
        // Check if name already exists in another group
        const [nameCheck] = await connection.execute(
            'SELECT id FROM contact_groups WHERE group_name = ? AND id != ?',
            [group_name, id]
        );
        
        if (nameCheck.length > 0) {
            return res.status(400).json({ error: 'Another group with this name already exists' });
        }
        
        const query = `
            UPDATE contact_groups 
            SET group_name = ?, description = ?, updated_at = NOW()
            WHERE id = ?
        `;
        
        await connection.execute(query, [
            group_name,
            description || null,
            id
        ]);
        
        res.json({ message: 'Group updated successfully' });
    } catch (error) {
        console.error('Error updating group:', error);
        res.status(500).json({ error: 'Error updating group' });
    } finally {
        if (connection) connection.release();
    }
});

// Delete group (admin only)
router.delete('/contact-groups/:id', async (req, res) => {
    let connection;
    try {
        // Check if is admin
        if (req.user.role !== 'admin') {
            return res.status(403).json({ error: 'Access denied. Only administrators can delete groups.' });
        }
        
        connection = await pool.getConnection();
        const { id } = req.params;
        
        // Check if group exists
        const [existingGroup] = await connection.execute(
            'SELECT id FROM contact_groups WHERE id = ?',
            [id]
        );
        
        if (existingGroup.length === 0) {
            return res.status(404).json({ error: 'Group not found' });
        }
        
        // Check if there are contacts in this group
        const [contactsInGroup] = await connection.execute(
            'SELECT COUNT(*) as count FROM contacts WHERE group_id = ?',
            [id]
        );
        
        if (contactsInGroup[0].count > 0) {
            return res.status(400).json({ 
                error: 'Cannot delete a group that has contacts. Move the contacts to another group first.' 
            });
        }
        
        await connection.execute('DELETE FROM contact_groups WHERE id = ?', [id]);
        
        res.json({ message: 'Group deleted successfully' });
    } catch (error) {
        console.error('Error deleting group:', error);
        res.status(500).json({ error: 'Error deleting group' });
    } finally {
        if (connection) connection.release();
    }
});

// ===================================================================
//  AUXILIARY ROUTES
// ===================================================================

// Fetch contacts by group
router.get('/contact-groups/:id/contacts', async (req, res) => {
    let connection;
    try {
        connection = await pool.getConnection();
        const { id } = req.params;
        
        const query = `
            SELECT * FROM contacts 
            WHERE group_id = ? 
            ORDER BY name ASC
        `;
        
        const [contacts] = await connection.execute(query, [id]);
        res.json(contacts);
    } catch (error) {
        console.error('Error fetching group contacts:', error);
        res.status(500).json({ error: 'Error fetching group contacts' });
    } finally {
        if (connection) connection.release();
    }
});

// Search contacts by term
router.get('/contacts/search/:term', async (req, res) => {
    let connection;
    try {
        connection = await pool.getConnection();
        const { term } = req.params;
        const searchTerm = `%${term}%`;
        
        const query = `
            SELECT c.*, cg.group_name as group_name 
            FROM contacts c 
            LEFT JOIN contact_groups cg ON c.group_id = cg.id 
            WHERE c.name LIKE ? OR c.phone_number LIKE ? OR c.email LIKE ?
            ORDER BY c.name ASC
        `;
        
        const [contacts] = await connection.execute(query, [searchTerm, searchTerm, searchTerm]);
        res.json(contacts);
    } catch (error) {
        console.error('Error searching contacts:', error);
        res.status(500).json({ error: 'Error searching contacts' });
    } finally {
        if (connection) connection.release();
    }
});

// ===================================================================
//  IMPORT AND EXPORT
// ===================================================================

// Import contacts from CSV
router.post('/contacts/import', upload.single('file'), async (req, res) => {
    let connection;
    let filePath;

    try {
        if (!req.file) {
            return res.status(400).json({ error: 'No file uploaded' });
        }


        const { group_id } = req.body;
        filePath = req.file.path;
        const contacts = [];
        const errors = [];
        let lineNumber = 0;

        // Processar o CSV
        await new Promise((resolve, reject) => {
            fs.createReadStream(filePath)
                .pipe(csv())
                .on('data', (row) => {
                    lineNumber++;
                    
                    // Support multiple header formats
                    const name = row['Nome'] || row['name'] || row['Name'] || row['Nombre'];
                    const phone = row['Telefone'] || row['phone_number'] || row['Teléfono'] || row['Phone'];
                    const email = row['Email'] || row['email'];
                    const tags = row['Tags'] || row['tags'];
                    const notes = row['Observações'] || row['notes'];
                    
                    if (!name || !phone) {
                        errors.push(`Line ${lineNumber}: Name="${name}", Phone="${phone}" - missing required data`);
                        return;
                    }

                    // Clean and format phone
                    const cleanPhone = String(phone).trim().replace(/\D/g, '');
                    if (cleanPhone.length < 8) {
                        errors.push(`Line ${lineNumber}: Phone "${phone}" is too short (minimum 8 digits)`);
                        return;
                    }

                    const formattedPhone = '+' + cleanPhone;
                    console.log(`Line ${lineNumber}: Name=${name}, Phone=${phone} -> ${formattedPhone}`);

                    contacts.push({
                        name: String(name).trim(),
                        phone_number: formattedPhone,
                        email: email ? String(email).trim() : null,
                        group_id: group_id ? parseInt(group_id) : null,
                        tags: tags ? String(tags).trim() : null,
                        notes: notes ? String(notes).trim() : null
                    });
                })
                .on('end', () => resolve())
                .on('error', (error) => reject(error));
        });

        console.log(`CSV processed: ${contacts.length} contacts found`);
        console.log('Contacts:', JSON.stringify(contacts, null, 2));

        if (contacts.length === 0) {
            return res.status(400).json({ 
                error: 'No valid contacts in file',
                errors: errors.slice(0, 10) // First 10 errors
            });
        }

        connection = await pool.getConnection();
        let imported = 0;
        let duplicates = 0;

        console.log('Starting database insertion...');
        
        for (const contact of contacts) {
            try {
                console.log(`Checking contact: ${contact.name} - ${contact.phone_number}`);
                
                const [existing] = await connection.execute(
                    'SELECT id FROM contacts WHERE phone = ?',
                    [contact.phone_number]
                );

                if (existing.length > 0) {
                    console.log(`Duplicate contact: ${contact.phone_number}`);
                    duplicates++;
                    continue;
                }

                console.log(`Inserting contact: ${contact.name}`);
                const [result] = await connection.execute(
                    'INSERT INTO contacts (name, phone, email, group_id, tags, notes, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())',
                    [contact.name, contact.phone_number, contact.email, contact.group_id, contact.tags, contact.notes]
                );
                console.log(`Contact inserted with ID: ${result.insertId}`);

                imported++;
            } catch (err) {
                console.error(`Error inserting ${contact.name}:`, err);
                errors.push(`${contact.name} (${contact.phone_number}): ${err.message}`);
            }
        }
        
        console.log(`Import completed: ${imported} imported, ${duplicates} duplicates`);

        // Delete file only after success
        if (filePath && fs.existsSync(filePath)) {
            fs.unlinkSync(filePath);
            console.log('CSV file deleted successfully');
        }

        res.json({
            message: 'Import completed',
            imported: imported,
            duplicates: duplicates,
            total: contacts.length,
            errors: errors.slice(0, 5)
        });

    } catch (error) {
        console.error('Error importing:', error);
        
        if (filePath && fs.existsSync(filePath)) {
            fs.unlinkSync(filePath);
        }
        
        res.status(500).json({ error: 'Error: ' + error.message });
    } finally {
        if (connection) connection.release();
    }
});

// Export contacts to CSV
router.get('/contacts/export', async (req, res) => {
    let connection;
    try {
        connection = await pool.getConnection();
        const { group_id } = req.query;
        
        let query = `
            SELECT c.name, c.phone_number, c.email, cg.group_name, c.tags, c.notes, c.created_at
            FROM contacts c
            LEFT JOIN contact_groups cg ON c.group_id = cg.id
        `;
        
        const params = [];
        
        if (group_id) {
            query += ' WHERE c.group_id = ?';
            params.push(group_id);
        }
        
        query += ' ORDER BY c.name ASC';
        
        const [contacts] = await connection.execute(query, params);
        
        if (contacts.length === 0) {
            return res.status(404).json({ error: 'No contacts found' });
        }
        
        const csvHeader = 'Name,Phone,Email,Group,Tags,Notes,Creation Date\n';
        const csvRows = contacts.map(contact => {
            return [
                contact.name,
                contact.phone_number,
                contact.email || '',
                contact.group_name || '',
                contact.tags || '',
                contact.notes || '',
                new Date(contact.created_at).toLocaleDateString('pt-BR')
            ].map(field => `"${field}"`).join(',');
        }).join('\n');
        
        const csv = csvHeader + csvRows;
        
        res.setHeader('Content-Type', 'text/csv; charset=utf-8');
        res.setHeader('Content-Disposition', 'attachment; filename=contacts.csv');
        res.send('\uFEFF' + csv);
        
    } catch (error) {
        console.error('Error exporting contacts:', error);
        res.status(500).json({ error: 'Error exporting contacts' });
    } finally {
        if (connection) connection.release();
    }
});

module.exports = router; 

