const express = require('express');
const crypto = require('crypto');
const { pool } = require('../config/database');
const { authenticateToken, requireAdmin } = require('../middleware/auth');

const router = express.Router();
router.use(authenticateToken);
router.use(requireAdmin);

// ===== AI CONFIGURATIONS =====

// List AI configurations
router.get('/settings', async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    
    // Fetch AI settings (without exposing sensitive keys)
    const [settings] = await connection.execute(`
      SELECT 
        id,
        provider,
        model_name,
        persona_name,
        persona_description,
        system_prompt,
        temperature,
        max_tokens,
        active,
        business_hours_start,
        business_hours_end,
        business_days,
        auto_response_enabled,
        response_delay,
        created_at,
        updated_at,
        CASE 
          WHEN api_key IS NOT NULL AND api_key != '' THEN 'configured'
          ELSE 'not_configured'
        END as api_key_status
      FROM ai_configurations 
      ORDER BY created_at DESC
    `);
    
    res.json(settings);
  } catch (error) {
    console.error('Error fetching AI configurations:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Create new AI configuration
router.post('/settings', async (req, res) => {
  let connection;
  try {
    const {
      provider,
      model_name,
      api_key,
      persona_name,
      persona_description,
      system_prompt,
      temperature = 0.7,
      max_tokens = 1000,
      business_hours_start = '08:00',
      business_hours_end = '18:00',
      business_days = 'monday,tuesday,wednesday,thursday,friday,saturday',
      auto_response_enabled = true,
      response_delay = 2
    } = req.body;
    
    if (!provider || !model_name || !api_key || !persona_name) {
      return res.status(400).json({ 
        error: 'Provider, model, API key and persona name are required' 
      });
    }

    if (!['deepseek', 'gpt'].includes(provider)) {
      return res.status(400).json({ 
        error: 'Provider must be "deepseek" or "gpt"' 
      });
    }

    // Validate times
    if (!isValidTime(business_hours_start) || !isValidTime(business_hours_end)) {
      return res.status(400).json({ 
        error: 'Times must be in HH:MM format' 
      });
    }

    // Encrypt API key
    const encryptedApiKey = encryptData(api_key);

    connection = await pool.getConnection();
    
    // Deactivate other configurations if this one is active
    if (req.body.active) {
      await connection.execute(
        'UPDATE ai_configurations SET active = FALSE'
      );
    }

    const [result] = await connection.execute(`
      INSERT INTO ai_configurations (
        provider, model_name, api_key, persona_name, persona_description,
        system_prompt, temperature, max_tokens, active,
        business_hours_start, business_hours_end, business_days,
        auto_response_enabled, response_delay
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `, [
      provider, model_name, encryptedApiKey, persona_name, persona_description,
      system_prompt, temperature, max_tokens, req.body.active || false,
      business_hours_start, business_hours_end, business_days,
      auto_response_enabled, response_delay
    ]);

    res.json({ 
      success: true, 
      message: 'AI configuration created successfully',
      id: result.insertId
    });
  } catch (error) {
    console.error('Error creating AI configuration:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Update AI configuration
router.put('/settings/:id', async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const {
      provider,
      model_name,
      api_key,
      persona_name,
      persona_description,
      system_prompt,
      temperature,
      max_tokens,
      business_hours_start,
      business_hours_end,
      business_days,
      auto_response_enabled,
      response_delay
    } = req.body;

    connection = await pool.getConnection();
    
    // Check if configuration exists
    const [existing] = await connection.execute(
      'SELECT id FROM ai_configurations WHERE id = ?',
      [id]
    );

    if (existing.length === 0) {
      return res.status(404).json({ error: 'Configuration not found' });
    }

    // Prepare fields for update
    let updateFields = [];
    let updateValues = [];

    if (provider) {
      updateFields.push('provider = ?');
      updateValues.push(provider);
    }

    if (model_name) {
      updateFields.push('model_name = ?');
      updateValues.push(model_name);
    }

    if (api_key) {
      updateFields.push('api_key = ?');
      updateValues.push(encryptData(api_key));
    }

    if (persona_name) {
      updateFields.push('persona_name = ?');
      updateValues.push(persona_name);
    }

    if (persona_description !== undefined) {
      updateFields.push('persona_description = ?');
      updateValues.push(persona_description);
    }

    if (system_prompt !== undefined) {
      updateFields.push('system_prompt = ?');
      updateValues.push(system_prompt);
    }

    if (temperature !== undefined) {
      updateFields.push('temperature = ?');
      updateValues.push(temperature);
    }

    if (max_tokens !== undefined) {
      updateFields.push('max_tokens = ?');
      updateValues.push(max_tokens);
    }

    if (business_hours_start) {
      updateFields.push('business_hours_start = ?');
      updateValues.push(business_hours_start);
    }

    if (business_hours_end) {
      updateFields.push('business_hours_end = ?');
      updateValues.push(business_hours_end);
    }

    if (business_days) {
      updateFields.push('business_days = ?');
      updateValues.push(business_days);
    }

    if (auto_response_enabled !== undefined) {
      updateFields.push('auto_response_enabled = ?');
      updateValues.push(auto_response_enabled);
    }

    if (response_delay !== undefined) {
      updateFields.push('response_delay = ?');
      updateValues.push(response_delay);
    }

    if (updateFields.length === 0) {
      return res.status(400).json({ error: 'No fields to update' });
    }

    updateFields.push('updated_at = CURRENT_TIMESTAMP');
    updateValues.push(id);

    await connection.execute(
      `UPDATE ai_configurations SET ${updateFields.join(', ')} WHERE id = ?`,
      updateValues
    );

    res.json({ 
      success: true, 
      message: 'Configuration updated successfully' 
    });
  } catch (error) {
    console.error('Error updating configuration:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Activate/Deactivate configuration
router.patch('/settings/:id/toggle', async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const { active } = req.body;

    connection = await pool.getConnection();
    
    if (active) {
      // Deactivate all other configurations
      await connection.execute(
        'UPDATE ai_configurations SET active = FALSE'
      );
    }

    // Activate/deactivate the specific configuration
    await connection.execute(
      'UPDATE ai_configurations SET active = ? WHERE id = ?',
      [active, id]
    );

    res.json({ 
      success: true, 
      message: `Configuration ${active ? 'activated' : 'deactivated'} successfully` 
    });
  } catch (error) {
    console.error('Error changing status:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Delete configuration
router.delete('/settings/:id', async (req, res) => {
  let connection;
  try {
    const { id } = req.params;

    connection = await pool.getConnection();
    
    const [result] = await connection.execute(
      'DELETE FROM ai_configurations WHERE id = ?',
      [id]
    );

    if (result.affectedRows === 0) {
      return res.status(404).json({ error: 'Configuration not found' });
    }

    res.json({ 
      success: true, 
      message: 'Configuration deleted successfully' 
    });
  } catch (error) {
    console.error('Error deleting configuration:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== TEST CONFIGURATION =====

// Test connection with AI provider
router.post('/test/:id', async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const { test_message = 'Hello, this is a connection test.' } = req.body;

    connection = await pool.getConnection();
    
    // Fetch configuration
    const [configs] = await connection.execute(
      'SELECT * FROM ai_configurations WHERE id = ?',
      [id]
    );

    if (configs.length === 0) {
      return res.status(404).json({ error: 'Configuration not found' });
    }

    const config = configs[0];
    
    // Decrypt API key
    const apiKey = decryptData(config.api_key);
    
    // Test connection based on provider
    let testResult;
    
    if (config.provider === 'deepseek') {
      testResult = await testDeepSeekConnection(apiKey, config, test_message);
    } else if (config.provider === 'gpt') {
      testResult = await testGPTConnection(apiKey, config, test_message);
    } else {
      return res.status(400).json({ error: 'Provider not supported' });
    }

    res.json(testResult);
  } catch (error) {
    console.error('Error testing configuration:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== AVAILABLE MODELS =====

// List available models by provider
router.get('/models/:provider', async (req, res) => {
  try {
    const { provider } = req.params;
    
    let models = [];
    
    if (provider === 'deepseek') {
      models = [
        {
          id: 'deepseek-chat',
          name: 'DeepSeek Chat',
          description: 'Main model for general conversations',
          max_tokens: 4096
        },
        {
          id: 'deepseek-coder',
          name: 'DeepSeek Coder',
          description: 'Specialized in programming and code',
          max_tokens: 4096
        }
      ];
    } else if (provider === 'gpt') {
      models = [
        {
          id: 'gpt-3.5-turbo',
          name: 'GPT-3.5 Turbo',
          description: 'Fast and efficient for general conversations',
          max_tokens: 4096
        },
        {
          id: 'gpt-4',
          name: 'GPT-4',
          description: 'Most advanced model with better understanding',
          max_tokens: 8192
        },
        {
          id: 'gpt-4-turbo',
          name: 'GPT-4 Turbo',
          description: 'Optimized version of GPT-4',
          max_tokens: 128000
        }
      ];
    } else {
      return res.status(400).json({ error: 'Provider not supported' });
    }
    
    res.json(models);
  } catch (error) {
    console.error('Error fetching models:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// ===== USAGE STATISTICS =====

// AI usage statistics
router.get('/stats', async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    
    // Messages processed by AI today
    const [todayAI] = await connection.execute(`
      SELECT COUNT(*) as count 
      FROM messages 
      WHERE DATE(timestamp) = CURDATE() 
      AND is_ai_response = TRUE
    `);

    // Messages processed by AI this week
    const [weekAI] = await connection.execute(`
      SELECT COUNT(*) as count 
      FROM messages 
      WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 7 DAY)
      AND is_ai_response = TRUE
    `);

    // Average AI response time
    const [avgResponse] = await connection.execute(`
      SELECT AVG(ai_response_time) as avg_time
      FROM messages 
      WHERE is_ai_response = TRUE 
      AND ai_response_time IS NOT NULL
      AND timestamp >= DATE_SUB(NOW(), INTERVAL 7 DAY)
    `);

    // Active configuration
    const [activeConfig] = await connection.execute(`
      SELECT provider, model_name, persona_name
      FROM ai_configurations 
      WHERE active = TRUE
      LIMIT 1
    `);

    res.json({
      today_ai_messages: todayAI[0].count,
      week_ai_messages: weekAI[0].count,
      avg_response_time: avgResponse[0].avg_time || 0,
      active_config: activeConfig.length > 0 ? activeConfig[0] : null
    });
  } catch (error) {
    console.error('Error fetching statistics:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== HELPER FUNCTIONS =====

// Encryption for sensitive keys
function encryptData(text) {
  const algorithm = 'aes-256-cbc';
  const secretKey = process.env.ENCRYPTION_KEY || 'default-key-change-in-production-32';
  
  // Ensure key has exactly 32 bytes for AES-256
  const key = crypto.createHash('sha256').update(String(secretKey)).digest();
  const iv = crypto.randomBytes(16);
  
  // Use createCipheriv instead of createCipher (deprecated)
  const cipher = crypto.createCipheriv(algorithm, key, iv);
  let encrypted = cipher.update(text, 'utf8');
  encrypted = Buffer.concat([encrypted, cipher.final()]);
  
  return iv.toString('hex') + ':' + encrypted.toString('hex');
}

function decryptData(encryptedText) {
  const algorithm = 'aes-256-cbc';
  const secretKey = process.env.ENCRYPTION_KEY || 'default-key-change-in-production-32';
  
  // Ensure key has exactly 32 bytes for AES-256
  const key = crypto.createHash('sha256').update(String(secretKey)).digest();
  
  const textParts = encryptedText.split(':');
  const iv = Buffer.from(textParts.shift(), 'hex');
  const encrypted = Buffer.from(textParts.join(':'), 'hex');
  
  // Use createDecipheriv instead of createDecipher (deprecated)
  const decipher = crypto.createDecipheriv(algorithm, key, iv);
  let decrypted = decipher.update(encrypted);
  decrypted = Buffer.concat([decrypted, decipher.final()]);
  
  return decrypted.toString('utf8');
}

function isValidTime(timeString) {
  return /^([01]?[0-9]|2[0-3]):[0-5][0-9]$/.test(timeString);
}

// Test DeepSeek connection
async function testDeepSeekConnection(apiKey, config, testMessage) {
  try {
    const axios = require('axios');
    
    const response = await axios.post('https://api.deepseek.com/v1/chat/completions', {
      model: config.model_name,
      messages: [
        {
          role: 'system',
          content: config.system_prompt || 'You are a helpful assistant.'
        },
        {
          role: 'user',
          content: testMessage
        }
      ],
      temperature: config.temperature,
      max_tokens: Math.min(config.max_tokens, 100) // Limit for test
    }, {
      headers: {
        'Authorization': `Bearer ${apiKey}`,
        'Content-Type': 'application/json'
      },
      timeout: 10000
    });

    return {
      success: true,
      message: 'Connection with DeepSeek established successfully',
      response: response.data.choices[0].message.content,
      usage: response.data.usage
    };
  } catch (error) {
    return {
      success: false,
      error: error.response?.data?.error?.message || error.message || 'Error connecting with DeepSeek'
    };
  }
}

// Test GPT connection
async function testGPTConnection(apiKey, config, testMessage) {
  try {
    const axios = require('axios');
    
    const response = await axios.post('https://api.openai.com/v1/chat/completions', {
      model: config.model_name,
      messages: [
        {
          role: 'system',
          content: config.system_prompt || 'You are a helpful assistant.'
        },
        {
          role: 'user',
          content: testMessage
        }
      ],
      temperature: config.temperature,
      max_tokens: Math.min(config.max_tokens, 100) // Limit for test
    }, {
      headers: {
        'Authorization': `Bearer ${apiKey}`,
        'Content-Type': 'application/json'
      },
      timeout: 10000
    });

    return {
      success: true,
      message: 'Connection with OpenAI GPT established successfully',
      response: response.data.choices[0].message.content,
      usage: response.data.usage
    };
  } catch (error) {
    return {
      success: false,
      error: error.response?.data?.error?.message || error.message || 'Error connecting with OpenAI'
    };
  }
}

module.exports = router;
