const express = require("express");
const bcrypt = require("bcryptjs");
const crypto = require("crypto");
const { pool } = require("../config/database");
const { authenticateToken, requireAdmin } = require("../middleware/auth");

const router = express.Router();

// Middleware for all admin routes
router.use(authenticateToken);
router.use(requireAdmin);

// Import controllers
const DashboardController = require('../controllers/DashboardController');

// Dashboard - Metrics
router.get("/dashboard", DashboardController.getMetrics);

// Create user - MODIFIED to support store OR department
router.post("/users", async (req, res) => {
  let connection;
  try {
    const { username, password, store, department } = req.body;

    console.log("Received data:", {
      username,
      password: "***",
      store,
      department,
    });

    if (!username || !password) {
      return res
        .status(400)
        .json({ error: "Username and password are required" });
    }

    if (!store && !department) {
      return res.status(400).json({ error: "Select a store or a department" });
    }

    if (store && department) {
      return res
        .status(400)
        .json({
          error:
            "User must be linked to a store OR a department, not both",
        });
    }

    if (password.length < 6) {
      return res
        .status(400)
        .json({ error: "Password must be at least 6 characters" });
    }

    const hashedPassword = await bcrypt.hash(password, 10);
    connection = await pool.getConnection();

    // Check if store or department exists
    if (store) {
      const [storeExists] = await connection.execute(
        "SELECT id FROM stores WHERE name = ?",
        [store]
      );

      if (storeExists.length === 0) {
        return res.status(400).json({ error: "Selected store is invalid" });
      }
    }

    if (department) {
      const [deptExists] = await connection.execute(
        "SELECT id FROM departments WHERE name = ?",
        [department]
      );

      if (deptExists.length === 0) {
        return res
          .status(400)
          .json({ error: "Selected department is invalid" });
      }
    }

    // Insert user with store OR department
    const [result] = await connection.execute(
      "INSERT INTO users (username, password, store, department, role, created_at) VALUES (?, ?, ?, ?, ?, NOW())",
      [username, hashedPassword, store || null, department || null, "user"]
    );

    console.log("User created successfully, ID:", result.insertId);

    res.json({
      message: "User created successfully",
      userId: result.insertId,
    });
  } catch (error) {
    console.error("Error creating user:", error);
    if (error.code === "ER_DUP_ENTRY") {
      return res.status(400).json({ error: "Username already exists" });
    }
    res.status(500).json({ error: "Internal server error: " + error.message });
  } finally {
    if (connection) connection.release();
  }
});

// List users - MODIFIED to include department
router.get("/users", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const [users] = await connection.execute(`
      SELECT id, username, role, created_at, store, department
      FROM users 
      WHERE role = 'user'
      ORDER BY created_at DESC
    `);

    console.log("Users found:", users.length);
    res.json(users);
  } catch (error) {
    console.error("Error listing users:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Update user
router.put("/users/:id", async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const { username, password, store, department } = req.body;

    if (!username) {
      return res.status(400).json({ error: "Username is required" });
    }

    if (!store && !department) {
      return res.status(400).json({ error: "Select a store or a department" });
    }

    if (store && department) {
      return res
        .status(400)
        .json({
          error:
            "User must be linked to a store OR a department, not both",
        });
    }

    connection = await pool.getConnection();

    const [users] = await connection.execute(
      "SELECT role FROM users WHERE id = ?",
      [id]
    );

    if (users.length === 0) {
      return res.status(404).json({ error: "User not found" });
    }

    if (users[0].role === "admin") {
      return res
        .status(400)
        .json({ error: "Cannot edit administrator" });
    }

    // Check if store or department exists
    if (store) {
      const [storeExists] = await connection.execute(
        "SELECT id FROM stores WHERE name = ?",
        [store]
      );

      if (storeExists.length === 0) {
        return res.status(400).json({ error: "Selected store is invalid" });
      }
    }

    if (department) {
      const [deptExists] = await connection.execute(
        "SELECT id FROM departments WHERE name = ?",
        [department]
      );

      if (deptExists.length === 0) {
        return res
          .status(400)
          .json({ error: "Selected department is invalid" });
      }
    }

    // Build query dynamically
    const updates = [];
    const values = [];

    updates.push("username = ?");
    values.push(username);

    updates.push("store = ?");
    values.push(store || null);

    updates.push("department = ?");
    values.push(department || null);

    // If password was provided, update it too
    if (password && password.trim() !== "") {
      const hashedPassword = await bcrypt.hash(password, 10);
      updates.push("password = ?");
      values.push(hashedPassword);
    }

    values.push(id);

    await connection.execute(
      `UPDATE users SET ${updates.join(", ")} WHERE id = ?`,
      values
    );

    res.json({ message: "User updated successfully" });
  } catch (error) {
    console.error("Error updating user:", error);
    if (error.code === "ER_DUP_ENTRY") {
      return res.status(400).json({ error: "Username already exists" });
    }
    res.status(500).json({ error: "Internal server error: " + error.message });
  } finally {
    if (connection) connection.release();
  }
});

// Delete user
router.delete("/users/:id", async (req, res) => {
  let connection;
  try {
    const { id } = req.params;

    connection = await pool.getConnection();

    const [users] = await connection.execute(
      "SELECT role FROM users WHERE id = ?",
      [id]
    );

    if (users.length === 0) {
      return res.status(404).json({ error: "User not found" });
    }

    if (users[0].role === "admin") {
      return res
        .status(400)
        .json({ error: "Cannot delete administrator" });
    }

    await connection.execute("DELETE FROM users WHERE id = ?", [id]);

    res.json({ message: "User deleted successfully" });
  } catch (error) {
    console.error("Error deleting user:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// List stores
router.get("/stores", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const [stores] = await connection.execute(
      "SELECT * FROM stores ORDER BY name"
    );

    console.log("Stores loaded from database:", stores);
    res.json(stores);
  } catch (error) {
    console.error("Error listing stores:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Create store
router.post("/stores", async (req, res) => {
  let connection;
  try {
    const { name, address, hours, promotions } = req.body;

    if (!name || !address) {
      return res
        .status(400)
        .json({ error: "Name and address are required" });
    }

    connection = await pool.getConnection();
    const [result] = await connection.execute(
      "INSERT INTO stores (name, address, hours, promotions, created_at) VALUES (?, ?, ?, ?, NOW())",
      [name, address, hours || "", promotions || ""]
    );

    res.json({
      message: "Store created successfully",
      storeId: result.insertId,
    });
  } catch (error) {
    console.error("Error creating store:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Update store
router.put("/stores/:id", async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const { name, address, hours, promotions } = req.body;

    if (!name) {
      return res.status(400).json({ error: "Name is required" });
    }

    connection = await pool.getConnection();

    const [storeExists] = await connection.execute(
      "SELECT id FROM stores WHERE id = ?",
      [id]
    );

    if (storeExists.length === 0) {
      return res.status(404).json({ error: "Store not found" });
    }

    // Build query dynamically based on provided fields
    const updates = [];
    const values = [];

    updates.push("name = ?");
    values.push(name);

    if (address !== undefined) {
      updates.push("address = ?");
      values.push(address || "");
    }

    if (hours !== undefined) {
      updates.push("hours = ?");
      values.push(hours || "");
    }

    if (promotions !== undefined) {
      updates.push("promotions = ?");
      values.push(promotions || "");
    }

    values.push(id);

    await connection.execute(
      `UPDATE stores SET ${updates.join(", ")} WHERE id = ?`,
      values
    );

    res.json({ message: "Store updated successfully" });
  } catch (error) {
    console.error("Error updating store:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Delete store
router.delete("/stores/:id", async (req, res) => {
  let connection;
  try {
    const { id } = req.params;

    connection = await pool.getConnection();

    // --- CORRECTION APPLIED HERE ---
    // The check now uses the 'store' column (name) instead of 'store_id'
    const [usersCount] = await connection.execute(
      "SELECT COUNT(*) as count FROM users u JOIN stores s ON u.store = s.name WHERE s.id = ?",
      [id]
    );

    if (usersCount[0].count > 0) {
      return res.status(400).json({
        error: "Cannot delete store with linked users",
      });
    }

    await connection.execute("DELETE FROM stores WHERE id = ?", [id]);

    res.json({ message: "Store deleted successfully" });
  } catch (error) {
    console.error("Error deleting store:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// ===== ROUTES FOR DEPARTMENTS =====

// List departments
router.get("/departments", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const [departments] = await connection.execute(
      "SELECT * FROM departments ORDER BY name"
    );

    console.log("Departments loaded from database:", departments);
    res.json(departments);
  } catch (error) {
    console.error("Error listing departments:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Create department
router.post("/departments", async (req, res) => {
  let connection;
  try {
    const { name, description } = req.body;

    if (!name) {
      return res.status(400).json({ error: "Department name is required" });
    }

    connection = await pool.getConnection();

    // Check if a department with this name already exists
    const [existing] = await connection.execute(
      "SELECT id FROM departments WHERE name = ?",
      [name]
    );

    if (existing.length > 0) {
      return res
        .status(400)
        .json({ error: "A department with this name already exists" });
    }

    const [result] = await connection.execute(
      "INSERT INTO departments (name, description, created_at) VALUES (?, ?, NOW())",
      [name, description || ""]
    );

    res.json({
      message: "Department created successfully",
      departmentId: result.insertId,
    });
  } catch (error) {
    console.error("Error creating department:", error);
    if (error.code === "ER_DUP_ENTRY") {
      return res
        .status(400)
        .json({ error: "A department with this name already exists" });
    }
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Update department
router.put("/departments/:id", async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const { name, description } = req.body;

    if (!name) {
      return res.status(400).json({ error: "Department name is required" });
    }

    connection = await pool.getConnection();

    const [deptExists] = await connection.execute(
      "SELECT id FROM departments WHERE id = ?",
      [id]
    );

    if (deptExists.length === 0) {
      return res.status(404).json({ error: "Department not found" });
    }

    // Build query dynamically based on provided fields
    const updates = [];
    const values = [];

    updates.push("name = ?");
    values.push(name);

    if (description !== undefined) {
      updates.push("description = ?");
      values.push(description || "");
    }

    values.push(id);

    await connection.execute(
      `UPDATE departments SET ${updates.join(", ")} WHERE id = ?`,
      values
    );

    res.json({ message: "Department updated successfully" });
  } catch (error) {
    console.error("Error updating department:", error);
    if (error.code === "ER_DUP_ENTRY") {
      return res
        .status(400)
        .json({ error: "A department with this name already exists" });
    }
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Delete department
router.delete("/departments/:id", async (req, res) => {
  let connection;
  try {
    const { id } = req.params;

    connection = await pool.getConnection();

    // Check if there are users linked to this department
    const [usersCount] = await connection.execute(
      "SELECT COUNT(*) as count FROM users u JOIN departments d ON u.department = d.name WHERE d.id = ?",
      [id]
    );

    if (usersCount[0].count > 0) {
      return res.status(400).json({
        error: "Cannot delete department with linked users",
      });
    }

    await connection.execute("DELETE FROM departments WHERE id = ?", [id]);

    res.json({ message: "Department deleted successfully" });
  } catch (error) {
    console.error("Error deleting department:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Bot settings
router.get("/bot-settings", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const [settings] = await connection.execute("SELECT * FROM bot_settings");

    const settingsObj = {};
    settings.forEach((setting) => {
      settingsObj[setting.setting_key] = setting.setting_value;
    });

    const defaultSettings = {
      bot_active: "true",
      auto_reply_groups: "true",
      bot_name: "Misayan",
      bot_persona: "Misayan virtual assistant",
      default_message:
        "Hello! I'm the Misayan virtual assistant. How can I help you today?",
      business_hours: "08:00-18:00",
      after_hours_message:
        "Hello! Our service hours are from 8am to 6pm. Leave your message and we will respond as soon as possible.",
      welcome_message: "Welcome to Misayan! How can I help you?",
    };

    const finalSettings = { ...defaultSettings, ...settingsObj };

    res.json(finalSettings);
  } catch (error) {
    console.error("Error fetching settings:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Update bot settings
router.put("/bot-settings", async (req, res) => {
  let connection;
  try {
    const settings = req.body;
    connection = await pool.getConnection();

    const validKeys = [
      "bot_active",
      "auto_reply_groups",
      "bot_name",
      "bot_persona",
      "default_message",
      "business_hours",
      "after_hours_message",
      "welcome_message",
      "auto_reply_message",
    ];

    for (const [key, value] of Object.entries(settings)) {
      if (!validKeys.includes(key)) {
        continue;
      }

      await connection.execute(
        `INSERT INTO bot_settings (setting_key, setting_value, updated_at) 
         VALUES (?, ?, NOW()) 
         ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value), updated_at = NOW()`,
        [key, String(value)]
      );
    }

    res.json({ message: "Settings updated successfully" });
  } catch (error) {
    console.error("Error updating settings:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// FAQs
router.get("/faqs", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const [faqs] = await connection.execute(`
      SELECT * FROM faqs 
      ORDER BY created_at DESC
    `);

    res.json(faqs);
  } catch (error) {
    console.error("Error fetching FAQs:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Get individual FAQ by ID
router.get("/faqs/:id", async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    connection = await pool.getConnection();

    const [faqs] = await connection.execute("SELECT * FROM faqs WHERE id = ?", [
      id,
    ]);

    if (faqs.length === 0) {
      return res.status(404).json({ error: "FAQ not found" });
    }

    res.json(faqs[0]);
  } catch (error) {
    console.error("Error fetching FAQ:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Create FAQ
router.post("/faqs", async (req, res) => {
  let connection;
  try {
    const { question, answer, emoji } = req.body;

    if (!question || !answer) {
      return res
        .status(400)
        .json({ error: "Question and answer are required" });
    }

    connection = await pool.getConnection();
    const [result] = await connection.execute(
      "INSERT INTO faqs (question, answer, emoji, active, created_at) VALUES (?, ?, ?, 1, NOW())",
      [question, answer, emoji || ""]
    );

    res.json({
      message: "FAQ created successfully",
      faqId: result.insertId,
    });
  } catch (error) {
    console.error("Error creating FAQ:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Update FAQ
router.put("/faqs/:id", async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const { question, answer, emoji, active } = req.body;

    if (!question || !answer) {
      return res
        .status(400)
        .json({ error: "Question and answer are required" });
    }

    connection = await pool.getConnection();

    const [faqExists] = await connection.execute(
      "SELECT id FROM faqs WHERE id = ?",
      [id]
    );

    if (faqExists.length === 0) {
      return res.status(404).json({ error: "FAQ not found" });
    }

    await connection.execute(
      "UPDATE faqs SET question = ?, answer = ?, emoji = ?, active = ? WHERE id = ?",
      [question, answer, emoji || "", active !== false ? 1 : 0, id]
    );

    res.json({ message: "FAQ updated successfully" });
  } catch (error) {
    console.error("Error updating FAQ:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Delete FAQ
router.delete("/faqs/:id", async (req, res) => {
  let connection;
  try {
    const { id } = req.params;

    connection = await pool.getConnection();

    const [result] = await connection.execute("DELETE FROM faqs WHERE id = ?", [
      id,
    ]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ error: "FAQ not found" });
    }

    res.json({ message: "FAQ deleted successfully" });
  } catch (error) {
    console.error("Error deleting FAQ:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// WhatsApp connection status
router.get("/whatsapp-status", (req, res) => {
  try {
    const whatsappService = req.app.get("whatsappService");
    if (!whatsappService) {
      return res.status(500).json({ error: "WhatsApp service not available" });
    }
    res.json(whatsappService.getConnectionStatus());
  } catch (error) {
    console.error("Error fetching WhatsApp status:", error);
    res.status(500).json({ error: "Internal server error" });
  }
});

// Disconnect WhatsApp
router.post("/whatsapp-disconnect", async (req, res) => {
  try {
    const whatsappService = req.app.get("whatsappService");
    if (!whatsappService) {
      return res.status(500).json({ error: "WhatsApp service not available" });
    }
    await whatsappService.disconnect();
    res.json({ message: "WhatsApp disconnected successfully" });
  } catch (error) {
    console.error("Error disconnecting WhatsApp:", error);
    res.status(500).json({ error: "Error disconnecting WhatsApp" });
  }
});

// Reconnect WhatsApp
router.post("/whatsapp-reconnect", async (req, res) => {
  try {
    const whatsappService = req.app.get("whatsappService");
    if (!whatsappService) {
      return res.status(500).json({ error: "WhatsApp service not available" });
    }
    await whatsappService.forceReconnect();
    res.json({ message: "WhatsApp reconnection initiated" });
  } catch (error) {
    console.error("Error reconnecting WhatsApp:", error);
    res.status(500).json({ error: "Error reconnecting WhatsApp" });
  }
});

// ===================================================================
//  ROUTES FOR CONTACTS
// ===================================================================

// List all contacts
router.get("/contacts", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const query = `
            SELECT c.*, cg.group_name 
            FROM contacts c 
            LEFT JOIN contact_groups cg ON c.group_id = cg.id 
            ORDER BY c.name ASC
        `;

    const [contacts] = await connection.execute(query);
    res.json(contacts);
  } catch (error) {
    console.error("Error fetching contacts:", error);
    res.status(500).json({ error: "Error fetching contacts" });
  } finally {
    if (connection) connection.release();
  }
});

// Get contact by ID
router.get("/contacts/:id", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { id } = req.params;
    const query = `
            SELECT c.*, cg.group_name 
            FROM contacts c 
            LEFT JOIN contact_groups cg ON c.group_id = cg.id 
            WHERE c.id = ?
        `;

    const [contacts] = await connection.execute(query, [id]);

    if (contacts.length === 0) {
      return res.status(404).json({ error: "Contact not found" });
    }

    res.json(contacts[0]);
  } catch (error) {
    console.error("Error fetching contact:", error);
    res.status(500).json({ error: "Error fetching contact" });
  } finally {
    if (connection) connection.release();
  }
});

// Create new contact
router.post("/contacts", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { name, phone, email, group_id, tags } = req.body;

    // Basic validations
    if (!name || !phone) {
      return res
        .status(400)
        .json({ error: "Name and phone are required" });
    }

    // Check if phone already exists
    const [existingContact] = await connection.execute(
      "SELECT id FROM contacts WHERE phone = ?",
      [phone]
    );

    if (existingContact.length > 0) {
      return res
        .status(400)
        .json({ error: "A contact with this phone already exists" });
    }

    const query = `
            INSERT INTO contacts (name, phone, email, group_id, tags, created_at, updated_at)
            VALUES (?, ?, ?, ?, ?, NOW(), NOW())
        `;

    const [result] = await connection.execute(query, [
      name,
      phone,
      email || null,
      group_id || null,
      tags || null,
    ]);

    res.status(201).json({
      id: result.insertId,
      message: "Contact created successfully",
    });
  } catch (error) {
    console.error("Error creating contact:", error);
    res.status(500).json({ error: "Error creating contact" });
  } finally {
    if (connection) connection.release();
  }
});

// Update contact
router.put("/contacts/:id", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { id } = req.params;
    const { name, phone, email, group_id, tags } = req.body;

    // Basic validations
    if (!name || !phone) {
      return res
        .status(400)
        .json({ error: "Name and phone are required" });
    }

    // Check if contact exists
    const [existingContact] = await connection.execute(
      "SELECT id FROM contacts WHERE id = ?",
      [id]
    );

    if (existingContact.length === 0) {
      return res.status(404).json({ error: "Contact not found" });
    }

    // Check if phone already exists in another contact
    const [phoneCheck] = await connection.execute(
      "SELECT id FROM contacts WHERE phone = ? AND id != ?",
      [phone, id]
    );

    if (phoneCheck.length > 0) {
      return res
        .status(400)
        .json({ error: "Another contact with this phone already exists" });
    }

    const query = `
            UPDATE contacts 
            SET name = ?, phone = ?, email = ?, group_id = ?, tags = ?, updated_at = NOW()
            WHERE id = ?
        `;

    await connection.execute(query, [
      name,
      phone,
      email || null,
      group_id || null,
      tags || null,
      id,
    ]);

    res.json({ message: "Contact updated successfully" });
  } catch (error) {
    console.error("Error updating contact:", error);
    res.status(500).json({ error: "Error updating contact" });
  } finally {
    if (connection) connection.release();
  }
});

// Delete contact
router.delete("/contacts/:id", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { id } = req.params;

    // Check if contact exists
    const [existingContact] = await connection.execute(
      "SELECT id FROM contacts WHERE id = ?",
      [id]
    );

    if (existingContact.length === 0) {
      return res.status(404).json({ error: "Contact not found" });
    }

    await connection.execute("DELETE FROM contacts WHERE id = ?", [id]);

    res.json({ message: "Contact deleted successfully" });
  } catch (error) {
    console.error("Error deleting contact:", error);
    res.status(500).json({ error: "Error deleting contact" });
  } finally {
    if (connection) connection.release();
  }
});

// ===================================================================
//  ROUTES FOR CONTACT GROUPS
// ===================================================================

// List all groups
router.get("/contact-groups", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const query = `
            SELECT cg.id, cg.group_name as name, cg.description, cg.created_at, cg.updated_at, cg.created_by,
                   COUNT(c.id) as contact_count
            FROM contact_groups cg
            LEFT JOIN contacts c ON cg.id = c.group_id
            GROUP BY cg.id, cg.group_name, cg.description, cg.created_at, cg.updated_at, cg.created_by
            ORDER BY cg.group_name ASC
        `;

    const [groups] = await connection.execute(query);
    res.json(groups);
  } catch (error) {
    console.error("Error fetching groups:", error);
    res.status(500).json({ error: "Error fetching groups" });
  } finally {
    if (connection) connection.release();
  }
});

// Get group by ID
router.get("/contact-groups/:id", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { id } = req.params;
    const query = `
            SELECT cg.*, 
                   COUNT(c.id) as contact_count
            FROM contact_groups cg
            LEFT JOIN contacts c ON cg.id = c.group_id
            WHERE cg.id = ?
            GROUP BY cg.id
        `;

    const [groups] = await connection.execute(query, [id]);

    if (groups.length === 0) {
      return res.status(404).json({ error: "Group not found" });
    }

    res.json(groups[0]);
  } catch (error) {
    console.error("Error fetching group:", error);
    res.status(500).json({ error: "Error fetching group" });
  } finally {
    if (connection) connection.release();
  }
});

// Create new group
router.post("/contact-groups", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { name, description } = req.body;

    // Basic validations
    if (!name) {
      return res.status(400).json({ error: "Group name is required" });
    }

    // Check if name already exists
    const [existingGroup] = await connection.execute(
      "SELECT id FROM contact_groups WHERE group_name = ?",
      [name]
    );

    if (existingGroup.length > 0) {
      return res
        .status(400)
        .json({ error: "A group with this name already exists" });
    }

    const query = `
            INSERT INTO contact_groups (group_name, description, created_at, updated_at)
            VALUES (?, ?, NOW(), NOW())
        `;

    const [result] = await connection.execute(query, [
      name,
      description || null,
    ]);

    res.status(201).json({
      id: result.insertId,
      message: "Group created successfully",
    });
  } catch (error) {
    console.error("Error creating group:", error);
    res.status(500).json({ error: "Error creating group" });
  } finally {
    if (connection) connection.release();
  }
});

// Update group
router.put("/contact-groups/:id", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { id } = req.params;
    const { name, description } = req.body;

    // Basic validations
    if (!name) {
      return res.status(400).json({ error: "Group name is required" });
    }

    // Check if group exists
    const [existingGroup] = await connection.execute(
      "SELECT id FROM contact_groups WHERE id = ?",
      [id]
    );

    if (existingGroup.length === 0) {
      return res.status(404).json({ error: "Group not found" });
    }

    // Check if name already exists in another group
    const [nameCheck] = await connection.execute(
      "SELECT id FROM contact_groups WHERE group_name = ? AND id != ?",
      [name, id]
    );

    if (nameCheck.length > 0) {
      return res
        .status(400)
        .json({ error: "Another group with this name already exists" });
    }

    const query = `
            UPDATE contact_groups 
            SET group_name = ?, description = ?, updated_at = NOW()
            WHERE id = ?
        `;

    await connection.execute(query, [name, description || null, id]);

    res.json({ message: "Group updated successfully" });
  } catch (error) {
    console.error("Error updating group:", error);
    res.status(500).json({ error: "Error updating group" });
  } finally {
    if (connection) connection.release();
  }
});

// Delete group
router.delete("/contact-groups/:id", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { id } = req.params;

    // Check if group exists
    const [existingGroup] = await connection.execute(
      "SELECT id FROM contact_groups WHERE id = ?",
      [id]
    );

    if (existingGroup.length === 0) {
      return res.status(404).json({ error: "Group not found" });
    }

    // Check if there are contacts in this group
    const [contactsInGroup] = await connection.execute(
      "SELECT COUNT(*) as count FROM contacts WHERE group_id = ?",
      [id]
    );

    if (contactsInGroup[0].count > 0) {
      return res.status(400).json({
        error:
          "Cannot delete a group that has contacts. Move the contacts to another group first.",
      });
    }

    await connection.execute("DELETE FROM contact_groups WHERE id = ?", [id]);

    res.json({ message: "Group deleted successfully" });
  } catch (error) {
    console.error("Error deleting group:", error);
    res.status(500).json({ error: "Error deleting group" });
  } finally {
    if (connection) connection.release();
  }
});

// Endpoint to toggle bot on/off
router.post("/toggle-bot", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();

    // Check current bot status in bot_settings table
    const [currentStatus] = await connection.execute(
      'SELECT setting_value FROM bot_settings WHERE setting_key = "bot_active"'
    );

    let newStatus;
    if (currentStatus.length === 0) {
      // If it doesn't exist, create with value true
      await connection.execute(
        'INSERT INTO bot_settings (setting_key, setting_value, updated_at) VALUES ("bot_active", "true", NOW())'
      );
      newStatus = true;
    } else {
      // Toggle current status
      newStatus = currentStatus[0].setting_value === "true" ? false : true;
      await connection.execute(
        'UPDATE bot_settings SET setting_value = ?, updated_at = NOW() WHERE setting_key = "bot_active"',
        [newStatus.toString()]
      );
    }

    res.json({
      message: `Bot ${newStatus ? "activated" : "deactivated"} successfully`,
      status: newStatus,
    });
  } catch (error) {
    console.error("Error changing bot status:", error);
    res.status(500).json({ error: "Error changing bot status" });
  } finally {
    if (connection) connection.release();
  }
});

// Endpoint to toggle groups on/off
router.post("/toggle-groups", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();

    // Check current status of groups in bot_settings table
    const [currentStatus] = await connection.execute(
      'SELECT setting_value FROM bot_settings WHERE setting_key = "auto_reply_groups"'
    );

    let newStatus;
    if (currentStatus.length === 0) {
      // If it doesn't exist, create with value true
      await connection.execute(
        'INSERT INTO bot_settings (setting_key, setting_value, updated_at) VALUES ("auto_reply_groups", "true", NOW())'
      );
      newStatus = true;
    } else {
      // Toggle current status
      newStatus = currentStatus[0].setting_value === "true" ? false : true;
      await connection.execute(
        'UPDATE bot_settings SET setting_value = ?, updated_at = NOW() WHERE setting_key = "auto_reply_groups"',
        [newStatus.toString()]
      );
    }

    res.json({
      message: `Groups ${newStatus ? "activated" : "deactivated"} successfully`,
      status: newStatus,
    });
  } catch (error) {
    console.error("Error changing groups status:", error);
    res.status(500).json({ error: "Error changing groups status" });
  } finally {
    if (connection) connection.release();
  }
});

// ===================================================================
//  ROUTES FOR CAMPAIGNS - ADD AFTER CONTACT ROUTES
// ===================================================================

// List all campaigns
router.get("/campaigns", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();

    // Check if mass_messages table exists
    const [tables] = await connection.execute(
      "SHOW TABLES LIKE 'mass_messages'"
    );

    if (tables.length === 0) {
      // Table doesn't exist, return empty array
      return res.json([]);
    }

    const query = `
            SELECT m.*, 
                   cg.group_name,
                   u.username as created_by_name,
                   COUNT(l.id) as total_logs,
                   SUM(CASE WHEN l.status = 'sent' THEN 1 ELSE 0 END) as sent_count,
                   SUM(CASE WHEN l.status = 'failed' THEN 1 ELSE 0 END) as failed_count
            FROM mass_messages m
            LEFT JOIN contact_groups cg ON m.group_id = cg.id
            LEFT JOIN users u ON m.created_by = u.id
            LEFT JOIN mass_message_logs l ON m.id = l.mass_message_id
            GROUP BY m.id
            ORDER BY m.created_at DESC
        `;

    const [campaigns] = await connection.execute(query);
    res.json(campaigns);
  } catch (error) {
    console.error("Error fetching campaigns:", error);
    res.status(500).json({ error: "Error fetching campaigns" });
  } finally {
    if (connection) connection.release();
  }
});

// Get campaign by ID
router.get("/campaigns/:id", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { id } = req.params;

    // Check if mass_messages table exists
    const [tables] = await connection.execute(
      "SHOW TABLES LIKE 'mass_messages'"
    );

    if (tables.length === 0) {
      return res.status(404).json({ error: "Campaign not found" });
    }

    const query = `
            SELECT m.*, 
                   cg.group_name,
                   u.username as created_by_name,
                   COUNT(l.id) as total_logs,
                   SUM(CASE WHEN l.status = 'sent' THEN 1 ELSE 0 END) as sent_count,
                   SUM(CASE WHEN l.status = 'failed' THEN 1 ELSE 0 END) as failed_count
            FROM mass_messages m
            LEFT JOIN contact_groups cg ON m.group_id = cg.id
            LEFT JOIN users u ON m.created_by = u.id
            LEFT JOIN mass_message_logs l ON m.id = l.mass_message_id
            WHERE m.id = ?
            GROUP BY m.id
        `;

    const [campaigns] = await connection.execute(query, [id]);

    if (campaigns.length === 0) {
      return res.status(404).json({ error: "Campaign not found" });
    }

    res.json(campaigns[0]);
  } catch (error) {
    console.error("Error fetching campaign:", error);
    res.status(500).json({ error: "Error fetching campaign" });
  } finally {
    if (connection) connection.release();
  }
});

// Create new campaign
router.post("/campaigns", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { name, message, group_id, schedule } = req.body;

    // Basic validations
    if (!name || !message || !group_id) {
      return res
        .status(400)
        .json({ error: "Name, message and group are required" });
    }

    // Check if group exists
    const [groupExists] = await connection.execute(
      "SELECT id FROM contact_groups WHERE id = ?",
      [group_id]
    );

    if (groupExists.length === 0) {
      return res.status(400).json({ error: "Selected group does not exist" });
    }

    // Check if there are contacts in the group
    const [contactsInGroup] = await connection.execute(
      "SELECT COUNT(*) as count FROM contacts WHERE group_id = ?",
      [group_id]
    );

    if (contactsInGroup[0].count === 0) {
      return res
        .status(400)
        .json({ error: "The selected group has no contacts" });
    }

    // Check if mass_messages table exists, if not, use campaigns
    const [tables] = await connection.execute(
      "SHOW TABLES LIKE 'mass_messages'"
    );

    let query, tableName;
    if (tables.length > 0) {
      tableName = "mass_messages";
      query = `
                INSERT INTO mass_messages (message_text, group_id, scheduled_at, total_contacts, status, created_by, created_at)
                VALUES (?, ?, ?, ?, 'pending', ?, NOW())
            `;
    } else {
      tableName = "campaigns";
      query = `
                INSERT INTO campaigns (name, message, group_id, schedule, status, created_at, updated_at)
                VALUES (?, ?, ?, ?, 'draft', NOW(), NOW())
            `;
    }

    const [result] = await connection.execute(
      query,
      tableName === "mass_messages"
        ? [
            message,
            group_id,
            schedule || null,
            contactsInGroup[0].count,
            req.user.userId,
          ]
        : [name, message, group_id, schedule || null]
    );

    // If not scheduled and is campaigns, mark for immediate sending
    if (!schedule && tableName === "campaigns") {
      await connection.execute(
        'UPDATE campaigns SET status = "pending" WHERE id = ?',
        [result.insertId]
      );
    }

    res.status(201).json({
      id: result.insertId,
      message: "Campaign created successfully",
      will_send_to: contactsInGroup[0].count,
    });
  } catch (error) {
    console.error("Error creating campaign:", error);
    res.status(500).json({ error: "Error creating campaign" });
  } finally {
    if (connection) connection.release();
  }
});

// Update campaign
router.put("/campaigns/:id", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { id } = req.params;
    const { name, message, group_id, schedule, status } = req.body;

    // Check if campaign exists
    const [existingCampaign] = await connection.execute(
      "SELECT id, status FROM campaigns WHERE id = ?",
      [id]
    );

    if (existingCampaign.length === 0) {
      return res.status(404).json({ error: "Campaign not found" });
    }

    // Do not allow editing of already sent campaigns
    if (
      existingCampaign[0].status === "sent" ||
      existingCampaign[0].status === "sending"
    ) {
      return res
        .status(400)
        .json({
          error: "Cannot edit campaigns that have already been sent or are being sent",
        });
    }

    // Basic validations
    if (!name || !message) {
      return res
        .status(400)
        .json({ error: "Name and message are required" });
    }

    // If group was changed, check if it exists
    if (group_id) {
      const [groupExists] = await connection.execute(
        "SELECT id FROM contact_groups WHERE id = ?",
        [group_id]
      );

      if (groupExists.length === 0) {
        return res.status(400).json({ error: "Selected group does not exist" });
      }
    }

    const query = `
            UPDATE campaigns 
            SET name = ?, message = ?, group_id = COALESCE(?, group_id), 
                schedule = ?, status = COALESCE(?, status), updated_at = NOW()
            WHERE id = ?
        `;

    await connection.execute(query, [
      name,
      message,
      group_id,
      schedule,
      status,
      id,
    ]);

    res.json({ message: "Campaign updated successfully" });
  } catch (error) {
    console.error("Error updating campaign:", error);
    res.status(500).json({ error: "Error updating campaign" });
  } finally {
    if (connection) connection.release();
  }
});

// Delete campaign
router.delete("/campaigns/:id", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { id } = req.params;

    // Check if campaign exists
    const [existingCampaign] = await connection.execute(
      "SELECT id, status FROM campaigns WHERE id = ?",
      [id]
    );

    if (existingCampaign.length === 0) {
      return res.status(404).json({ error: "Campaign not found" });
    }

    // Do not allow deletion of campaigns being sent
    if (existingCampaign[0].status === "sending") {
      return res
        .status(400)
        .json({ error: "Cannot delete campaigns that are being sent" });
    }

    // Delete related records first
    await connection.execute(
      "DELETE FROM campaign_sends WHERE campaign_id = ?",
      [id]
    );

    // Delete the campaign
    await connection.execute("DELETE FROM campaigns WHERE id = ?", [id]);

    res.json({ message: "Campaign deleted successfully" });
  } catch (error) {
    console.error("Error deleting campaign:", error);
    res.status(500).json({ error: "Error deleting campaign" });
  } finally {
    if (connection) connection.release();
  }
});

// Start campaign sending manually
router.post("/campaigns/:id/send", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { id } = req.params;

    // Check if campaign exists and can be sent
    const [campaign] = await connection.execute(
      'SELECT * FROM campaigns WHERE id = ? AND status IN ("draft", "pending")',
      [id]
    );

    if (campaign.length === 0) {
      return res
        .status(404)
        .json({ error: "Campaign not found or already sent" });
    }

    // Get contacts from group
    const [contacts] = await connection.execute(
      "SELECT * FROM contacts WHERE group_id = ? AND phone IS NOT NULL",
      [campaign[0].group_id]
    );

    if (contacts.length === 0) {
      return res
        .status(400)
        .json({ error: "No contacts found in group" });
    }

    // Mark campaign as sending
    await connection.execute(
      'UPDATE campaigns SET status = "sending", updated_at = NOW() WHERE id = ?',
      [id]
    );

    // Create send records for each contact
    for (const contact of contacts) {
      await connection.execute(
        `
                INSERT INTO campaign_sends (campaign_id, contact_id, phone_number, status, created_at)
                VALUES (?, ?, ?, 'pending', NOW())
            `,
        [id, contact.id, contact.phone]
      );
    }

    // Here you can integrate with WhatsApp service to send messages
    // For now, we'll simulate sending
    setTimeout(async () => {
      try {
        const conn = await pool.getConnection();

        // Simulate successful sending
        await conn.execute(
          'UPDATE campaign_sends SET status = "sent", sent_at = NOW() WHERE campaign_id = ?',
          [id]
        );

        // Mark campaign as sent
        await conn.execute(
          'UPDATE campaigns SET status = "sent", sent_count = ?, updated_at = NOW() WHERE id = ?',
          [contacts.length, id]
        );

        conn.release();
      } catch (error) {
        console.error("Error in simulated sending:", error);
      }
    }, 2000);

    res.json({
      message: "Campaign sending started successfully",
      contacts_count: contacts.length,
    });
  } catch (error) {
    console.error("Error sending campaign:", error);
    res.status(500).json({ error: "Error sending campaign" });
  } finally {
    if (connection) connection.release();
  }
});

// Cancel campaign
router.post("/campaigns/:id/cancel", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { id } = req.params;

    // Check if campaign exists
    const [campaign] = await connection.execute(
      "SELECT * FROM campaigns WHERE id = ?",
      [id]
    );

    if (campaign.length === 0) {
      return res.status(404).json({ error: "Campaign not found" });
    }

    // Can only cancel campaigns that haven't been completely sent
    if (campaign[0].status === "sent") {
      return res
        .status(400)
        .json({ error: "Campaign has already been completely sent" });
    }

    // Cancel pending sends
    await connection.execute(
      'UPDATE campaign_sends SET status = "cancelled" WHERE campaign_id = ? AND status = "pending"',
      [id]
    );

    // Mark campaign as cancelled
    await connection.execute(
      'UPDATE campaigns SET status = "cancelled", updated_at = NOW() WHERE id = ?',
      [id]
    );

    res.json({ message: "Campaign cancelled successfully" });
  } catch (error) {
    console.error("Error cancelling campaign:", error);
    res.status(500).json({ error: "Error cancelling campaign" });
  } finally {
    if (connection) connection.release();
  }
});

// Statistics for a specific campaign
router.get("/campaigns/:id/stats", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { id } = req.params;

    const [stats] = await connection.execute(
      `
            SELECT 
                c.name,
                c.status,
                c.created_at,
                COUNT(cs.id) as total_sends,
                COUNT(CASE WHEN cs.status = 'sent' THEN 1 END) as sent,
                COUNT(CASE WHEN cs.status = 'delivered' THEN 1 END) as delivered,
                COUNT(CASE WHEN cs.status = 'failed' THEN 1 END) as failed,
                COUNT(CASE WHEN cs.status = 'pending' THEN 1 END) as pending,
                COUNT(CASE WHEN cs.status = 'cancelled' THEN 1 END) as cancelled
            FROM campaigns c
            LEFT JOIN campaign_sends cs ON c.id = cs.campaign_id
            WHERE c.id = ?
            GROUP BY c.id
        `,
      [id]
    );

    if (stats.length === 0) {
      return res.status(404).json({ error: "Campaign not found" });
    }

    res.json(stats[0]);
  } catch (error) {
    console.error("Error fetching statistics:", error);
    res.status(500).json({ error: "Error fetching statistics" });
  } finally {
    if (connection) connection.release();
  }
});

// ===== PAYMENT SETTINGS =====

// Encryption helper functions
function encryptData(text) {
  const algorithm = "aes-256-cbc";
  const keyString =
    process.env.ENCRYPTION_KEY || "default-key-change-in-production-32";
  const key = crypto.createHash("sha256").update(keyString).digest();
  const iv = crypto.randomBytes(16);

  const cipher = crypto.createCipheriv(algorithm, key, iv);
  let encrypted = cipher.update(text, "utf8", "hex");
  encrypted += cipher.final("hex");

  return iv.toString("hex") + ":" + encrypted;
}

function decryptData(encryptedText) {
  const algorithm = "aes-256-cbc";
  const keyString =
    process.env.ENCRYPTION_KEY || "default-key-change-in-production-32";
  const key = crypto.createHash("sha256").update(keyString).digest();

  const textParts = encryptedText.split(":");
  const iv = Buffer.from(textParts.shift(), "hex");
  const encrypted = textParts.join(":");

  const decipher = crypto.createDecipheriv(algorithm, key, iv);
  let decrypted = decipher.update(encrypted, "hex", "utf8");
  decrypted += decipher.final("utf8");

  return decrypted;
}

// Configure payment gateway
router.post("/payment-config/:gateway", async (req, res) => {
  let connection;
  try {
    const { gateway } = req.params;
    const { sandbox, active } = req.body;

    if (!["pagseguro", "paypal"].includes(gateway)) {
      return res
        .status(400)
        .json({ error: "Payment gateway not supported" });
    }

    let api_key, api_secret;

    if (gateway === "pagseguro") {
      const { email, token } = req.body;
      if (!email || !token) {
        return res
          .status(400)
          .json({ error: "Email and token are required for PagSeguro" });
      }
      api_key = encryptData(email);
      api_secret = encryptData(token);
    } else if (gateway === "paypal") {
      const { clientId, clientSecret } = req.body;
      if (!clientId || !clientSecret) {
        return res
          .status(400)
          .json({
            error: "Client ID and Client Secret are required for PayPal",
          });
      }
      api_key = encryptData(clientId);
      api_secret = encryptData(clientSecret);
    }

    connection = await pool.getConnection();

    // Check if configuration already exists
    const [existing] = await connection.execute(
      "SELECT id FROM payment_methods WHERE method_name = ?",
      [gateway]
    );

    if (existing.length > 0) {
      // Update
      await connection.execute(
        `
        UPDATE payment_methods 
        SET api_key = ?, api_secret = ?, sandbox_mode = ?, active = ?, updated_at = CURRENT_TIMESTAMP
        WHERE method_name = ?
      `,
        [api_key, api_secret, sandbox || false, active || false, gateway]
      );
    } else {
      // Create new
      await connection.execute(
        `
        INSERT INTO payment_methods (method_name, api_key, api_secret, sandbox_mode, active) 
        VALUES (?, ?, ?, ?, ?)
      `,
        [gateway, api_key, api_secret, sandbox || false, active || false]
      );
    }

    res.json({
      success: true,
      message: `${gateway.toUpperCase()} configuration saved successfully`,
    });
  } catch (error) {
    console.error("Error configuring payment gateway:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Search gateway configuration
router.get("/payment-config/:gateway", async (req, res) => {
  let connection;
  try {
    const { gateway } = req.params;

    if (!["pagseguro", "paypal"].includes(gateway)) {
      return res
        .status(400)
        .json({ error: "Payment gateway not supported" });
    }

    connection = await pool.getConnection();

    const [methods] = await connection.execute(
      "SELECT * FROM payment_methods WHERE method_name = ?",
      [gateway]
    );

    if (methods.length === 0) {
      return res.json({ configured: false });
    }

    const method = methods[0];

    // Decrypt only for verification (don't send full keys)
    const result = {
      configured: true,
      sandbox: method.sandbox_mode,
      active: method.active,
    };

    // Send only indication that it's configured, not the credentials
    if (gateway === "pagseguro") {
      result.hasEmail = !!method.api_key;
      result.hasToken = !!method.api_secret;
    } else if (gateway === "paypal") {
      result.hasClientId = !!method.api_key;
      result.hasClientSecret = !!method.api_secret;
    }

    res.json(result);
  } catch (error) {
    console.error("Error fetching configuration:", error);
    res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
});

// Endpoint to fetch profile settings
router.get("/profile-settings", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();

    const [rows] = await connection.execute(
      "SELECT setting_value FROM bot_settings WHERE setting_key = ?",
      ["profile_settings"]
    );

    if (rows.length > 0) {
      const settings = JSON.parse(rows[0].setting_value);
      res.json(settings);
    } else {
      res.json({});
    }
  } catch (error) {
    console.error("Error fetching profile settings:", error);
    res.status(500).json({ error: "Error fetching settings" });
  } finally {
    if (connection) connection.release();
  }
});

// Endpoint to save profile settings
router.post("/profile-settings", async (req, res) => {
  let connection;
  try {
    const {
      customColors,
      systemName,
      loginLogo,
      systemLogo,
      favicon,
      loginBodyColor,
      loginCardColor,
    } = req.body;

    connection = await pool.getConnection();

    // Fetch current settings
    const [rows] = await connection.execute(
      "SELECT setting_value FROM bot_settings WHERE setting_key = ?",
      ["profile_settings"]
    );

    let settings = {};
    if (rows.length > 0) {
      settings = JSON.parse(rows[0].setting_value);
    }

    // Update only the provided fields
    if (customColors !== undefined) {
      settings.customColors = customColors;
    }
    if (systemName !== undefined) {
      settings.systemName = systemName;
    }
    if (loginLogo !== undefined) {
      settings.loginLogo = loginLogo;
    }
    if (systemLogo !== undefined) {
      settings.systemLogo = systemLogo;
    }
    if (favicon !== undefined) {
      settings.favicon = favicon;
    }
    if (loginBodyColor !== undefined) {
      settings.loginBodyColor = loginBodyColor;
    }
    if (loginCardColor !== undefined) {
      settings.loginCardColor = loginCardColor;
    }

    // Save to database
    await connection.execute(
      `INSERT INTO bot_settings (setting_key, setting_value) 
             VALUES (?, ?) 
             ON DUPLICATE KEY UPDATE setting_value = ?`,
      ["profile_settings", JSON.stringify(settings), JSON.stringify(settings)]
    );

    res.json({ success: true, message: "Settings saved successfully" });
  } catch (error) {
    console.error("Error saving profile settings:", error);
    res.status(500).json({ error: "Error saving settings" });
  } finally {
    if (connection) connection.release();
  }
});

// Endpoint to fetch language settings
router.get("/language-settings", async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();

    const [rows] = await connection.execute(
      "SELECT setting_value FROM bot_settings WHERE setting_key = ?",
      ["language"]
    );

    if (rows.length > 0) {
      res.json({ language: rows[0].setting_value });
    } else {
      res.json({ language: "pt" });
    }
  } catch (error) {
    console.error("Error fetching language:", error);
    res.status(500).json({ error: "Error fetching language" });
  } finally {
    if (connection) connection.release();
  }
});

// Endpoint to save language settings
router.post("/language-settings", async (req, res) => {
  let connection;
  try {
    const { language } = req.body;

    connection = await pool.getConnection();

    await connection.execute(
      `INSERT INTO bot_settings (setting_key, setting_value) 
             VALUES (?, ?) 
             ON DUPLICATE KEY UPDATE setting_value = ?`,
      ["language", language, language]
    );

    res.json({ success: true, message: "Language saved successfully" });
  } catch (error) {
    console.error("Error saving language:", error);
    res.status(500).json({ error: "Error saving language" });
  } finally {
    if (connection) connection.release();
  }
});

// Endpoint to get user language preference
router.get("/users/language-preference", async (req, res) => {
  let connection;
  try {
    const userId = req.user?.id;

    if (!userId) {
      return res.json({ language: "pt" });
    }

    connection = await pool.getConnection();
    const [rows] = await connection.execute(
      "SELECT language_preference FROM users WHERE id = ?",
      [userId]
    );

    if (rows.length === 0) {
      return res.json({ language: "pt" });
    }

    res.json({ language: rows[0].language_preference || "pt" });
  } catch (error) {
    console.error("Error fetching language preference:", error);
    res.json({ language: "pt" });
  } finally {
    if (connection) connection.release();
  }
});

// Endpoint to save user language preference
router.post("/users/language-preference", async (req, res) => {
  let connection;
  try {
    const userId = req.user?.id;
    const { language } = req.body;

    if (!language) {
      return res.status(400).json({ error: "Language is required" });
    }

    if (!userId) {
      return res.json({
        success: true,
        message: "Language set (local session)",
      });
    }

    connection = await pool.getConnection();
    await connection.execute(
      "UPDATE users SET language_preference = ? WHERE id = ?",
      [language, userId]
    );

    res.json({
      success: true,
      message: "Language preference saved successfully",
    });
  } catch (error) {
    console.error("Error saving language preference:", error);
    res.json({ success: true, message: "Language set (local session)" });
  } finally {
    if (connection) connection.release();
  }
});

// Endpoint to set active system language
router.post("/translations/set-active", async (req, res) => {
  let connection;
  try {
    const { language_code } = req.body;

    if (!language_code) {
      return res.status(400).json({ error: "Language code is required" });
    }

    connection = await pool.getConnection();

    // Check if language exists
    const [languages] = await connection.execute(
      "SELECT id FROM translations WHERE language_code = ? LIMIT 1",
      [language_code]
    );

    if (languages.length === 0) {
      return res.status(404).json({ error: "Language not found" });
    }

    // Save active language in bot settings
    await connection.execute(
      `INSERT INTO bot_settings (setting_key, setting_value, updated_at) 
       VALUES ('active_language', ?, NOW()) 
       ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value), updated_at = NOW()`,
      [language_code]
    );

    res.json({ success: true, message: "Active language set successfully" });
  } catch (error) {
    console.error("Error setting active language:", error);
    res.status(500).json({ error: "Error setting active language" });
  } finally {
    if (connection) connection.release();
  }
});

module.exports = router;
