// Service Worker - AI WhatsApp Web
// Version with complete offline functionality

const CACHE_NAME = 'AI-whatsapp-v1.0.0';
const STATIC_CACHE = 'ai-static-v1.0.0';
const DYNAMIC_CACHE = 'ai-dynamic-v1.0.0';
const OFFLINE_URL = '/offline.html';

// Files for static cache (always available offline)
const STATIC_FILES = [
  '/',
  '/user',
  '/admin',
  '/mobile',
  '/css/style.css',
  '/css/admin.css',
  '/js/user.js',
  '/js/admin.js',
  '/js/chat.js',
  '/js/mobile-whatsapp.js',
  '/js/pwa.js',
  '/js/offline-storage.js',
  '/js/background-sync.js',
  '/manifest.json',
  '/icon-192x192.png',
  '/icon-512x512.png',
  '/offline.html'
];

// Dynamic files that can be cached
const DYNAMIC_FILES = [
  '/api/chat/messages',
  '/api/chat/contacts',
  '/api/admin/dashboard'
];

// URLs that should never be cached
const NEVER_CACHE = [
  '/api/auth/login',
  '/api/auth/logout',
  '/api/payments/',
  '/api/mass-improved/campaigns'
];

// ===== SERVICE WORKER INSTALLATION =====

self.addEventListener('install', event => {
  console.log('[SW] Installing Service Worker...');
  
  event.waitUntil(
    caches.open(STATIC_CACHE)
      .then(cache => {
        console.log('[SW] Caching static files...');
        return cache.addAll(STATIC_FILES);
      })
      .then(() => {
        console.log('[SW] Static files cached successfully');
        return self.skipWaiting();
      })
      .catch(error => {
        console.error('[SW] Error caching static files:', error);
      })
  );
});

// ===== SERVICE WORKER ACTIVATION =====

self.addEventListener('activate', event => {
  console.log('[SW] Activating Service Worker...');
  
  event.waitUntil(
    caches.keys()
      .then(cacheNames => {
        return Promise.all(
          cacheNames.map(cacheName => {
            // Remove old caches
            if (cacheName !== STATIC_CACHE && cacheName !== DYNAMIC_CACHE) {
              console.log('[SW] Removing old cache:', cacheName);
              return caches.delete(cacheName);
            }
          })
        );
      })
      .then(() => {
        console.log('[SW] Service Worker activated');
        return self.clients.claim();
      })
  );
});

// ===== REQUEST INTERCEPTION =====

self.addEventListener('fetch', event => {
  const { request } = event;
  const url = new URL(request.url);
  
  // Ignore requests that should not be cached
  if (NEVER_CACHE.some(path => url.pathname.startsWith(path))) {
    return;
  }
  
  // Different strategy for different request types
  if (request.method === 'GET') {
    if (STATIC_FILES.includes(url.pathname)) {
      // Cache First for static files
      event.respondWith(cacheFirst(request));
    } else if (url.pathname.startsWith('/api/')) {
      // Network First for APIs
      event.respondWith(networkFirst(request));
    } else {
      // Stale While Revalidate for pages
      event.respondWith(staleWhileRevalidate(request));
    }
  }
});

// ===== CACHE STRATEGIES =====

// Cache First - Search cache first, then network
async function cacheFirst(request) {
  try {
    const cachedResponse = await caches.match(request);
    if (cachedResponse) {
      return cachedResponse;
    }
    
    const networkResponse = await fetch(request);
    
    // Cache response if successful
    if (networkResponse.status === 200) {
      const cache = await caches.open(STATIC_CACHE);
      cache.put(request, networkResponse.clone());
    }
    
    return networkResponse;
  } catch (error) {
    console.error('[SW] Error in cacheFirst:', error);
    
    // Return offline page if available
    if (request.destination === 'document') {
      return caches.match('/offline.html');
    }
    
    throw error;
  }
}

// Network First - Search network first, then cache
async function networkFirst(request) {
  try {
    const networkResponse = await fetch(request);
    
    // Cache response if successful
    if (networkResponse.status === 200) {
      const cache = await caches.open(DYNAMIC_CACHE);
      cache.put(request, networkResponse.clone());
    }
    
    return networkResponse;
  } catch (error) {
    console.log('[SW] Network unavailable, searching cache:', request.url);
    
    const cachedResponse = await caches.match(request);
    if (cachedResponse) {
      return cachedResponse;
    }
    
    // Return offline response for APIs
    return new Response(
      JSON.stringify({
        error: 'No internet connection',
        offline: true,
        timestamp: new Date().toISOString()
      }),
      {
        status: 503,
        statusText: 'Service Unavailable',
        headers: {
          'Content-Type': 'application/json'
        }
      }
    );
  }
}

// Stale While Revalidate - Return from cache and update in background
async function staleWhileRevalidate(request) {
  const cache = await caches.open(DYNAMIC_CACHE);
  const cachedResponse = await cache.match(request);
  
  // Search network in background
  const networkResponsePromise = fetch(request)
    .then(networkResponse => {
      if (networkResponse.status === 200) {
        cache.put(request, networkResponse.clone());
      }
      return networkResponse;
    })
    .catch(error => {
      console.log('[SW] Network error:', error);
      return null;
    });
  
  // Return cache immediately if available
  if (cachedResponse) {
    return cachedResponse;
  }
  
  // Wait for network response if no cache
  return networkResponsePromise || caches.match('/offline.html');
}

// ===== PUSH NOTIFICATIONS =====

self.addEventListener('push', event => {
  console.log('[SW] Push received:', event);
  
  let notificationData = {
    title: 'AI WhatsApp',
    body: 'New message received',
    icon: '/icon-192x192.png',
    badge: '/icon-192x192.png',
    tag: 'message',
    requireInteraction: true,
    actions: [
      {
        action: 'open',
        title: 'Open Chat',
        icon: '/icon-192x192.png'
      },
      {
        action: 'reply',
        title: 'Reply',
        icon: '/icon-192x192.png'
      }
    ],
    data: {
      url: '/user',
      timestamp: Date.now()
    }
  };
  
  // Parse data if sent by server
  if (event.data) {
    try {
      const pushData = event.data.json();
      notificationData = { ...notificationData, ...pushData };
    } catch (error) {
      console.error('[SW] Error parsing push data:', error);
    }
  }
  
  // Customize notification based on type
  if (notificationData.type === 'message') {
    notificationData.title = `New message from ${notificationData.sender || 'Contact'}`;
    notificationData.body = notificationData.message || 'You received a new message';
    notificationData.tag = `message-${notificationData.chatId || 'default'}`;
  } else if (notificationData.type === 'payment') {
    notificationData.title = 'Payment Received';
    notificationData.body = `Payment of $${notificationData.amount} has been confirmed`;
    notificationData.tag = 'payment';
  }
  
  event.waitUntil(
    self.registration.showNotification(notificationData.title, notificationData)
  );
});

// ===== NOTIFICATION CLICKS =====

self.addEventListener('notificationclick', event => {
  console.log('[SW] Notification clicked:', event);
  
  event.notification.close();
  
  const action = event.action;
  const data = event.notification.data || {};
  
  if (action === 'reply') {
    // Open quick reply interface
    event.waitUntil(
      clients.openWindow('/user?action=reply&chatId=' + (data.chatId || ''))
    );
  } else {
    // Default action or 'open'
    event.waitUntil(
      clients.matchAll({ type: 'window', includeUncontrolled: true })
        .then(clientList => {
          // Check if there's already an open window
          for (const client of clientList) {
            if (client.url.includes('/user') && 'focus' in client) {
              return client.focus();
            }
          }
          
          // Open new window if none is open
          if (clients.openWindow) {
            return clients.openWindow(data.url || '/user');
          }
        })
    );
  }
});

// ===== BACKGROUND SYNC =====

self.addEventListener('sync', event => {
  console.log('[SW] Background sync:', event.tag);
  
  if (event.tag === 'sync-messages') {
    event.waitUntil(syncMessages());
  } else if (event.tag === 'sync-contacts') {
    event.waitUntil(syncContacts());
  }
});

async function syncMessages() {
  try {
    console.log('[SW] Syncing messages...');
    
    // Fetch pending messages from IndexedDB
    const pendingMessages = await getPendingMessages();
    
    for (const message of pendingMessages) {
      try {
        const response = await fetch('/api/chat/send', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${message.token}`
          },
          body: JSON.stringify(message.data)
        });
        
        if (response.ok) {
          // Remove message from pending queue
          await removePendingMessage(message.id);
          console.log('[SW] Message synced:', message.id);
        }
      } catch (error) {
        console.error('[SW] Error syncing message:', error);
      }
    }
  } catch (error) {
    console.error('[SW] Error in message sync:', error);
  }
}

async function syncContacts() {
  try {
    console.log('[SW] Syncing contacts...');
    
    const response = await fetch('/api/chat/contacts');
    if (response.ok) {
      const contacts = await response.json();
      await saveContactsToCache(contacts);
    }
  } catch (error) {
    console.error('[SW] Error in contact sync:', error);
  }
}

// ===== HELPER FUNCTIONS FOR INDEXEDDB =====

async function getPendingMessages() {
  // Implement IndexedDB search
  return [];
}

async function removePendingMessage(messageId) {
  // Implement IndexedDB removal
  console.log('[SW] Removing pending message:', messageId);
}

async function saveContactsToCache(contacts) {
  // Implement IndexedDB save
  console.log('[SW] Saving contacts to cache:', contacts.length);
}

// ===== CACHE CLEANUP =====

self.addEventListener('message', event => {
  if (event.data && event.data.type === 'CLEAR_CACHE') {
    event.waitUntil(
      caches.keys().then(cacheNames => {
        return Promise.all(
          cacheNames.map(cacheName => {
            if (cacheName.startsWith('ai-')) {
              console.log('[SW] Clearing cache:', cacheName);
              return caches.delete(cacheName);
            }
          })
        );
      })
    );
  }
});

// ===== SERVICE WORKER UPDATE =====

self.addEventListener('message', event => {
  if (event.data && event.data.type === 'SKIP_WAITING') {
    self.skipWaiting();
  }
});

console.log('[SW] Service Worker loaded - Version:', CACHE_NAME);
