// ===================================================================
//  PERSISTENCE FIX - ENSURE SETTINGS ARE MAINTAINED
// ===================================================================

/**
 * This file ensures that profile settings and translations
 * are loaded and applied correctly when reloading the page
 */

(function() {
    'use strict';
    
    console.log('[PERSISTENCE FIX] Starting persistence fix...');
    
    /**
     * Apply saved settings as soon as possible
     */
    function applyStoredSettings() {
        // 1. Apply saved translations
        applyStoredTranslations();
        
        // 2. Apply custom colors
        applyStoredColors();
        
        // 3. Apply system name
        applyStoredSystemName();
    }
    
    /**
     * Apply saved translations from localStorage
     */
    function applyStoredTranslations() {
        try {
            const savedLang = localStorage.getItem('language');
            const savedTranslations = localStorage.getItem('i18nTranslations');
            
            if (savedLang && typeof i18n !== 'undefined') {
                i18n.currentLanguage = savedLang;
                console.log('[PERSISTENCE FIX] Language loaded:', savedLang);
            }
            
            if (savedTranslations && typeof i18n !== 'undefined') {
                const translations = JSON.parse(savedTranslations);
                
                // Merge saved translations with defaults
                Object.keys(translations).forEach(lang => {
                    if (!i18n.translations[lang]) {
                        i18n.translations[lang] = {};
                    }
                    Object.assign(i18n.translations[lang], translations[lang]);
                });
                
                console.log('[PERSISTENCE FIX] Custom translations loaded');
            }
            
            // Apply translations to elements
            if (typeof i18n !== 'undefined' && typeof i18n.updatePageTranslations === 'function') {
                i18n.updatePageTranslations();
                console.log('[PERSISTENCE FIX] Translations applied to elements');
            }
        } catch (error) {
            console.error('[PERSISTENCE FIX] Error applying translations:', error);
        }
    }
    
    /**
     * Apply custom colors from localStorage
     */
    function applyStoredColors() {
        try {
            const savedColors = localStorage.getItem('customColors');
            
            if (savedColors) {
                const colors = JSON.parse(savedColors);
                
                // Apply via CSS custom properties
                const root = document.documentElement;
                root.style.setProperty('--primary-color', colors.primaryColor);
                root.style.setProperty('--menu-background', colors.menuBackground);
                root.style.setProperty('--menu-active', colors.menuActive);
                root.style.setProperty('--menu-hover', colors.menuHover);
                root.style.setProperty('--card-background', colors.cardBackground);
                root.style.setProperty('--card-border', colors.cardBorder);
                root.style.setProperty('--title-color', colors.titleColor);
                root.style.setProperty('--text-color', colors.textColor);
                root.style.setProperty('--button-primary', colors.buttonPrimary);
                root.style.setProperty('--button-success', colors.buttonSuccess);
                root.style.setProperty('--button-danger', colors.buttonDanger);
                root.style.setProperty('--button-warning', colors.buttonWarning);
                
                // Apply directly to specific elements
                setTimeout(() => {
                    const sidebar = document.querySelector('.sidebar');
                    if (sidebar) {
                        sidebar.style.background = colors.menuBackground;
                    }
                    
                    const activeNavItem = document.querySelector('.nav-item.active');
                    if (activeNavItem) {
                        activeNavItem.style.background = colors.menuActive;
                    }
                }, 100);
                
                console.log('[PERSISTENCE FIX] Custom colors applied');
            }
        } catch (error) {
            console.error('[PERSISTENCE FIX] Error applying colors:', error);
        }
    }
    
    /**
     * Apply system name from localStorage
     */
    function applyStoredSystemName() {
        try {
            const savedSystemName = localStorage.getItem('systemName');
            
            if (savedSystemName) {
                // Update sidebar logo
                setTimeout(() => {
                    const logoElement = document.querySelector('.sidebar .logo h2');
                    if (logoElement) {
                        logoElement.textContent = savedSystemName;
                    }
                    
                    // Update page title
                    document.title = savedSystemName;
                }, 100);
                
                console.log('[PERSISTENCE FIX] System name applied:', savedSystemName);
            }
        } catch (error) {
            console.error('[PERSISTENCE FIX] Error applying system name:', error);
        }
    }
    
    /**
     * Observe DOM changes to reapply settings
     */
    function setupDOMObserver() {
        // Observe sidebar changes to reapply colors
        const observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                if (mutation.type === 'childList' || mutation.type === 'attributes') {
                    // Reapply colors when DOM changes
                    setTimeout(() => {
                        applyStoredColors();
                        applyStoredSystemName();
                    }, 50);
                }
            });
        });
        
        // Observe the body
        const targetNode = document.body;
        if (targetNode) {
            observer.observe(targetNode, {
                childList: true,
                subtree: true,
                attributes: false
            });
        }
    }
    
    /**
     * Intercept showPage function to reapply settings
     */
    function interceptShowPage() {
        if (typeof window.showPage === 'function') {
            const originalShowPage = window.showPage;
            
            window.showPage = async function(...args) {
                const result = await originalShowPage.apply(this, args);
                
                // Reapply settings after changing page
                setTimeout(() => {
                    applyStoredTranslations();
                    applyStoredColors();
                }, 200);
                
                return result;
            };
            
            console.log('[PERSISTENCE FIX] showPage interceptor installed');
        }
    }
    
    // Execute when DOM loads
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            applyStoredSettings();
            setupDOMObserver();
            interceptShowPage();
        });
    } else {
        applyStoredSettings();
        setupDOMObserver();
        interceptShowPage();
    }
    
    // Also execute when page loads completely
    window.addEventListener('load', () => {
        setTimeout(() => {
            applyStoredSettings();
        }, 100);
    });
    
    console.log('[PERSISTENCE FIX] Persistence system initialized');
})();

