// Offline Storage System using IndexedDB

class OfflineStorage {
    constructor() {
        this.dbName = 'AIWhatsAppDB';
        this.dbVersion = 1;
        this.db = null;
        
        this.stores = {
            messages: 'messages',
            contacts: 'contacts',
            conversations: 'conversations',
            media: 'media',
            settings: 'settings',
            pendingActions: 'pendingActions',
            translations: 'translations',
            faqData: 'faqData'
        };
        
        this.init();
    }
    
    async init() {
        try {
            this.db = await this.openDatabase();
            console.log('[OfflineStorage] Database initialized successfully');
            
            // Clean old data periodically
            this.scheduleCleanup();
        } catch (error) {
            console.error('[OfflineStorage] Failed to initialize database:', error);
        }
    }
    
    // ===== DATABASE SETUP =====
    
    openDatabase() {
        return new Promise((resolve, reject) => {
            const request = indexedDB.open(this.dbName, this.dbVersion);
            
            request.onerror = () => {
                reject(new Error('Failed to open database'));
            };
            
            request.onsuccess = (event) => {
                resolve(event.target.result);
            };
            
            request.onupgradeneeded = (event) => {
                const db = event.target.result;
                
                // Messages store
                if (!db.objectStoreNames.contains(this.stores.messages)) {
                    const messagesStore = db.createObjectStore(this.stores.messages, { 
                        keyPath: 'id' 
                    });
                    messagesStore.createIndex('chatId', 'chatId', { unique: false });
                    messagesStore.createIndex('timestamp', 'timestamp', { unique: false });
                    messagesStore.createIndex('type', 'type', { unique: false });
                }
                
                // Contacts store
                if (!db.objectStoreNames.contains(this.stores.contacts)) {
                    const contactsStore = db.createObjectStore(this.stores.contacts, { 
                        keyPath: 'id' 
                    });
                    contactsStore.createIndex('phone', 'phone', { unique: true });
                    contactsStore.createIndex('name', 'name', { unique: false });
                }
                
                // Conversations store
                if (!db.objectStoreNames.contains(this.stores.conversations)) {
                    const conversationsStore = db.createObjectStore(this.stores.conversations, { 
                        keyPath: 'id' 
                    });
                    conversationsStore.createIndex('lastActivity', 'lastActivity', { unique: false });
                    conversationsStore.createIndex('unreadCount', 'unreadCount', { unique: false });
                }
                
                // Media store
                if (!db.objectStoreNames.contains(this.stores.media)) {
                    const mediaStore = db.createObjectStore(this.stores.media, { 
                        keyPath: 'id' 
                    });
                    mediaStore.createIndex('messageId', 'messageId', { unique: false });
                    mediaStore.createIndex('type', 'type', { unique: false });
                    mediaStore.createIndex('size', 'size', { unique: false });
                }
                
                // Settings store
                if (!db.objectStoreNames.contains(this.stores.settings)) {
                    db.createObjectStore(this.stores.settings, { 
                        keyPath: 'key' 
                    });
                }
                
                // Pending actions store
                if (!db.objectStoreNames.contains(this.stores.pendingActions)) {
                    const pendingStore = db.createObjectStore(this.stores.pendingActions, { 
                        keyPath: 'id' 
                    });
                    pendingStore.createIndex('type', 'type', { unique: false });
                    pendingStore.createIndex('timestamp', 'timestamp', { unique: false });
                }
                
                // Translations store
                if (!db.objectStoreNames.contains(this.stores.translations)) {
                    const translationsStore = db.createObjectStore(this.stores.translations, { 
                        keyPath: 'key' 
                    });
                    translationsStore.createIndex('language', 'language', { unique: false });
                }
                
                // FAQ data store
                if (!db.objectStoreNames.contains(this.stores.faqData)) {
                    const faqStore = db.createObjectStore(this.stores.faqData, { 
                        keyPath: 'id' 
                    });
                    faqStore.createIndex('category', 'category', { unique: false });
                    faqStore.createIndex('active', 'active', { unique: false });
                }
            };
        });
    }
    
    // ===== GENERIC CRUD OPERATIONS =====
    
    async add(storeName, data) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], 'readwrite');
            const store = transaction.objectStore(storeName);
            
            // Add timestamp if it doesn't exist
            if (!data.createdAt) {
                data.createdAt = new Date().toISOString();
            }
            data.updatedAt = new Date().toISOString();
            
            const request = store.add(data);
            
            request.onsuccess = () => {
                resolve(data);
            };
            
            request.onerror = () => {
                reject(new Error(`Failed to add data to ${storeName}`));
            };
        });
    }
    
    async update(storeName, data) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], 'readwrite');
            const store = transaction.objectStore(storeName);
            
            data.updatedAt = new Date().toISOString();
            
            const request = store.put(data);
            
            request.onsuccess = () => {
                resolve(data);
            };
            
            request.onerror = () => {
                reject(new Error(`Failed to update data in ${storeName}`));
            };
        });
    }
    
    async get(storeName, id) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], 'readonly');
            const store = transaction.objectStore(storeName);
            const request = store.get(id);
            
            request.onsuccess = () => {
                resolve(request.result);
            };
            
            request.onerror = () => {
                reject(new Error(`Failed to get data from ${storeName}`));
            };
        });
    }
    
    async getAll(storeName, limit = null) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], 'readonly');
            const store = transaction.objectStore(storeName);
            const request = limit ? store.getAll(null, limit) : store.getAll();
            
            request.onsuccess = () => {
                resolve(request.result);
            };
            
            request.onerror = () => {
                reject(new Error(`Failed to get all data from ${storeName}`));
            };
        });
    }
    
    async delete(storeName, id) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], 'readwrite');
            const store = transaction.objectStore(storeName);
            const request = store.delete(id);
            
            request.onsuccess = () => {
                resolve(true);
            };
            
            request.onerror = () => {
                reject(new Error(`Failed to delete data from ${storeName}`));
            };
        });
    }
    
    async clear(storeName) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], 'readwrite');
            const store = transaction.objectStore(storeName);
            const request = store.clear();
            
            request.onsuccess = () => {
                resolve(true);
            };
            
            request.onerror = () => {
                reject(new Error(`Failed to clear ${storeName}`));
            };
        });
    }
    
    // ===== MESSAGES =====
    
    async saveMessage(message) {
        try {
            // Garantir que a mensagem tem um ID único
            if (!message.id) {
                message.id = this.generateId();
            }
            
            // Adicionar informações de cache
            message.cached = true;
            message.syncStatus = message.syncStatus || 'pending';
            
            return await this.add(this.stores.messages, message);
        } catch (error) {
            // Se já existe, atualizar
            return await this.update(this.stores.messages, message);
        }
    }
    
    async getMessages(chatId, limit = 50) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([this.stores.messages], 'readonly');
            const store = transaction.objectStore(this.stores.messages);
            const index = store.index('chatId');
            const request = index.getAll(chatId);
            
            request.onsuccess = () => {
                let messages = request.result;
                
                // Sort by timestamp (newest first)
                messages.sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp));
                
                // Limit quantity if specified
                if (limit) {
                    messages = messages.slice(0, limit);
                }
                
                resolve(messages);
            };
            
            request.onerror = () => {
                reject(new Error('Failed to get messages'));
            };
        });
    }
    
    async getUnsentMessages() {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([this.stores.messages], 'readonly');
            const store = transaction.objectStore(this.stores.messages);
            const request = store.getAll();
            
            request.onsuccess = () => {
                const unsentMessages = request.result.filter(msg => 
                    msg.syncStatus === 'pending' || msg.syncStatus === 'failed'
                );
                resolve(unsentMessages);
            };
            
            request.onerror = () => {
                reject(new Error('Failed to get unsent messages'));
            };
        });
    }
    
    async markMessageAsSent(messageId) {
        try {
            const message = await this.get(this.stores.messages, messageId);
            if (message) {
                message.syncStatus = 'sent';
                await this.update(this.stores.messages, message);
            }
        } catch (error) {
            console.error('[OfflineStorage] Failed to mark message as sent:', error);
        }
    }
    
    // ===== CONTACTS =====
    
    async saveContact(contact) {
        try {
            if (!contact.id) {
                contact.id = this.generateId();
            }
            
            return await this.add(this.stores.contacts, contact);
        } catch (error) {
            return await this.update(this.stores.contacts, contact);
        }
    }
    
    async getContacts() {
        return await this.getAll(this.stores.contacts);
    }
    
    async searchContacts(query) {
        const contacts = await this.getContacts();
        const lowerQuery = query.toLowerCase();
        
        return contacts.filter(contact => 
            contact.name.toLowerCase().includes(lowerQuery) ||
            contact.phone.includes(query)
        );
    }
    
    // ===== CONVERSATIONS =====
    
    async saveConversation(conversation) {
        try {
            if (!conversation.id) {
                conversation.id = this.generateId();
            }
            
            conversation.lastActivity = new Date().toISOString();
            
            return await this.add(this.stores.conversations, conversation);
        } catch (error) {
            return await this.update(this.stores.conversations, conversation);
        }
    }
    
    async getConversations() {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([this.stores.conversations], 'readonly');
            const store = transaction.objectStore(this.stores.conversations);
            const index = store.index('lastActivity');
            const request = index.getAll();
            
            request.onsuccess = () => {
                // Sort by last activity (newest first)
                const conversations = request.result.sort((a, b) => 
                    new Date(b.lastActivity) - new Date(a.lastActivity)
                );
                resolve(conversations);
            };
            
            request.onerror = () => {
                reject(new Error('Failed to get conversations'));
            };
        });
    }
    
    // ===== MEDIA =====
    
    async saveMedia(media) {
        try {
            if (!media.id) {
                media.id = this.generateId();
            }
            
            // Convert blob to base64 for storage
            if (media.blob) {
                media.data = await this.blobToBase64(media.blob);
                delete media.blob;
            }
            
            return await this.add(this.stores.media, media);
        } catch (error) {
            return await this.update(this.stores.media, media);
        }
    }
    
    async getMedia(messageId) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([this.stores.media], 'readonly');
            const store = transaction.objectStore(this.stores.media);
            const index = store.index('messageId');
            const request = index.get(messageId);
            
            request.onsuccess = () => {
                const media = request.result;
                if (media && media.data) {
                    // Convert base64 back to blob
                    media.blob = this.base64ToBlob(media.data, media.mimeType);
                    media.url = URL.createObjectURL(media.blob);
                }
                resolve(media);
            };
            
            request.onerror = () => {
                reject(new Error('Failed to get media'));
            };
        });
    }
    
    // ===== SETTINGS =====
    
    async saveSetting(key, value) {
        const setting = {
            key: key,
            value: value,
            updatedAt: new Date().toISOString()
        };
        
        try {
            return await this.add(this.stores.settings, setting);
        } catch (error) {
            return await this.update(this.stores.settings, setting);
        }
    }
    
    async getSetting(key, defaultValue = null) {
        try {
            const setting = await this.get(this.stores.settings, key);
            return setting ? setting.value : defaultValue;
        } catch (error) {
            return defaultValue;
        }
    }
    
    // ===== PENDING ACTIONS =====
    
    async addPendingAction(action) {
        action.id = this.generateId();
        action.timestamp = new Date().toISOString();
        action.retryCount = 0;
        
        return await this.add(this.stores.pendingActions, action);
    }
    
    async getPendingActions() {
        return await this.getAll(this.stores.pendingActions);
    }
    
    async removePendingAction(actionId) {
        return await this.delete(this.stores.pendingActions, actionId);
    }
    
    async incrementRetryCount(actionId) {
        try {
            const action = await this.get(this.stores.pendingActions, actionId);
            if (action) {
                action.retryCount = (action.retryCount || 0) + 1;
                action.lastRetry = new Date().toISOString();
                await this.update(this.stores.pendingActions, action);
            }
        } catch (error) {
            console.error('[OfflineStorage] Failed to increment retry count:', error);
        }
    }
    
    // ===== TRANSLATIONS =====
    
    async saveTranslation(key, language, value) {
        const translation = {
            key: `${language}_${key}`,
            language: language,
            originalKey: key,
            value: value,
            updatedAt: new Date().toISOString()
        };
        
        try {
            return await this.add(this.stores.translations, translation);
        } catch (error) {
            return await this.update(this.stores.translations, translation);
        }
    }
    
    async getTranslation(key, language) {
        try {
            const translation = await this.get(this.stores.translations, `${language}_${key}`);
            return translation ? translation.value : null;
        } catch (error) {
            return null;
        }
    }
    
    async getTranslationsByLanguage(language) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([this.stores.translations], 'readonly');
            const store = transaction.objectStore(this.stores.translations);
            const index = store.index('language');
            const request = index.getAll(language);
            
            request.onsuccess = () => {
                resolve(request.result);
            };
            
            request.onerror = () => {
                reject(new Error('Failed to get translations'));
            };
        });
    }
    
    // ===== FAQ DATA =====
    
    async saveFAQItem(faqItem) {
        try {
            if (!faqItem.id) {
                faqItem.id = this.generateId();
            }
            
            return await this.add(this.stores.faqData, faqItem);
        } catch (error) {
            return await this.update(this.stores.faqData, faqItem);
        }
    }
    
    async getFAQItems() {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([this.stores.faqData], 'readonly');
            const store = transaction.objectStore(this.stores.faqData);
            const index = store.index('active');
            const request = index.getAll(true);
            
            request.onsuccess = () => {
                resolve(request.result);
            };
            
            request.onerror = () => {
                reject(new Error('Failed to get FAQ items'));
            };
        });
    }
    
    // ===== SYNC OPERATIONS =====
    
    async syncWithServer() {
        if (!navigator.onLine) {
            console.log('[OfflineStorage] Offline, skipping sync');
            return;
        }
        
        try {
            // Sync pending messages
            await this.syncPendingMessages();
            
            // Sync pending actions
            await this.syncPendingActions();
            
            // Download updated data from server
            await this.downloadServerData();
            
            console.log('[OfflineStorage] Sync completed successfully');
        } catch (error) {
            console.error('[OfflineStorage] Sync failed:', error);
        }
    }
    
    async syncPendingMessages() {
        const unsentMessages = await this.getUnsentMessages();
        
        for (const message of unsentMessages) {
            try {
                const token = localStorage.getItem('token');
                const response = await fetch('/api/chat/send', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${token}`
                    },
                    body: JSON.stringify(message)
                });
                
                if (response.ok) {
                    await this.markMessageAsSent(message.id);
                } else {
                    throw new Error('Failed to send message');
                }
            } catch (error) {
                console.error('[OfflineStorage] Failed to sync message:', error);
                
                // Mark as failed if too many attempts
                const retryCount = message.retryCount || 0;
                if (retryCount >= 3) {
                    message.syncStatus = 'failed';
                    await this.update(this.stores.messages, message);
                }
            }
        }
    }
    
    async syncPendingActions() {
        const pendingActions = await this.getPendingActions();
        
        for (const action of pendingActions) {
            try {
                await this.executePendingAction(action);
                await this.removePendingAction(action.id);
            } catch (error) {
                console.error('[OfflineStorage] Failed to execute pending action:', error);
                await this.incrementRetryCount(action.id);
                
                // Remove if too many attempts
                if (action.retryCount >= 5) {
                    await this.removePendingAction(action.id);
                }
            }
        }
    }
    
    async executePendingAction(action) {
        const token = localStorage.getItem('token');
        
        const response = await fetch(action.url, {
            method: action.method || 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`,
                ...action.headers
            },
            body: action.data ? JSON.stringify(action.data) : undefined
        });
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        return await response.json();
    }
    
    async downloadServerData() {
        try {
            const token = localStorage.getItem('token');
            
            // Download updated contacts
            const contactsResponse = await fetch('/api/chat/contacts', {
                headers: { 'Authorization': `Bearer ${token}` }
            });
            
            if (contactsResponse.ok) {
                const contacts = await contactsResponse.json();
                for (const contact of contacts) {
                    await this.saveContact(contact);
                }
            }
            
            // Download updated settings
            const settingsResponse = await fetch('/api/settings', {
                headers: { 'Authorization': `Bearer ${token}` }
            });
            
            if (settingsResponse.ok) {
                const settings = await settingsResponse.json();
                for (const [key, value] of Object.entries(settings)) {
                    await this.saveSetting(key, value);
                }
            }
            
        } catch (error) {
            console.error('[OfflineStorage] Failed to download server data:', error);
        }
    }
    
    // ===== CLEANUP =====
    
    scheduleCleanup() {
        // Run cleanup every 24 hours
        setInterval(() => {
            this.cleanup();
        }, 24 * 60 * 60 * 1000);
        
        // Run initial cleanup after 1 minute
        setTimeout(() => {
            this.cleanup();
        }, 60 * 1000);
    }
    
    async cleanup() {
        try {
            console.log('[OfflineStorage] Starting cleanup...');
            
            // Remove old messages (more than 30 days)
            await this.cleanupOldMessages();
            
            // Remove unreferenced media
            await this.cleanupOrphanedMedia();
            
            // Remove very old pending actions
            await this.cleanupOldPendingActions();
            
            console.log('[OfflineStorage] Cleanup completed');
        } catch (error) {
            console.error('[OfflineStorage] Cleanup failed:', error);
        }
    }
    
    async cleanupOldMessages() {
        const thirtyDaysAgo = new Date();
        thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);
        
        const messages = await this.getAll(this.stores.messages);
        
        for (const message of messages) {
            const messageDate = new Date(message.timestamp);
            if (messageDate < thirtyDaysAgo) {
                await this.delete(this.stores.messages, message.id);
            }
        }
    }
    
    async cleanupOrphanedMedia() {
        const media = await this.getAll(this.stores.media);
        const messages = await this.getAll(this.stores.messages);
        const messageIds = new Set(messages.map(m => m.id));
        
        for (const mediaItem of media) {
            if (!messageIds.has(mediaItem.messageId)) {
                await this.delete(this.stores.media, mediaItem.id);
            }
        }
    }
    
    async cleanupOldPendingActions() {
        const sevenDaysAgo = new Date();
        sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);
        
        const actions = await this.getPendingActions();
        
        for (const action of actions) {
            const actionDate = new Date(action.timestamp);
            if (actionDate < sevenDaysAgo) {
                await this.removePendingAction(action.id);
            }
        }
    }
    
    // ===== UTILITIES =====
    
    generateId() {
        return Date.now().toString(36) + Math.random().toString(36).substr(2);
    }
    
    async blobToBase64(blob) {
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            reader.onload = () => resolve(reader.result.split(',')[1]);
            reader.onerror = reject;
            reader.readAsDataURL(blob);
        });
    }
    
    base64ToBlob(base64, mimeType) {
        const byteCharacters = atob(base64);
        const byteNumbers = new Array(byteCharacters.length);
        
        for (let i = 0; i < byteCharacters.length; i++) {
            byteNumbers[i] = byteCharacters.charCodeAt(i);
        }
        
        const byteArray = new Uint8Array(byteNumbers);
        return new Blob([byteArray], { type: mimeType });
    }
    
    // ===== PUBLIC API =====
    
    async getStorageInfo() {
        const info = {};
        
        for (const [name, storeName] of Object.entries(this.stores)) {
            try {
                const data = await this.getAll(storeName);
                info[name] = {
                    count: data.length,
                    size: JSON.stringify(data).length
                };
            } catch (error) {
                info[name] = { count: 0, size: 0, error: error.message };
            }
        }
        
        return info;
    }
    
    async clearAllData() {
        for (const storeName of Object.values(this.stores)) {
            await this.clear(storeName);
        }
        console.log('[OfflineStorage] All data cleared');
    }
    
    async exportData() {
        const data = {};
        
        for (const [name, storeName] of Object.entries(this.stores)) {
            data[name] = await this.getAll(storeName);
        }
        
        return data;
    }
    
    async importData(data) {
        for (const [name, items] of Object.entries(data)) {
            const storeName = this.stores[name];
            if (storeName && Array.isArray(items)) {
                await this.clear(storeName);
                for (const item of items) {
                    await this.add(storeName, item);
                }
            }
        }
        
        console.log('[OfflineStorage] Data imported successfully');
    }
}

// ===== GLOBAL INSTANCE =====

let offlineStorage;

// Initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        offlineStorage = new OfflineStorage();
        window.offlineStorage = offlineStorage;
    });
} else {
    offlineStorage = new OfflineStorage();
    window.offlineStorage = offlineStorage;
}

// Exportar para uso em módulos
if (typeof module !== 'undefined' && module.exports) {
    module.exports = OfflineStorage;
}
