// Mobile WhatsApp Interface JavaScript

class MobileWhatsApp {
    constructor() {
        this.currentChat = null;
        this.contacts = [];
        this.messages = [];
        this.isRecording = false;
        this.recordingStartTime = null;
        this.mediaRecorder = null;
        this.recordedChunks = [];
        this.socket = null;
        this.typingTimeout = null;
        
        this.init();
    }
    
    async init() {
        // Initialize components
        await this.loadContacts();
        this.setupEventListeners();
        this.setupWebSocket();
        this.setupMediaAccess();
        this.setupGeolocation();
        
        // Check if there's a specific chat in URL
        const urlParams = new URLSearchParams(window.location.search);
        const chatId = urlParams.get('chat');
        if (chatId) {
            this.openChat(chatId);
        } else {
            this.showContactsList();
        }
    }
    
    // ===== SETUP =====
    
    setupEventListeners() {
        // Prevent zoom on input
        document.addEventListener('touchstart', (e) => {
            if (e.touches.length > 1) {
                e.preventDefault();
            }
        });
        
        // Adjust viewport height
        this.adjustViewportHeight();
        window.addEventListener('resize', () => this.adjustViewportHeight());
        
        // Hide attachment menu when clicking outside
        document.addEventListener('click', (e) => {
            const attachMenu = document.getElementById('attachmentMenu');
            const attachButton = document.getElementById('attachButton');
            
            if (attachMenu && !attachMenu.contains(e.target) && !attachButton.contains(e.target)) {
                attachMenu.classList.remove('show');
            }
        });
        
        // Back button
        document.getElementById('backButton')?.addEventListener('click', () => {
            this.showContactsList();
        });
        
        // Attachment button
        document.getElementById('attachButton')?.addEventListener('click', () => {
            this.toggleAttachmentMenu();
        });
        
        // Attachment buttons
        document.getElementById('photoButton')?.addEventListener('click', () => {
            this.capturePhoto();
        });
        
        document.getElementById('cameraButton')?.addEventListener('click', () => {
            this.captureVideo();
        });
        
        document.getElementById('documentButton')?.addEventListener('click', () => {
            this.selectDocument();
        });
        
        document.getElementById('locationButton')?.addEventListener('click', () => {
            this.shareLocation();
        });
        
        // Audio recording button
        document.getElementById('recordButton')?.addEventListener('mousedown', () => {
            this.startRecording();
        });
        
        document.getElementById('recordButton')?.addEventListener('mouseup', () => {
            this.stopRecording();
        });
        
        document.getElementById('recordButton')?.addEventListener('touchstart', (e) => {
            e.preventDefault();
            this.startRecording();
        });
        
        document.getElementById('recordButton')?.addEventListener('touchend', (e) => {
            e.preventDefault();
            this.stopRecording();
        });
        
        // Input de mensagem
        const messageInput = document.getElementById('messageInput');
        if (messageInput) {
            messageInput.addEventListener('input', () => {
                this.adjustTextareaHeight();
                this.handleTyping();
            });
            
            messageInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    this.sendMessage();
                }
            });
        }
        
        // Botão enviar
        document.getElementById('sendButton')?.addEventListener('click', () => {
            this.sendMessage();
        });
        
        // Busca de contatos
        document.getElementById('searchInput')?.addEventListener('input', (e) => {
            this.filterContacts(e.target.value);
        });
    }
    
    setupWebSocket() {
        this.socket = io();
        
        this.socket.on('connect', () => {
            console.log('Conectado ao servidor');
        });
        
        this.socket.on('newMessage', (message) => {
            this.handleNewMessage(message);
        });
        
        this.socket.on('messageStatus', (data) => {
            this.updateMessageStatus(data);
        });
        
        this.socket.on('typing', (data) => {
            this.showTypingIndicator(data);
        });
        
        this.socket.on('stopTyping', (data) => {
            this.hideTypingIndicator(data);
        });
    }
    
    async setupMediaAccess() {
        try {
            // Solicitar permissões de mídia
            await navigator.mediaDevices.getUserMedia({ 
                audio: true, 
                video: true 
            });
            console.log('Permissões de mídia concedidas');
        } catch (error) {
            console.warn('Permissões de mídia negadas:', error);
        }
    }
    
    setupGeolocation() {
        if ('geolocation' in navigator) {
            console.log('Geolocalização disponível');
        } else {
            console.warn('Geolocalização não disponível');
        }
    }
    
    // ===== CONTATOS =====
    
    async loadContacts() {
        try {
            // Tentar carregar contatos do dispositivo
            if ('contacts' in navigator && 'ContactsManager' in window) {
                const contacts = await navigator.contacts.select(['name', 'tel'], { multiple: true });
                this.contacts = contacts.map(contact => ({
                    id: this.generateId(),
                    name: contact.name[0],
                    phone: contact.tel[0],
                    avatar: null,
                    lastMessage: '',
                    timestamp: new Date(),
                    unread: 0
                }));
            } else {
                // Carregar contatos do servidor
                const response = await fetch('/api/contacts');
                this.contacts = await response.json();
            }
            
            this.renderContacts();
        } catch (error) {
            console.error('Erro ao carregar contatos:', error);
            // Contatos de exemplo
            this.contacts = [
                {
                    id: '1',
                    name: 'João Silva',
                    phone: '+5511999999999',
                    avatar: null,
                    lastMessage: 'Olá, como você está?',
                    timestamp: new Date(),
                    unread: 2
                },
                {
                    id: '2',
                    name: 'Maria Santos',
                    phone: '+5511888888888',
                    avatar: null,
                    lastMessage: 'Obrigada pela ajuda!',
                    timestamp: new Date(Date.now() - 3600000),
                    unread: 0
                }
            ];
            this.renderContacts();
        }
    }
    
    renderContacts() {
        const contactsList = document.getElementById('contactsList');
        if (!contactsList) return;
        
        contactsList.innerHTML = this.contacts.map(contact => `
            <div class="contact-item" onclick="mobileWhatsApp.openChat('${contact.id}')">
                <div class="contact-avatar">
                    ${contact.avatar ? 
                        `<img src="${contact.avatar}" alt="${contact.name}">` : 
                        `<div class="avatar-placeholder">${contact.name.charAt(0).toUpperCase()}</div>`
                    }
                </div>
                <div class="contact-info">
                    <div class="contact-name">${contact.name}</div>
                    <div class="contact-last-message">${contact.lastMessage}</div>
                </div>
                <div class="contact-meta">
                    <div class="contact-time">${this.formatTime(contact.timestamp)}</div>
                    ${contact.unread > 0 ? `<div class="unread-badge">${contact.unread}</div>` : ''}
                </div>
            </div>
        `).join('');
    }
    
    filterContacts(query) {
        const filtered = this.contacts.filter(contact => 
            contact.name.toLowerCase().includes(query.toLowerCase()) ||
            contact.phone.includes(query)
        );
        
        const contactsList = document.getElementById('contactsList');
        if (!contactsList) return;
        
        contactsList.innerHTML = filtered.map(contact => `
            <div class="contact-item" onclick="mobileWhatsApp.openChat('${contact.id}')">
                <div class="contact-avatar">
                    ${contact.avatar ? 
                        `<img src="${contact.avatar}" alt="${contact.name}">` : 
                        `<div class="avatar-placeholder">${contact.name.charAt(0).toUpperCase()}</div>`
                    }
                </div>
                <div class="contact-info">
                    <div class="contact-name">${contact.name}</div>
                    <div class="contact-last-message">${contact.lastMessage}</div>
                </div>
                <div class="contact-meta">
                    <div class="contact-time">${this.formatTime(contact.timestamp)}</div>
                    ${contact.unread > 0 ? `<div class="unread-badge">${contact.unread}</div>` : ''}
                </div>
            </div>
        `).join('');
    }
    
    // ===== CHAT =====
    
    async openChat(contactId) {
        this.currentChat = contactId;
        const contact = this.contacts.find(c => c.id === contactId);
        
        if (!contact) return;
        
        // Atualizar interface
        document.getElementById('contactName').textContent = contact.name;
        document.getElementById('contactStatus').textContent = 'online';
        
        // Carregar mensagens
        await this.loadMessages(contactId);
        
        // Mostrar tela de chat
        this.showChatScreen();
        
        // Marcar como lido
        contact.unread = 0;
        this.renderContacts();
        
        // Entrar na sala do chat
        this.socket.emit('joinChat', contactId);
    }
    
    async loadMessages(contactId) {
        try {
            const response = await fetch(`/api/messages/${contactId}`);
            this.messages = await response.json();
            this.renderMessages();
        } catch (error) {
            console.error('Erro ao carregar mensagens:', error);
            // Mensagens de exemplo
            this.messages = [
                {
                    id: '1',
                    text: 'Olá! Como você está?',
                    sender: 'other',
                    timestamp: new Date(Date.now() - 3600000),
                    status: 'read'
                },
                {
                    id: '2',
                    text: 'Estou bem, obrigado! E você?',
                    sender: 'me',
                    timestamp: new Date(Date.now() - 3000000),
                    status: 'read'
                },
                {
                    id: '3',
                    text: 'Também estou bem! Vamos nos encontrar hoje?',
                    sender: 'other',
                    timestamp: new Date(Date.now() - 1800000),
                    status: 'read'
                }
            ];
            this.renderMessages();
        }
    }
    
    renderMessages() {
        const messagesContainer = document.getElementById('messagesContainer');
        if (!messagesContainer) return;
        
        messagesContainer.innerHTML = this.messages.map(message => {
            const messageClass = message.sender === 'me' ? 'message-sent' : 'message-received';
            
            let content = '';
            if (message.type === 'text') {
                content = `<div class="message-text">${message.text}</div>`;
            } else if (message.type === 'image') {
                content = `
                    <div class="message-media">
                        <img src="${message.url}" alt="Imagem" onclick="this.requestFullscreen()">
                        ${message.caption ? `<div class="message-caption">${message.caption}</div>` : ''}
                    </div>
                `;
            } else if (message.type === 'video') {
                content = `
                    <div class="message-media">
                        <video controls>
                            <source src="${message.url}" type="video/mp4">
                        </video>
                        ${message.caption ? `<div class="message-caption">${message.caption}</div>` : ''}
                    </div>
                `;
            } else if (message.type === 'audio') {
                content = `
                    <div class="message-audio">
                        <audio controls style="display: none;">
                            <source src="${message.url}" type="audio/mpeg">
                        </audio>
                        <div class="audio-player">
                            <button class="play-button" onclick="this.parentElement.previousElementSibling.play()">▶</button>
                            <div class="audio-duration">${message.duration || '0:00'}</div>
                        </div>
                    </div>
                `;
            } else if (message.type === 'document') {
                content = `
                    <div class="message-document">
                        <div class="document-icon">📄</div>
                        <div class="document-info">
                            <div class="document-name">${message.filename}</div>
                            <div class="document-size">${message.size}</div>
                        </div>
                        <a href="${message.url}" download class="document-download">⬇</a>
                    </div>
                `;
            } else if (message.type === 'location') {
                content = `
                    <div class="message-location">
                        <div class="location-map" onclick="window.open('https://maps.google.com/?q=${message.latitude},${message.longitude}', '_blank')">
                            <div class="location-pin">📍</div>
                            <div class="location-text">Localização compartilhada</div>
                        </div>
                    </div>
                `;
            }
            
            return `
                <div class="message ${messageClass}">
                    ${content}
                    <div class="message-info">
                        <span class="message-time">${this.formatTime(message.timestamp)}</span>
                        ${message.sender === 'me' ? `<span class="message-status ${message.status}">${this.getStatusIcon(message.status)}</span>` : ''}
                    </div>
                </div>
            `;
        }).join('');
        
        // Scroll para a última mensagem
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
    }
    
    async sendMessage() {
        const messageInput = document.getElementById('messageInput');
        const text = messageInput.value.trim();
        
        if (!text || !this.currentChat) return;
        
        const message = {
            id: this.generateId(),
            text: text,
            sender: 'me',
            timestamp: new Date(),
            status: 'sending',
            type: 'text'
        };
        
        // Adicionar à lista local
        this.messages.push(message);
        this.renderMessages();
        
        // Limpar input
        messageInput.value = '';
        this.adjustTextareaHeight();
        
        try {
            // Enviar via socket
            this.socket.emit('sendMessage', {
                chatId: this.currentChat,
                message: message
            });
            
            // Atualizar status
            message.status = 'sent';
            this.renderMessages();
            
        } catch (error) {
            console.error('Erro ao enviar mensagem:', error);
            message.status = 'error';
            this.renderMessages();
        }
    }
    
    // ===== MÍDIA =====
    
    async capturePhoto() {
        try {
            const stream = await navigator.mediaDevices.getUserMedia({ video: true });
            
            // Criar elemento de vídeo temporário
            const video = document.createElement('video');
            video.srcObject = stream;
            video.play();
            
            // Aguardar o vídeo carregar
            await new Promise(resolve => {
                video.onloadedmetadata = resolve;
            });
            
            // Criar canvas para captura
            const canvas = document.createElement('canvas');
            canvas.width = video.videoWidth;
            canvas.height = video.videoHeight;
            
            const ctx = canvas.getContext('2d');
            ctx.drawImage(video, 0, 0);
            
            // Parar stream
            stream.getTracks().forEach(track => track.stop());
            
            // Converter para blob
            canvas.toBlob(async (blob) => {
                await this.sendMediaMessage(blob, 'image');
            }, 'image/jpeg', 0.8);
            
        } catch (error) {
            console.error('Erro ao capturar foto:', error);
            alert('Erro ao acessar a câmera');
        }
    }
    
    async captureVideo() {
        try {
            const stream = await navigator.mediaDevices.getUserMedia({ 
                video: true, 
                audio: true 
            });
            
            // Criar interface de gravação
            const recordingInterface = this.createRecordingInterface(stream, 'video');
            document.body.appendChild(recordingInterface);
            
        } catch (error) {
            console.error('Erro ao capturar vídeo:', error);
            alert('Erro ao acessar a câmera');
        }
    }
    
    createRecordingInterface(stream, type) {
        const container = document.createElement('div');
        container.className = 'recording-interface';
        
        const video = document.createElement('video');
        video.srcObject = stream;
        video.autoplay = true;
        video.muted = true;
        video.style.maxWidth = '250px';
        
        const controls = document.createElement('div');
        controls.className = 'recording-controls';
        
        const recordButton = document.createElement('button');
        recordButton.textContent = 'Gravar';
        recordButton.className = 'record-btn';
        
        const stopButton = document.createElement('button');
        stopButton.textContent = 'Parar';
        stopButton.className = 'stop-btn';
        stopButton.disabled = true;
        
        const cancelButton = document.createElement('button');
        cancelButton.textContent = 'Cancelar';
        cancelButton.className = 'cancel-btn';
        
        let mediaRecorder;
        let recordedChunks = [];
        
        recordButton.onclick = () => {
            recordedChunks = [];
            mediaRecorder = new MediaRecorder(stream);
            
            mediaRecorder.ondataavailable = (event) => {
                if (event.data.size > 0) {
                    recordedChunks.push(event.data);
                }
            };
            
            mediaRecorder.onstop = async () => {
                const blob = new Blob(recordedChunks, { 
                    type: type === 'video' ? 'video/webm' : 'audio/webm' 
                });
                await this.sendMediaMessage(blob, type);
                this.closeRecordingInterface(container, stream);
            };
            
            mediaRecorder.start();
            recordButton.disabled = true;
            stopButton.disabled = false;
        };
        
        stopButton.onclick = () => {
            mediaRecorder.stop();
        };
        
        cancelButton.onclick = () => {
            if (mediaRecorder && mediaRecorder.state === 'recording') {
                mediaRecorder.stop();
            }
            this.closeRecordingInterface(container, stream);
        };
        
        controls.appendChild(recordButton);
        controls.appendChild(stopButton);
        controls.appendChild(cancelButton);
        
        container.appendChild(video);
        container.appendChild(controls);
        
        return container;
    }
    
    closeRecordingInterface(container, stream) {
        stream.getTracks().forEach(track => track.stop());
        container.remove();
    }
    
    selectDocument() {
        const input = document.createElement('input');
        input.type = 'file';
        input.accept = '*/*';
        
        input.onchange = async (e) => {
            const file = e.target.files[0];
            if (file) {
                await this.sendMediaMessage(file, 'document');
            }
        };
        
        input.click();
    }
    
    async shareLocation() {
        if (!('geolocation' in navigator)) {
            alert('Geolocalização não disponível');
            return;
        }
        
        try {
            const position = await new Promise((resolve, reject) => {
                navigator.geolocation.getCurrentPosition(resolve, reject, {
                    enableHighAccuracy: true,
                    timeout: 10000,
                    maximumAge: 60000
                });
            });
            
            const message = {
                id: this.generateId(),
                type: 'location',
                latitude: position.coords.latitude,
                longitude: position.coords.longitude,
                sender: 'me',
                timestamp: new Date(),
                status: 'sending'
            };
            
            this.messages.push(message);
            this.renderMessages();
            
            // Enviar via socket
            this.socket.emit('sendMessage', {
                chatId: this.currentChat,
                message: message
            });
            
        } catch (error) {
            console.error('Erro ao obter localização:', error);
            alert('Erro ao obter localização');
        }
    }
    
    // ===== ÁUDIO =====
    
    async startRecording() {
        if (this.isRecording) return;
        
        try {
            const stream = await navigator.mediaDevices.getUserMedia({ audio: true });
            
            this.mediaRecorder = new MediaRecorder(stream);
            this.recordedChunks = [];
            this.isRecording = true;
            this.recordingStartTime = Date.now();
            
            this.mediaRecorder.ondataavailable = (event) => {
                if (event.data.size > 0) {
                    this.recordedChunks.push(event.data);
                }
            };
            
            this.mediaRecorder.onstop = async () => {
                const blob = new Blob(this.recordedChunks, { type: 'audio/webm' });
                await this.sendMediaMessage(blob, 'audio');
                
                // Parar stream
                stream.getTracks().forEach(track => track.stop());
            };
            
            this.mediaRecorder.start();
            
            // Atualizar UI
            document.getElementById('recordButton').classList.add('recording');
            this.startRecordingTimer();
            
        } catch (error) {
            console.error('Erro ao iniciar gravação:', error);
            alert('Erro ao acessar o microfone');
        }
    }
    
    stopRecording() {
        if (!this.isRecording || !this.mediaRecorder) return;
        
        this.mediaRecorder.stop();
        this.isRecording = false;
        
        // Atualizar UI
        document.getElementById('recordButton').classList.remove('recording');
        this.stopRecordingTimer();
    }
    
    startRecordingTimer() {
        const timerElement = document.getElementById('recordingTimer');
        if (!timerElement) return;
        
        timerElement.style.display = 'block';
        
        const updateTimer = () => {
            if (!this.isRecording) return;
            
            const elapsed = Date.now() - this.recordingStartTime;
            const seconds = Math.floor(elapsed / 1000);
            const minutes = Math.floor(seconds / 60);
            
            timerElement.textContent = `${minutes}:${(seconds % 60).toString().padStart(2, '0')}`;
            
            setTimeout(updateTimer, 1000);
        };
        
        updateTimer();
    }
    
    stopRecordingTimer() {
        const timerElement = document.getElementById('recordingTimer');
        if (timerElement) {
            timerElement.style.display = 'none';
        }
    }
    
    // ===== ENVIO DE MÍDIA =====
    
    async sendMediaMessage(file, type) {
        if (!this.currentChat) return;
        
        const formData = new FormData();
        formData.append('file', file);
        formData.append('type', type);
        formData.append('chatId', this.currentChat);
        
        const message = {
            id: this.generateId(),
            type: type,
            sender: 'me',
            timestamp: new Date(),
            status: 'sending'
        };
        
        if (type === 'document') {
            message.filename = file.name;
            message.size = this.formatFileSize(file.size);
        }
        
        this.messages.push(message);
        this.renderMessages();
        
        try {
            const response = await fetch('/api/upload-media', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            
            if (result.success) {
                message.url = result.url;
                message.status = 'sent';
                
                if (type === 'audio') {
                    message.duration = result.duration;
                }
                
                this.renderMessages();
                
                // Enviar via socket
                this.socket.emit('sendMessage', {
                    chatId: this.currentChat,
                    message: message
                });
            } else {
                throw new Error(result.error);
            }
            
        } catch (error) {
            console.error('Erro ao enviar mídia:', error);
            message.status = 'error';
            this.renderMessages();
        }
    }
    
    // ===== INTERFACE =====
    
    showContactsList() {
        document.getElementById('contactsScreen').style.display = 'block';
        document.getElementById('chatScreen').style.display = 'none';
        document.getElementById('backButton').style.display = 'none';
    }
    
    showChatScreen() {
        document.getElementById('contactsScreen').style.display = 'none';
        document.getElementById('chatScreen').style.display = 'block';
        document.getElementById('backButton').style.display = 'block';
    }
    
    toggleAttachmentMenu() {
        const menu = document.getElementById('attachmentMenu');
        menu.classList.toggle('show');
    }
    
    // ===== UTILITÁRIOS =====
    
    adjustTextareaHeight() {
        const textarea = document.getElementById('messageInput');
        if (!textarea) return;
        
        textarea.style.height = 'auto';
        textarea.style.height = Math.min(textarea.scrollHeight, 96) + 'px';
    }
    
    adjustViewportHeight() {
        // Ajustar altura do viewport para mobile
        const vh = window.innerHeight * 0.01;
        document.documentElement.style.setProperty('--vh', `${vh}px`);
    }
    
    handleTyping() {
        if (!this.currentChat) return;
        
        // Emitir evento de digitação
        this.socket.emit('typing', { chatId: this.currentChat });
        
        // Limpar timeout anterior
        if (this.typingTimeout) {
            clearTimeout(this.typingTimeout);
        }
        
        // Parar de digitar após 3 segundos
        this.typingTimeout = setTimeout(() => {
            this.socket.emit('stopTyping', { chatId: this.currentChat });
        }, 3000);
    }
    
    handleNewMessage(message) {
        if (message.chatId === this.currentChat) {
            this.messages.push(message);
            this.renderMessages();
        } else {
            // Atualizar contador de não lidas
            const contact = this.contacts.find(c => c.id === message.chatId);
            if (contact) {
                contact.unread++;
                contact.lastMessage = message.text || 'Mídia';
                contact.timestamp = new Date(message.timestamp);
                this.renderContacts();
            }
        }
    }
    
    updateMessageStatus(data) {
        const message = this.messages.find(m => m.id === data.messageId);
        if (message) {
            message.status = data.status;
            this.renderMessages();
        }
    }
    
    showTypingIndicator(data) {
        if (data.chatId === this.currentChat) {
            // Mostrar indicador de digitação
            const indicator = document.getElementById('typingIndicator');
            if (indicator) {
                indicator.style.display = 'block';
            }
        }
    }
    
    hideTypingIndicator(data) {
        if (data.chatId === this.currentChat) {
            // Esconder indicador de digitação
            const indicator = document.getElementById('typingIndicator');
            if (indicator) {
                indicator.style.display = 'none';
            }
        }
    }
    
    formatTime(timestamp) {
        const date = new Date(timestamp);
        const now = new Date();
        
        if (date.toDateString() === now.toDateString()) {
            return date.toLocaleTimeString('pt-BR', { 
                hour: '2-digit', 
                minute: '2-digit' 
            });
        } else {
            return date.toLocaleDateString('pt-BR', { 
                day: '2-digit', 
                month: '2-digit' 
            });
        }
    }
    
    formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }
    
    getStatusIcon(status) {
        switch (status) {
            case 'sending': return '🕐';
            case 'sent': return '✓';
            case 'delivered': return '✓✓';
            case 'read': return '✓✓';
            case 'error': return '❌';
            default: return '';
        }
    }
    
    generateId() {
        return Date.now().toString(36) + Math.random().toString(36).substr(2);
    }
}

// Inicializar quando o DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => {
    window.mobileWhatsApp = new MobileWhatsApp();
});

// Adicionar estilos CSS dinamicamente
const mobileStylesElement = document.createElement('style');
mobileStylesElement.textContent = `
    .recording-interface {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.9);
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        z-index: 1000;
    }
    
    .recording-controls {
        display: flex;
        gap: 10px;
        margin-top: 20px;
    }
    
    .recording-controls button {
        padding: 10px 20px;
        border: none;
        border-radius: 5px;
        cursor: pointer;
        font-size: 16px;
    }
    
    .record-btn {
        background: #ff4444;
        color: white;
    }
    
    .stop-btn {
        background: #666;
        color: white;
    }
    
    .cancel-btn {
        background: #ccc;
        color: black;
    }
    
    .recording-controls button:disabled {
        opacity: 0.5;
        cursor: not-allowed;
    }
    
    #recordButton.recording {
        background: #ff4444 !important;
        animation: pulse 1s infinite;
    }
    
    @keyframes pulse {
        0% { opacity: 1; }
        50% { opacity: 0.5; }
        100% { opacity: 1; }
    }
    
    #recordingTimer {
        position: absolute;
        top: 10px;
        right: 10px;
        background: rgba(0, 0, 0, 0.7);
        color: white;
        padding: 5px 10px;
        border-radius: 15px;
        font-size: 12px;
        display: none;
    }
`;
document.head.appendChild(mobileStylesElement);
