// ===================================================================
//  INTERNATIONALIZATION SYSTEM (i18n)
// ===================================================================

const i18n = {
  currentLanguage: "pt",
  translations: {
    pt: {
      // Menu
      "menu.dashboard": "Dashboard",
      "menu.conversations": "Conversas",
      "menu.stores": "Lojas",
      "menu.departments": "Setores",
      "menu.users": "Usuários",
      "menu.contacts": "Contatos",
      "menu.faqs": "FAQs",
      "menu.payments": "Pagamentos",
      "menu.ai_config": "Configuração IA",
      "menu.translations": "Traduções",
      "menu.profile": "Editar Perfil",
      "menu.edit_profile": "Editar Perfil",

      // Botões gerais
      "btn.logout": "Sair",
      "btn.save": "Salvar",
      "btn.cancel": "Cancelar",
      "btn.edit": "Editar",
      "btn.delete": "Excluir",
      "btn.add": "Adicionar",
      "btn.close": "Fechar",
      "btn.confirm": "Confirmar",
      "btn.back": "Voltar",
      "btn.next": "Próximo",
      "btn.previous": "Anterior",
      "btn.finish": "Finalizar",
      "btn.search": "Buscar",
      "btn.filter": "Filtrar",
      "btn.export": "Exportar",
      "btn.import": "Importar",
      "btn.refresh": "Atualizar",
      "btn.yes": "Sim",
      "btn.no": "Não",

      // Dashboard
      "dashboard.title": "Dashboard",
      "dashboard.learn_tour": "Aprender - Tour Guiado",
      "dashboard.toggle_bot": "Ativar/Desativar Bot",
      "dashboard.toggle_groups": "Ativar/Desativar Grupos",
      "dashboard.pause_bot": "Pausar Bot",
      "dashboard.pause_groups": "Pausar Grupos",
      "dashboard.activate_bot": "Ativar Bot",
      "dashboard.activate_groups": "Ativar Grupos",
      "dashboard.refresh_qr": "Atualizar QR",
      "dashboard.disconnect_whatsapp": "Desconectar WhatsApp",
      "dashboard.messages_today": "Mensagens Hoje",
      "dashboard.messages": "Mensagens",
      "dashboard.unique_contacts": "Contatos Únicos",
      "dashboard.waiting_conversations": "Aguardando Atendimento",
      "dashboard.active_conversations": "Em Atendimento",
      "dashboard.messages_per_hour": "Mensagens por Hora",
      "dashboard.whatsapp_status": "Status do WhatsApp",
      "dashboard.checking": "Verificando...",
      "dashboard.loading_connection": "Carregando status da conexão...",
      "dashboard.whatsapp_connected": "✅ WhatsApp Conectado",
      "dashboard.whatsapp_disconnected": "⚠️ WhatsApp Desconectado",
      "dashboard.bot_ready": "Seu bot está pronto para uso",
      "dashboard.bot_operational":
        "Bot está operacional e pronto para receber mensagens",
      "dashboard.connected": "Conectado",

      // Usuários
      "users.title": "Gerenciar Usuários",
      "users.add": "Adicionar Usuário",
      "users.edit": "Editar Usuário",
      "users.search": "Buscar usuários...",
      "users.all_roles": "Todos os papéis",
      "users.role_admin": "Admin",
      "users.role_operator": "Operador",
      "users.role_viewer": "Visualizador",
      "users.name": "Nome",
      "users.role": "Papel",
      "users.store": "Loja",
      "users.department": "Setor",
      "users.created_at": "Criado em",
      "users.actions": "Ações",
      "users.active": "Ativo",

      // Conversas
      "conversations.title": "Conversas",
      "conversations.all_status": "Todos os status",
      "conversations.status_waiting": "Aguardando",
      "conversations.status_attended": "Em atendimento",
      "conversations.status_closed": "Finalizada",
      "conversations.all_stores": "Todas as lojas",
      "conversations.count": "conversas",
      "conversations.select": "Selecione uma conversa",
      "conversations.select_message":
        "Selecione uma conversa para visualizar as mensagens",
      "conversations.read_only":
        "Modo somente leitura - Admin não pode responder",
      "conversations.view_mode": "Modo visualização - Apenas leitura",
      "conversations.attended_by": "Atendido por:",

      // Lojas
      "stores.title": "Gerenciar Lojas",
      "stores.add": "Adicionar Loja",
      "stores.edit": "Editar Loja",
      "stores.name": "Nome",
      "stores.description": "Descrição",
      "stores.active": "Ativa",
      "stores.users": "Usuários",
      "stores.created_at": "Criada em",
      "stores.actions": "Ações",

      // Setores
      "departments.title": "Gerenciar Setores",
      "departments.add": "Adicionar Setor",
      "departments.edit": "Editar Setor",
      "departments.name": "Nome",
      "departments.description": "Descrição",
      "departments.users": "Usuários",
      "departments.created_at": "Criado em",
      "departments.actions": "Ações",
      "departments.active": "Ativo",

      // FAQs
      "faqs.title": "Gerenciar FAQs",
      "faqs.add": "Adicionar FAQ",
      "faqs.search": "Buscar FAQs...",
      "faqs.all_categories": "Todas as categorias",
      "faqs.category_general": "Geral",
      "faqs.category_products": "Produtos",
      "faqs.category_payments": "Pagamentos",
      "faqs.category_delivery": "Entrega",
      "faqs.question": "Pergunta",
      "faqs.answer": "Resposta",
      "faqs.emoji": "Emoji",
      "faqs.key": "Chave",
      "faqs.actions": "Ações",
      "faqs.welcome_title": "Mensagens de Boas-Vindas",
      "faqs.welcome_desc": "Configure as mensagens que serão enviadas quando um cliente iniciar uma conversa (apenas 1x por dia)",
      "faqs.welcome_msg1": "Mensagem 1",
      "faqs.welcome_msg1_placeholder": "Digite a primeira mensagem...",
      "faqs.welcome_msg2": "Mensagem 2",
      "faqs.welcome_msg2_placeholder": "Digite a segunda mensagem...",
      "faqs.welcome_msg3": "Mensagem 3 (com mídia opcional)",
      "faqs.welcome_msg3_placeholder": "Digite a terceira mensagem...",
      "faqs.welcome_placeholders": "Placeholders:",
      "faqs.welcome_media": "Mídia (opcional):",
      "faqs.welcome_save": "Salvar Mensagens de Boas-Vindas",
      "faqs.welcome_preview": "Preview Mobile",
      "faqs.welcome_preview_empty": "As mensagens aparecerão aqui conforme você digita",
      "faqs.reaction_time": "Tempo de Reação (segundos)",
      "faqs.response_time": "Tempo de Resposta (segundos)",
      "faqs.schedule_hours": "Horário de Funcionamento",
      "faqs.schedule_days": "Dias da Semana",
      "faqs.order_position": "Ordem de Exibição",
      "faqs.test_faq": "Testar FAQ",
      "faqs.no_messages": "Nenhuma mensagem configurada",
      "faqs.message_saved": "Mensagens de boas-vindas salvas com sucesso!",
      "faqs.message_error": "Erro ao salvar mensagens de boas-vindas",
      "faqs.faq_saved": "FAQ salvo com sucesso!",
      "faqs.faq_error": "Erro ao salvar FAQ",
      "faqs.faq_deleted": "FAQ excluído com sucesso!",
      "faqs.faq_delete_error": "Erro ao excluir FAQ",
      "faqs.confirm_delete": "Tem certeza que deseja excluir este FAQ?",

      // Pagamentos
      "payments.title": "Gerenciar Pagamentos",
      "payments.config_gateways": "Configurar Gateways",
      "payments.today": "Pagamentos Hoje",
      "payments.total_amount": "Valor Total",
      "payments.pending": "Pendentes",
      "payments.success_rate": "Taxa de Sucesso",
      "payments.all_status": "Todos os status",
      "payments.status_pending": "Pendente",
      "payments.status_completed": "Concluído",
      "payments.status_failed": "Falhou",
      "payments.status_cancelled": "Cancelado",
      "payments.id": "ID",
      "payments.customer": "Cliente",
      "payments.amount": "Valor",
      "payments.gateway": "Gateway",
      "payments.status": "Status",
      "payments.date": "Data",
      "payments.actions": "Ações",
      "payments.loading": "Carregando pagamentos...",

      // Traduções
      "translations.title": "Gerenciar Traduções",
      "translations.import": "Importar",
      "translations.export": "Exportar",
      "translations.add_language": "Adicionar Idioma",
      "translations.apply_system": "Aplicar ao Sistema",
      "translations.active_language": "Idioma Ativo:",
      "translations.total_strings": "Total de Strings",
      "translations.translated_strings": "Strings Traduzidas",
      "translations.completion_rate": "Taxa de Completude",
      "translations.tab_languages": "Idiomas",
      "translations.tab_strings": "Strings",
      "translations.code": "Código",
      "translations.name": "Nome",
      "translations.strings": "Strings",
      "translations.completeness": "Completude",
      "translations.default": "Padrão",
      "translations.actions": "Ações",
      "translations.search": "Buscar strings...",
      "translations.all_categories": "Todas as categorias",
      "translations.add_string": "Adicionar String",
      "translations.key": "Chave",
      "translations.value": "Valor",
      "translations.category": "Categoria",
      "translations.edit_string": "Editar String",
      "translations.language": "Idioma",
      "translations.language_code": "Código do Idioma",
      "translations.language_name": "Nome do Idioma",
      "translations.copy_from_default": "Copiar strings do idioma padrão",
      "translations.set_default": "Definir como Padrão",

      // Configuração IA
      "ai_config.title": "Configuração de IA",
      "ai_config.test_connection": "Testar Conexão",
      "ai_config.deepseek": "DeepSeek",
      "ai_config.openai": "OpenAI",
      "ai_config.api_key": "API Key:",
      "ai_config.api_key_placeholder": "Sua chave da API",
      "ai_config.persona_name": "Nome da Persona:",
      "ai_config.persona_name_placeholder": "Assistente IA",
      "ai_config.persona_description": "Descrição da Persona:",
      "ai_config.persona_description_placeholder": "Sou um assistente virtual inteligente...",
      "ai_config.working_hours": "Horário de Funcionamento:",
      "ai_config.working_hours_placeholder": "00:00-23:59",
      "ai_config.model": "Modelo:",
      "ai_config.temperature": "Temperatura:",
      "ai_config.temperature_desc": "Controla a criatividade das respostas (0.0 = mais preciso, 2.0 = mais criativo)",
      "ai_config.max_tokens": "Máximo de Tokens:",
      "ai_config.max_tokens_desc": "Limite de tokens para as respostas da IA",
      "ai_config.save_deepseek": "Salvar DeepSeek",
      "ai_config.save_openai": "Salvar OpenAI",
      "ai_config.loading": "Carregando configurações de IA...",

      // Contatos
      "contacts.title": "Gerenciar Contatos",
      "contacts.add": "Adicionar Contato",
      "contacts.tab_contacts": "Contatos",
      "contacts.tab_groups": "Grupos",
      "contacts.search": "Buscar contatos...",
      "contacts.all_groups": "Todos os grupos",
      "contacts.import_csv": "Importar CSV",
      "contacts.name": "Nome",
      "contacts.phone": "Telefone",
      "contacts.email": "Email",
      "contacts.group": "Grupo",
      "contacts.tags": "Tags",
      "contacts.created_at": "Criado em",
      "contacts.actions": "Ações",
      "contacts.new_group": "Novo Grupo",
      "contacts.group_name": "Nome do Grupo",
      "contacts.group_description": "Descrição",
      "contacts.group_contacts": "Contatos",
      "contacts.loading": "Carregando contatos...",

      // App do Usuário
      "user.no_conversations": "Nenhuma conversa ainda",
      "user.no_messages": "Sem mensagens",
      "user.no_messages_conversation": "Nenhuma mensagem nesta conversa",
      "user.no_conversation_selected": "Nenhuma conversa selecionada",
      "user.media_photo": "📷 Foto",
      "user.media_video": "🎥 Vídeo",
      "user.media_audio": "🎤 Áudio",
      "user.media_document": "📄 Documento",
      "user.media_location": "📍 Localização",
      "user.media_message": "Mensagem",
      "user.theme_dark_activated": "Tema escuro ativado",
      "user.theme_light_activated": "Tema claro ativado",
      "user.conversation_transferred": "Conversa transferida para",
      "user.mass_send_development": "Funcionalidade de envio em massa em desenvolvimento",
      "user.contact_development": "Funcionalidade de contato em desenvolvimento",
      "user.error_init_app": "Erro ao inicializar aplicação",
      "user.error_load_conversations": "Erro ao carregar conversas",
      "user.error_open_conversation": "Erro ao abrir conversa",
      "user.error_load_messages": "Erro ao carregar mensagens",
      "user.error_send_message": "Erro ao enviar mensagem",
      "user.error_send_file": "Erro ao enviar arquivo",
      "user.error_send_media": "Erro ao enviar mídia",
      "user.error_send_audio": "Erro ao enviar áudio",
      "user.error_send_location": "Erro ao enviar localização",
      "user.error_microphone_access": "Erro ao acessar microfone",
      "user.error_location_not_found": "Localização não encontrada",
      "user.error_get_location": "Erro ao buscar localização",
      "user.error_load_stores": "Erro ao carregar lista de lojas",
      "user.error_load_departments": "Erro ao carregar lista de departamentos",
      "user.error_transfer_conversation": "Erro ao transferir conversa",
      "user.error_load_contacts": "Erro ao carregar contatos",
      "user.select_location": "Selecione uma localização",
      "user.select_conversation_first": "Selecione uma conversa primeiro",
      "user.select_store": "Selecione uma loja",
      "user.select_department": "Selecione um departamento",
      "user.fill_required_fields": "Preencha todos os campos obrigatórios",
      "user.customer_phone_not_found": "Telefone do cliente não encontrado",
      "user.payment_link_created": "Link de pagamento criado e enviado!",
      "user.error_create_payment_link": "Erro ao criar link de pagamento",
      "user.session_expired": "Sessão expirada. Faça login novamente.",
      "user.conversation_transferred_success": "Conversa transferida com sucesso",
      "user.conversation_transferred_support_success": "Conversa transferida para suporte com sucesso",
      "user.recording_cancelled": "Gravação cancelada",
      "user.sending_audio": "Enviando áudio...",
      "user.audio_sent": "Áudio enviado!",
      "user.location_sent": "Localização enviada",
      "user.no_contacts_registered": "Nenhum contato cadastrado",
      "user.access_denied_admin_only": "Acesso negado. Apenas administradores podem visualizar contatos.",
      "user.conversation_started_with": "Conversa iniciada com",
      "user.error_open_conversation_contact": "Erro ao abrir conversa",
      "user.placeholder_message": "Mensagem",
      "user.placeholder_search_conversations": "Buscar conversas...",
      "user.placeholder_full_name": "Nome completo",
      "user.placeholder_payment_description": "Descrição do pagamento",
      "user.placeholder_address_location": "Digite um endereço ou local...",
      "user.placeholder_search_name_number": "Buscar por nome ou número...",
      "user.title_search": "Buscar",
      "user.title_toggle_theme": "Alternar tema",
      "user.title_conversation_menu": "Menu da conversa",
      "user.title_menu": "Menu",
      "user.title_cancel": "Cancelar",
      "user.title_pause": "Pausar",
      "user.title_continue": "Continuar",
      "user.title_send": "Enviar",
      "user.title_view_contacts": "Ver Contatos",
      "user.transfer_to_store": "Transferir para Loja",
      "user.transfer_to_support": "Transferir para Suporte",
      "user.logout_title": "Confirmar Saída",
      "user.logout_message": "Tem certeza que deseja sair?",
      "user.select_store_message": "Selecione a loja para onde deseja transferir esta conversa:",
      "user.select_department_message": "Selecione o departamento de suporte para onde deseja transferir esta conversa:",
      "user.store": "Loja",
      "user.department": "Departamento",
      "user.select_option": "Selecione...",
      "user.attention": "Atenção:",
      "user.transfer_store_warning": "Após a transferência, esta conversa será movida para a loja selecionada.",
      "user.transfer_support_warning": "Após a transferência, esta conversa será movida para o departamento selecionado e você não terá mais acesso a ela.",
      "user.confirm_transfer": "Confirmar Transferência",
      "user.placeholder_search_name_number": "Buscar por nome ou número...",
      "user.gallery": "Galeria",
      "user.camera": "Câmera",
      "user.document": "Documento",
      "user.audio": "Áudio",
      "user.location": "Localização",
      "user.payment": "Pagamento",

      // Editar Perfil
      "profile.title": "Editar Perfil",
      "profile.system_name": "Nome do Sistema",
      "profile.system_name_placeholder": "Ex: BeZap Admin",
      "profile.color_customization": "Personalização de Cores",
      "profile.menu_colors": "Cores do Menu",
      "profile.menu_background": "Fundo do Menu",
      "profile.menu_active": "Item Ativo",
      "profile.menu_hover": "Hover do Menu",
      "profile.card_colors": "Cores dos Cards",
      "profile.card_background": "Fundo dos Cards",
      "profile.card_border": "Borda dos Cards",
      "profile.text_colors": "Cores dos Textos",
      "profile.primary_color": "Cor Primária",
      "profile.secondary_color": "Cor Secundária",
      "profile.title_color": "Cor dos Títulos",
      "profile.text_color": "Cor do Texto",
      "profile.button_colors": "Cores dos Botões",
      "profile.primary_button": "Botão Primário",
      "profile.secondary_button": "Botão Secundário",
      "profile.success_button": "Botão de Sucesso",
      "profile.danger_button": "Botão de Perigo",
      "profile.warning_button": "Botão de Aviso",
      "profile.preview": "Preview",
      "profile.reset_colors": "Restaurar Cores Padrão",
      "profile.save_changes": "Salvar Alterações",
      "profile.system_logos": "Logos do Sistema",
      "profile.login_logo": "Logo para Tela de Login",
      "profile.system_logo": "Logo para Sistema Logado",
      "profile.upload_logo": "Upload Logo",
      "profile.remove_logo": "Remover Logo",
      "profile.preview_login": "Preview Tela de Login",
      "profile.crop_image": "Recortar Imagem",
      "profile.max_size": "Tamanho máximo:",
      "profile.favicon": "Favicon",
      "profile.upload_favicon": "Upload Favicon",

      // Logout
      "logout.confirm_message": "Tem certeza que deseja sair?",

      // Mensagens de notificação
      "notification.success": "Operação realizada com sucesso!",
      "notification.error": "Ocorreu um erro. Tente novamente.",
      "notification.saved": "Salvo com sucesso!",
      "notification.deleted": "Excluído com sucesso!",
      "notification.updated": "Atualizado com sucesso!",
      "notification.unauthorized": "Não autorizado",
      "notification.bot_activated": "Bot ativado",
      "notification.bot_paused": "Bot pausado",
      "notification.groups_activated": "Respostas em grupos ativadas",
      "notification.groups_paused": "Respostas em grupos pausadas",
      "notification.connected_server": "Conectado ao servidor",
      "notification.reconnected_server": "Reconectado ao servidor",
      "notification.connection_lost": "Conexão com servidor perdida",
      "notification.connection_error": "Erro de conexão com servidor",
      "notification.qr_updating": "QR Code será atualizado em instantes",
      "notification.testing_connection": "Testando conexão WhatsApp...",
      "notification.whatsapp_working": "WhatsApp está conectado e funcionando!",
      "notification.whatsapp_not_connected":
        "WhatsApp não está conectado. Verifique o QR Code.",
      "notification.test_connection_error": "Erro ao testar conexão WhatsApp",
      "notification.updating_qr": "Atualizando QR Code...",
      "notification.qr_error": "Erro ao atualizar QR Code",
      "notification.disconnecting_whatsapp": "Desconectando WhatsApp...",
      "notification.whatsapp_disconnected_success":
        "WhatsApp desconectado com sucesso",
      "notification.whatsapp_disconnect_error": "Erro ao desconectar WhatsApp",
      "notification.cleaning_session": "Limpando sessão WhatsApp...",
      "notification.session_cleaned": "Sessão limpa. Novo QR Code será gerado.",
      "notification.reset_session_error": "Erro ao resetar sessão WhatsApp",
      "notification.bot_status_error": "Erro ao alterar status do bot",
      "notification.groups_status_error": "Erro ao alterar status dos grupos",
      "notification.users_load_error": "Erro ao carregar usuários",
      "notification.stores_load_error": "Erro ao carregar lojas",
      "notification.departments_load_error": "Erro ao carregar setores",
      "notification.username_required": "Nome de usuário é obrigatório",
      "notification.password_required": "Senha é obrigatória",
      "notification.password_min_length":
        "Senha deve ter pelo menos 6 caracteres",
      "notification.store_required": "Selecione uma loja",
      "notification.user_created": "Usuário criado com sucesso",
      "notification.user_create_error": "Erro ao criar usuário",
      "notification.user_connection_error": "Erro de conexão ao criar usuário",
      "notification.faq_fields_required":
        "Pergunta e resposta são obrigatórias",
      "notification.faq_created": "FAQ criado com sucesso",
      "notification.faq_create_error": "Erro ao criar FAQ",
      "notification.store_created": "Loja criada com sucesso",
      "notification.store_create_error": "Erro ao criar loja",
      "notification.store_deleted": "Loja excluída com sucesso",
      "notification.store_delete_error": "Erro ao excluir loja",
      "notification.department_created": "Setor criado com sucesso",
      "notification.department_create_error": "Erro ao criar setor",
      "notification.department_deleted": "Setor excluído com sucesso",
      "notification.department_delete_error": "Erro ao excluir setor",
      "notification.user_deleted": "Usuário excluído com sucesso",
      "notification.user_delete_error": "Erro ao excluir usuário",
      "notification.page_load_error": "Erro ao carregar página",
      "notification.dashboard_load_error": "Erro ao carregar dashboard",
      "notification.conversation_attended": "Conversa atendida por",
      "notification.confirm_delete_store":
        "Tem certeza que deseja excluir esta loja?",
      "notification.confirm_delete_department":
        "Tem certeza que deseja excluir este setor?",
      "notification.confirm_delete_user":
        "Tem certeza que deseja excluir este usuário?",

      // Tour guiado
      "tour.welcome": "Bem-vindo ao Painel Administrativo!",
      "tour.dashboard_desc":
        "Este é o seu painel de controle completo do BeZap.",
      "tour.step": "de",
      "tour.previous": "Anterior",
      "tour.next": "Próximo",
      "tour.finish": "Finalizar",
    },
    en: {
      // Menu
      "menu.dashboard": "Dashboard",
      "menu.conversations": "Conversations",
      "menu.stores": "Stores",
      "menu.departments": "Departments",
      "menu.users": "Users",
      "menu.contacts": "Contacts",
      "menu.faqs": "FAQs",
      "menu.payments": "Payments",
      "menu.ai_config": "AI Configuration",
      "menu.translations": "Translations",
      "menu.profile": "Edit Profile",
      "menu.edit_profile": "Edit Profile",

      // General buttons
      "btn.logout": "Logout",
      "btn.save": "Save",
      "btn.cancel": "Cancel",
      "btn.edit": "Edit",
      "btn.delete": "Delete",
      "btn.add": "Add",
      "btn.close": "Close",
      "btn.confirm": "Confirm",
      "btn.back": "Back",
      "btn.next": "Next",
      "btn.previous": "Previous",
      "btn.finish": "Finish",
      "btn.search": "Search",
      "btn.filter": "Filter",
      "btn.export": "Export",
      "btn.import": "Import",
      "btn.refresh": "Refresh",
      "btn.yes": "Yes",
      "btn.no": "No",

      // Dashboard
      "dashboard.title": "Dashboard",
      "dashboard.learn_tour": "Learn - Guided Tour",
      "dashboard.toggle_bot": "Toggle Bot",
      "dashboard.toggle_groups": "Toggle Groups",
      "dashboard.pause_bot": "Pause Bot",
      "dashboard.pause_groups": "Pause Groups",
      "dashboard.activate_bot": "Activate Bot",
      "dashboard.activate_groups": "Activate Groups",
      "dashboard.refresh_qr": "Refresh QR",
      "dashboard.disconnect_whatsapp": "Disconnect WhatsApp",
      "dashboard.messages_today": "Messages Today",
      "dashboard.messages": "Messages",
      "dashboard.unique_contacts": "Unique Contacts",
      "dashboard.waiting_conversations": "Waiting for Service",
      "dashboard.active_conversations": "In Service",
      "dashboard.messages_per_hour": "Messages per Hour",
      "dashboard.whatsapp_status": "WhatsApp Status",
      "dashboard.checking": "Checking...",
      "dashboard.loading_connection": "Loading connection status...",
      "dashboard.whatsapp_connected": "✅ WhatsApp Connected",
      "dashboard.whatsapp_disconnected": "⚠️ WhatsApp Disconnected",
      "dashboard.bot_ready": "Your bot is ready to use",
      "dashboard.bot_operational":
        "Bot is operational and ready to receive messages",
      "dashboard.connected": "Connected",

      // Users
      "users.title": "Manage Users",
      "users.add": "Add User",
      "users.search": "Search users...",
      "users.all_roles": "All Roles",
      "users.role_admin": "Admin",
      "users.role_operator": "Operator",
      "users.role_viewer": "Viewer",
      "users.name": "Name",
      "users.role": "Role",
      "users.store": "Store",
      "users.department": "Department",
      "users.created_at": "Created At",
      "users.actions": "Actions",
      "users.edit": "Edit User",
      "users.active": "Active",
      "users.edit": "Edit User",
      "users.active": "Active",

      // Conversations
      "conversations.title": "Conversations",
      "conversations.all_status": "All Status",
      "conversations.status_waiting": "Waiting",
      "conversations.status_attended": "In Service",
      "conversations.status_closed": "Closed",
      "conversations.all_stores": "All Stores",
      "conversations.count": "conversations",
      "conversations.select": "Select a conversation",
      "conversations.select_message": "Select a conversation to view messages",
      "conversations.read_only": "Read-only mode - Admin cannot respond",
      "conversations.view_mode": "View mode - Read only",
      "conversations.attended_by": "Attended by:",

      // Stores
      "stores.title": "Manage Stores",
      "stores.add": "Add Store",
      "stores.edit": "Edit Store",
      "stores.name": "Name",
      "stores.description": "Description",
      "stores.active": "Active",
      "stores.users": "Users",
      "stores.created_at": "Created At",
      "stores.actions": "Actions",

      // Departments
      "departments.title": "Manage Departments",
      "departments.add": "Add Department",
      "departments.name": "Name",
      "departments.description": "Description",
      "departments.users": "Users",
      "departments.created_at": "Created At",
      "departments.actions": "Actions",
      "departments.edit": "Edit Department",
      "departments.active": "Active",
      "departments.edit": "Edit Department",
      "departments.active": "Active",

      // FAQs
      "faqs.title": "Manage FAQs",
      "faqs.add": "Add FAQ",
      "faqs.search": "Search FAQs...",
      "faqs.all_categories": "All Categories",
      "faqs.category_general": "General",
      "faqs.category_products": "Products",
      "faqs.category_payments": "Payments",
      "faqs.category_delivery": "Delivery",
      "faqs.question": "Question",
      "faqs.answer": "Answer",
      "faqs.emoji": "Emoji",
      "faqs.key": "Key",
      "faqs.actions": "Actions",
      "faqs.welcome_title": "Welcome Messages",
      "faqs.welcome_desc": "Configure messages that will be sent when a customer starts a conversation (only once per day)",
      "faqs.welcome_msg1": "Message 1",
      "faqs.welcome_msg1_placeholder": "Type the first message...",
      "faqs.welcome_msg2": "Message 2",
      "faqs.welcome_msg2_placeholder": "Type the second message...",
      "faqs.welcome_msg3": "Message 3 (with optional media)",
      "faqs.welcome_msg3_placeholder": "Type the third message...",
      "faqs.welcome_placeholders": "Placeholders:",
      "faqs.welcome_media": "Media (optional):",
      "faqs.welcome_save": "Save Welcome Messages",
      "faqs.welcome_preview": "Mobile Preview",
      "faqs.welcome_preview_empty": "Messages will appear here as you type",
      "faqs.reaction_time": "Reaction Time (seconds)",
      "faqs.response_time": "Response Time (seconds)",
      "faqs.schedule_hours": "Working Hours",
      "faqs.schedule_days": "Weekdays",
      "faqs.order_position": "Display Order",
      "faqs.test_faq": "Test FAQ",
      "faqs.no_messages": "No messages configured",
      "faqs.message_saved": "Welcome messages saved successfully!",
      "faqs.message_error": "Error saving welcome messages",
      "faqs.faq_saved": "FAQ saved successfully!",
      "faqs.faq_error": "Error saving FAQ",
      "faqs.faq_deleted": "FAQ deleted successfully!",
      "faqs.faq_delete_error": "Error deleting FAQ",
      "faqs.confirm_delete": "Are you sure you want to delete this FAQ?",

      // Payments
      "payments.title": "Manage Payments",
      "payments.config_gateways": "Configure Gateways",
      "payments.today": "Payments Today",
      "payments.total_amount": "Total Amount",
      "payments.pending": "Pending",
      "payments.success_rate": "Success Rate",
      "payments.all_status": "All Status",
      "payments.status_pending": "Pending",
      "payments.status_completed": "Completed",
      "payments.status_failed": "Failed",
      "payments.status_cancelled": "Cancelled",
      "payments.id": "ID",
      "payments.customer": "Customer",
      "payments.amount": "Amount",
      "payments.gateway": "Gateway",
      "payments.status": "Status",
      "payments.date": "Date",
      "payments.actions": "Actions",
      "payments.loading": "Loading payments...",

      // Translations
      "translations.title": "Manage Translations",
      "translations.import": "Import",
      "translations.export": "Export",
      "translations.add_language": "Add Language",
      "translations.apply_system": "Apply to System",
      "translations.active_language": "Active Language:",
      "translations.total_strings": "Total Strings",
      "translations.translated_strings": "Translated Strings",
      "translations.completion_rate": "Completion Rate",
      "translations.tab_languages": "Languages",
      "translations.tab_strings": "Strings",
      "translations.code": "Code",
      "translations.name": "Name",
      "translations.strings": "Strings",
      "translations.completeness": "Completeness",
      "translations.default": "Default",
      "translations.actions": "Actions",
      "translations.search": "Search strings...",
      "translations.all_categories": "All Categories",
      "translations.add_string": "Add String",
      "translations.key": "Key",
      "translations.value": "Value",
      "translations.category": "Category",
      "translations.edit_string": "Edit String",
      "translations.language": "Language",
      "translations.language_code": "Language Code",
      "translations.language_name": "Language Name",
      "translations.copy_from_default": "Copy strings from default language",
      "translations.set_default": "Set as Default",

      // AI Configuration
      "ai_config.title": "AI Configuration",
      "ai_config.test_connection": "Test Connection",
      "ai_config.deepseek": "DeepSeek",
      "ai_config.openai": "OpenAI",
      "ai_config.api_key": "API Key:",
      "ai_config.api_key_placeholder": "Your API key",
      "ai_config.persona_name": "Persona Name:",
      "ai_config.persona_name_placeholder": "AI Assistant",
      "ai_config.persona_description": "Persona Description:",
      "ai_config.persona_description_placeholder": "I am an intelligent virtual assistant...",
      "ai_config.working_hours": "Working Hours:",
      "ai_config.working_hours_placeholder": "00:00-23:59",
      "ai_config.model": "Model:",
      "ai_config.temperature": "Temperature:",
      "ai_config.temperature_desc": "Controls response creativity (0.0 = more precise, 2.0 = more creative)",
      "ai_config.max_tokens": "Maximum Tokens:",
      "ai_config.max_tokens_desc": "Token limit for AI responses",
      "ai_config.save_deepseek": "Save DeepSeek",
      "ai_config.save_openai": "Save OpenAI",
      "ai_config.loading": "Loading AI settings...",

      // Contacts
      "contacts.title": "Manage Contacts",
      "contacts.add": "Add Contact",
      "contacts.tab_contacts": "Contacts",
      "contacts.tab_groups": "Groups",
      "contacts.search": "Search contacts...",
      "contacts.all_groups": "All Groups",
      "contacts.import_csv": "Import CSV",
      "contacts.name": "Name",
      "contacts.phone": "Phone",
      "contacts.email": "Email",
      "contacts.group": "Group",
      "contacts.tags": "Tags",
      "contacts.created_at": "Created At",
      "contacts.actions": "Actions",
      "contacts.new_group": "New Group",
      "contacts.group_name": "Group Name",
      "contacts.group_description": "Description",
      "contacts.group_contacts": "Contacts",
      "contacts.loading": "Loading contacts...",

      // User App
      "user.no_conversations": "No conversations yet",
      "user.no_messages": "No messages",
      "user.no_messages_conversation": "No messages in this conversation",
      "user.no_conversation_selected": "No conversation selected",
      "user.media_photo": "📷 Photo",
      "user.media_video": "🎥 Video",
      "user.media_audio": "🎤 Audio",
      "user.media_document": "📄 Document",
      "user.media_location": "📍 Location",
      "user.media_message": "Message",
      "user.theme_dark_activated": "Dark theme activated",
      "user.theme_light_activated": "Light theme activated",
      "user.conversation_transferred": "Conversation transferred to",
      "user.mass_send_development": "Mass send feature in development",
      "user.contact_development": "Contact feature in development",
      "user.error_init_app": "Error initializing application",
      "user.error_load_conversations": "Error loading conversations",
      "user.error_open_conversation": "Error opening conversation",
      "user.error_load_messages": "Error loading messages",
      "user.error_send_message": "Error sending message",
      "user.error_send_file": "Error sending file",
      "user.error_send_media": "Error sending media",
      "user.error_send_audio": "Error sending audio",
      "user.error_send_location": "Error sending location",
      "user.error_microphone_access": "Error accessing microphone",
      "user.error_location_not_found": "Location not found",
      "user.error_get_location": "Error getting location",
      "user.error_load_stores": "Error loading stores list",
      "user.error_load_departments": "Error loading departments list",
      "user.error_transfer_conversation": "Error transferring conversation",
      "user.error_load_contacts": "Error loading contacts",
      "user.select_location": "Select a location",
      "user.select_conversation_first": "Select a conversation first",
      "user.select_store": "Select a store",
      "user.select_department": "Select a department",
      "user.fill_required_fields": "Fill in all required fields",
      "user.customer_phone_not_found": "Customer phone not found",
      "user.payment_link_created": "Payment link created and sent!",
      "user.error_create_payment_link": "Error creating payment link",
      "user.session_expired": "Session expired. Please login again.",
      "user.conversation_transferred_success": "Conversation transferred successfully",
      "user.conversation_transferred_support_success": "Conversation transferred to support successfully",
      "user.recording_cancelled": "Recording cancelled",
      "user.sending_audio": "Sending audio...",
      "user.audio_sent": "Audio sent!",
      "user.location_sent": "Location sent",
      "user.no_contacts_registered": "No contacts registered",
      "user.access_denied_admin_only": "Access denied. Only administrators can view contacts.",
      "user.conversation_started_with": "Conversation started with",
      "user.error_open_conversation_contact": "Error opening conversation",
      "user.placeholder_message": "Message",
      "user.placeholder_search_conversations": "Search conversations...",
      "user.placeholder_full_name": "Full name",
      "user.placeholder_payment_description": "Payment description",
      "user.placeholder_address_location": "Type an address or location...",
      "user.placeholder_search_name_number": "Search by name or number...",
      "user.title_search": "Search",
      "user.title_toggle_theme": "Toggle theme",
      "user.title_conversation_menu": "Conversation menu",
      "user.title_menu": "Menu",
      "user.title_cancel": "Cancel",
      "user.title_pause": "Pause",
      "user.title_continue": "Continue",
      "user.title_send": "Send",
      "user.title_view_contacts": "View Contacts",
      "user.transfer_to_store": "Transfer to Store",
      "user.transfer_to_support": "Transfer to Support",
      "user.logout_title": "Confirm Logout",
      "user.logout_message": "Are you sure you want to logout?",
      "user.select_store_message": "Select the store to transfer this conversation to:",
      "user.select_department_message": "Select the support department to transfer this conversation to:",
      "user.store": "Store",
      "user.department": "Department",
      "user.select_option": "Select...",
      "user.attention": "Attention:",
      "user.transfer_store_warning": "After the transfer, this conversation will be moved to the selected store.",
      "user.transfer_support_warning": "After the transfer, this conversation will be moved to the selected department and you will no longer have access to it.",
      "user.confirm_transfer": "Confirm Transfer",
      "user.placeholder_search_name_number": "Search by name or number...",
      "user.gallery": "Gallery",
      "user.camera": "Camera",
      "user.document": "Document",
      "user.audio": "Audio",
      "user.location": "Location",
      "user.payment": "Payment",

      // Edit Profile
      "profile.title": "Edit Profile",
      "profile.system_name": "System Name",
      "profile.system_name_placeholder": "Ex: BeZap Admin",
      "profile.color_customization": "Color Customization",
      "profile.menu_colors": "Menu Colors",
      "profile.menu_background": "Menu Background",
      "profile.menu_active": "Active Item",
      "profile.menu_hover": "Menu Hover",
      "profile.card_colors": "Card Colors",
      "profile.card_background": "Card Background",
      "profile.card_border": "Card Border",
      "profile.text_colors": "Text Colors",
      "profile.primary_color": "Primary Color",
      "profile.secondary_color": "Secondary Color",
      "profile.title_color": "Title Color",
      "profile.text_color": "Text Color",
      "profile.button_colors": "Button Colors",
      "profile.primary_button": "Primary Button",
      "profile.secondary_button": "Secondary Button",
      "profile.success_button": "Success Button",
      "profile.danger_button": "Danger Button",
      "profile.warning_button": "Warning Button",
      "profile.preview": "Preview",
      "profile.reset_colors": "Reset Default Colors",
      "profile.save_changes": "Save Changes",
      "profile.system_logos": "System Logos",
      "profile.login_logo": "Login Screen Logo",
      "profile.system_logo": "Logged System Logo",
      "profile.upload_logo": "Upload Logo",
      "profile.remove_logo": "Remove Logo",
      "profile.preview_login": "Preview Login Screen",
      "profile.crop_image": "Crop Image",
      "profile.max_size": "Maximum size:",
      "profile.favicon": "Favicon",
      "profile.upload_favicon": "Upload Favicon",

      // Logout
      "logout.confirm_message": "Are you sure you want to logout?",

      // Notification messages
      "notification.success": "Operation completed successfully!",
      "notification.error": "An error occurred. Please try again.",
      "notification.saved": "Saved successfully!",
      "notification.deleted": "Deleted successfully!",
      "notification.updated": "Updated successfully!",
      "notification.unauthorized": "Unauthorized",
      "notification.bot_activated": "Bot activated",
      "notification.bot_paused": "Bot paused",
      "notification.groups_activated": "Group responses activated",
      "notification.groups_paused": "Group responses paused",
      "notification.connected_server": "Connected to server",
      "notification.reconnected_server": "Reconnected to server",
      "notification.connection_lost": "Connection to server lost",
      "notification.connection_error": "Server connection error",
      "notification.qr_updating": "QR Code will be updated shortly",
      "notification.testing_connection": "Testing WhatsApp connection...",
      "notification.whatsapp_working": "WhatsApp is connected and working!",
      "notification.whatsapp_not_connected":
        "WhatsApp is not connected. Check the QR Code.",
      "notification.test_connection_error": "Error testing WhatsApp connection",
      "notification.updating_qr": "Updating QR Code...",
      "notification.qr_error": "Error updating QR Code",
      "notification.disconnecting_whatsapp": "Disconnecting WhatsApp...",
      "notification.whatsapp_disconnected_success":
        "WhatsApp disconnected successfully",
      "notification.whatsapp_disconnect_error": "Error disconnecting WhatsApp",
      "notification.cleaning_session": "Cleaning WhatsApp session...",
      "notification.session_cleaned":
        "Session cleaned. New QR Code will be generated.",
      "notification.reset_session_error": "Error resetting WhatsApp session",
      "notification.bot_status_error": "Error changing bot status",
      "notification.groups_status_error": "Error changing groups status",
      "notification.users_load_error": "Error loading users",
      "notification.stores_load_error": "Error loading stores",
      "notification.departments_load_error": "Error loading departments",
      "notification.username_required": "Username is required",
      "notification.password_required": "Password is required",
      "notification.password_min_length":
        "Password must be at least 6 characters",
      "notification.store_required": "Select a store",
      "notification.user_created": "User created successfully",
      "notification.user_create_error": "Error creating user",
      "notification.user_connection_error": "Connection error creating user",
      "notification.faq_fields_required": "Question and answer are required",
      "notification.faq_created": "FAQ created successfully",
      "notification.faq_create_error": "Error creating FAQ",
      "notification.store_created": "Store created successfully",
      "notification.store_create_error": "Error creating store",
      "notification.store_deleted": "Store deleted successfully",
      "notification.store_delete_error": "Error deleting store",
      "notification.department_created": "Department created successfully",
      "notification.department_create_error": "Error creating department",
      "notification.department_deleted": "Department deleted successfully",
      "notification.department_delete_error": "Error deleting department",
      "notification.user_deleted": "User deleted successfully",
      "notification.user_delete_error": "Error deleting user",
      "notification.page_load_error": "Error loading page",
      "notification.dashboard_load_error": "Error loading dashboard",
      "notification.conversation_attended": "Conversation attended by",
      "notification.confirm_delete_store":
        "Are you sure you want to delete this store?",
      "notification.confirm_delete_department":
        "Are you sure you want to delete this department?",
      "notification.confirm_delete_user":
        "Are you sure you want to delete this user?",

      // Guided tour
      "tour.welcome": "Welcome to the Admin Panel!",
      "tour.dashboard_desc": "This is your complete BeZap control panel.",
      "tour.step": "of",
      "tour.previous": "Previous",
      "tour.next": "Next",
      "tour.finish": "Finish",
    },
  },

  /**
   * Translates a key to the current language
   * @param {string} key - Translation key (e.g.: 'menu.dashboard')
   * @param {object} params - Parameters for interpolation (optional)
   * @returns {string} - Translated text
   */
  t: function (key, params = {}) {
    const translation = this.translations[this.currentLanguage]?.[key] || key;

    // Parameter interpolation
    let result = translation;
    Object.keys(params).forEach((param) => {
      result = result.replace(`{${param}}`, params[param]);
    });

    return result;
  },

  /**
   * Changes the current language and saves to server
   * @param {string} lang - Language code (e.g.: 'pt', 'en', 'es')
   */
  setLanguage: async function (lang) {
    if (this.translations[lang]) {
      this.currentLanguage = lang;

      // Save preference to server
      try {
        const response = await authenticatedFetch(
          `${API_BASE}/api/admin/users/language-preference`,
          {
            method: "POST",
            body: JSON.stringify({ language: lang }),
          }
        );

        if (!response.ok) {
          console.error("Error saving language to server");
        }
      } catch (error) {
        console.error("Error saving language:", error);
      }

      // Update translations immediately on page
      this.updatePageTranslations();
    }
  },

  /**
   * Loads the active system language and all translations from database
   */
  loadLanguage: async function () {
    try {
      // First fetch active system language
      const activeResponse = await authenticatedFetch(
        `${API_BASE}/api/translations/active`
      );
      if (activeResponse.ok) {
        const activeData = await activeResponse.json();
        const activeLanguage = activeData.active_language || "pt";

        // Fetch all translations from database
        const translationsResponse = await authenticatedFetch(
          `${API_BASE}/api/translations`
        );
        if (translationsResponse.ok) {
          const translationsData = await translationsResponse.json();

          // Load translations for all languages
          // translationsData is an array of objects with: language_code, translation_key, translation_value
          translationsData.forEach((item) => {
            if (!this.translations[item.language_code]) {
              this.translations[item.language_code] = {};
            }
            // Add each translation to the language object
            this.translations[item.language_code][item.translation_key] = item.translation_value;
          });
          
          console.log('Translations loaded from database:', Object.keys(this.translations));
        }

        // Set active language
        if (this.translations[activeLanguage]) {
          this.currentLanguage = activeLanguage;
          this.updatePageTranslations();

          // Update language selector on translations page if it exists
          const activeLanguageSelect = document.getElementById(
            "activeLanguageSelect"
          );
          if (activeLanguageSelect) {
            activeLanguageSelect.value = activeLanguage;
          }

          return;
        }
      }

      // Use default
      console.warn("Using default language (pt)");
      this.currentLanguage = "pt";
      this.updatePageTranslations();
    } catch (error) {
      console.warn("Error loading language, using default (pt):", error);
      this.currentLanguage = "pt";
      this.updatePageTranslations();
    }
  },

  /**
   * Updates all translations on the page
   */
  updatePageTranslations: function () {
    // Update elements with data-i18n
    document.querySelectorAll("[data-i18n]").forEach((element) => {
      const key = element.getAttribute("data-i18n");
      element.textContent = this.t(key);
    });

    // Update placeholders with data-i18n-placeholder
    document.querySelectorAll("[data-i18n-placeholder]").forEach((element) => {
      const key = element.getAttribute("data-i18n-placeholder");
      element.placeholder = this.t(key);
    });

    // Update titles with data-i18n-title
    document.querySelectorAll("[data-i18n-title]").forEach((element) => {
      const key = element.getAttribute("data-i18n-title");
      element.title = this.t(key);
    });

    // Update current page title (pageTitle)
    const pageTitle = document.getElementById("pageTitle");
    if (pageTitle && pageTitle.hasAttribute("data-i18n")) {
      const key = pageTitle.getAttribute("data-i18n");
      pageTitle.textContent = this.t(key);
    }
  },

  /**
   * Adds new translations dynamically and saves to database
   * @param {string} lang - Language code
   * @param {object} translations - Object with translations
   */
  addTranslations: async function (lang, translations) {
    if (!this.translations[lang]) {
      this.translations[lang] = {};
    }
    Object.assign(this.translations[lang], translations);

    try {
      const response = await authenticatedFetch(
        `${API_BASE}/api/translations`,
        {
          method: "POST",
          body: JSON.stringify({
            language_code: lang,
            translations: translations,
          }),
        }
      );

      if (!response.ok) {
        console.error("Error saving translations to database");
        throw new Error("Error saving translations");
      }

      // Update page after saving
      this.updatePageTranslations();

      return true;
    } catch (error) {
      console.error("Error saving translations:", error);
      throw error;
    }
  },

  /**
   * Loads translations from database
   */
  loadTranslationsFromDB: async function () {
    try {
      const response = await authenticatedFetch(`${API_BASE}/api/translations`);
      if (response.ok) {
        const data = await response.json();

        // Organize translations by language
        data.forEach((item) => {
          if (!this.translations[item.language_code]) {
            this.translations[item.language_code] = {};
          }
          this.translations[item.language_code][item.translation_key] =
            item.translation_value;
        });

        console.log(
          "Translations loaded from database:",
          Object.keys(this.translations)
        );
      }
    } catch (error) {
      console.error("Error loading translations from database:", error);
    }
  },
};

// Global shortcut for translation function
window.t = (key, params) => i18n.t(key, params);

// Function to translate a specific element
i18n.translateElement = function (element) {
  // Traduzir data-i18n
  if (element.hasAttribute && element.hasAttribute("data-i18n")) {
    const key = element.getAttribute("data-i18n");
    element.textContent = this.t(key);
  }

  // Traduzir data-i18n-placeholder
  if (element.hasAttribute && element.hasAttribute("data-i18n-placeholder")) {
    const key = element.getAttribute("data-i18n-placeholder");
    element.placeholder = this.t(key);
  }

  // Traduzir data-i18n-title
  if (element.hasAttribute && element.hasAttribute("data-i18n-title")) {
    const key = element.getAttribute("data-i18n-title");
    element.title = this.t(key);
  }

  // Translate child elements
  if (element.querySelectorAll) {
    element.querySelectorAll("[data-i18n]").forEach((el) => {
      const key = el.getAttribute("data-i18n");
      el.textContent = this.t(key);
    });

    element.querySelectorAll("[data-i18n-placeholder]").forEach((el) => {
      const key = el.getAttribute("data-i18n-placeholder");
      el.placeholder = this.t(key);
    });

    element.querySelectorAll("[data-i18n-title]").forEach((el) => {
      const key = el.getAttribute("data-i18n-title");
      el.title = this.t(key);
    });
  }
};

// Load language and translations on startup
document.addEventListener("DOMContentLoaded", async () => {
  await i18n.loadTranslationsFromDB();
  await i18n.loadLanguage();
  i18n.updatePageTranslations();

  // Configure language selector if it exists
  const languageSelector = document.getElementById("languageSelector");
  if (languageSelector) {
    languageSelector.value = i18n.currentLanguage;
  }

  // Configure MutationObserver to automatically translate dynamic elements
  const observer = new MutationObserver((mutations) => {
    mutations.forEach((mutation) => {
      mutation.addedNodes.forEach((node) => {
        if (node.nodeType === 1) {
          // Element node
          i18n.translateElement(node);
        }
      });
    });
  });

  // Observe changes in main content
  const mainContent = document.querySelector(".main-content");
  if (mainContent) {
    observer.observe(mainContent, {
      childList: true,
      subtree: true,
    });
  }
});

// Global function to change language
window.changeLanguage = async function (lang) {
  await i18n.setLanguage(lang);

  // Update selector
  const languageSelector = document.getElementById("languageSelector");
  if (languageSelector) {
    languageSelector.value = lang;
  }

  // Show notification
  if (typeof showNotification === "function") {
    showNotification("Language changed successfully", "success");
  }
};
