// FAQ Admin JavaScript

let welcomeMessages = [];
let placeholders = [];
let faqs = [];
let currentEditingFaq = null;

// Initialization
document.addEventListener('DOMContentLoaded', function() {
    loadWelcomeMessages();
    loadPlaceholders();
    loadFaqs();
    
    // Event listeners
    document.getElementById('faqForm').addEventListener('submit', handleFaqSubmit);
    
    // Update preview in real time
    document.getElementById('welcomeMessages').addEventListener('input', updateMobilePreview);
});

// ===== WELCOME MESSAGES =====

async function loadWelcomeMessages() {
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/faq/welcome-messages', {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            welcomeMessages = await response.json();
            renderWelcomeMessages();
            updateMobilePreview();
        }
    } catch (error) {
        console.error('Error loading welcome messages:', error);
    }
}

function renderWelcomeMessages() {
    const container = document.getElementById('welcomeMessages');
    container.innerHTML = '';
    
    // Ensure at least 3 default messages
    while (welcomeMessages.length < 3) {
        welcomeMessages.push({ message_text: '', order_position: welcomeMessages.length + 1 });
    }
    
    welcomeMessages.forEach((message, index) => {
        const messageDiv = document.createElement('div');
        messageDiv.className = 'message-line';
        messageDiv.innerHTML = `
            <input type="text" 
                   class="message-input" 
                   placeholder="Type message ${index + 1}..." 
                   value="${message.message_text || ''}"
                   onchange="updateWelcomeMessage(${index}, this.value)"
                   oninput="updateMobilePreview()">
            ${index >= 3 ? `<button type="button" class="btn-remove" onclick="removeWelcomeMessage(${index})">❌</button>` : ''}
        `;
        container.appendChild(messageDiv);
    });
    
    // Control add button visibility
    const addBtn = document.getElementById('addMessageBtn');
    addBtn.style.display = welcomeMessages.length >= 7 ? 'none' : 'block';
}

function updateWelcomeMessage(index, value) {
    welcomeMessages[index].message_text = value;
    updateMobilePreview();
}

function addWelcomeMessage() {
    if (welcomeMessages.length < 7) {
        welcomeMessages.push({ 
            message_text: '', 
            order_position: welcomeMessages.length + 1 
        });
        renderWelcomeMessages();
    }
}

function removeWelcomeMessage(index) {
    if (index >= 3) { // Can only remove extra messages
        welcomeMessages.splice(index, 1);
        // Reorder positions
        welcomeMessages.forEach((msg, i) => {
            msg.order_position = i + 1;
        });
        renderWelcomeMessages();
        updateMobilePreview();
    }
}

async function saveWelcomeMessages() {
    try {
        const token = localStorage.getItem('token');
        const validMessages = welcomeMessages.filter(msg => msg.message_text.trim() !== '');
        
        const response = await fetch('/api/faq/welcome-messages', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({
                messages: validMessages.map(msg => ({ text: msg.message_text }))
            })
        });
        
        if (response.ok) {
            showNotification('Mensagens de boas-vindas salvas com sucesso!', 'success');
            loadWelcomeMessages();
        } else {
            throw new Error('Erro ao salvar mensagens');
        }
    } catch (error) {
        console.error('Erro ao salvar mensagens:', error);
        showNotification('Erro ao salvar mensagens de boas-vindas', 'error');
    }
}

// ===== PLACEHOLDERS =====

async function loadPlaceholders() {
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/faq/placeholders', {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            placeholders = await response.json();
            renderPlaceholders();
        }
    } catch (error) {
        console.error('Erro ao carregar placeholders:', error);
    }
}

function renderPlaceholders() {
    const container = document.getElementById('placeholdersList');
    container.innerHTML = '';
    
    placeholders.forEach(placeholder => {
        const placeholderDiv = document.createElement('div');
        placeholderDiv.className = 'placeholder-item';
        placeholderDiv.onclick = () => insertPlaceholder(placeholder.placeholder_key);
        placeholderDiv.innerHTML = `
            <div class="placeholder-key">${placeholder.placeholder_key}</div>
            <div class="placeholder-value">${placeholder.placeholder_value}</div>
        `;
        container.appendChild(placeholderDiv);
    });
}

function insertPlaceholder(placeholderKey) {
    // Inserir no campo ativo (último clicado)
    const activeInput = document.activeElement;
    if (activeInput && (activeInput.tagName === 'INPUT' || activeInput.tagName === 'TEXTAREA')) {
        const cursorPos = activeInput.selectionStart;
        const textBefore = activeInput.value.substring(0, cursorPos);
        const textAfter = activeInput.value.substring(activeInput.selectionEnd);
        
        activeInput.value = textBefore + placeholderKey + textAfter;
        activeInput.focus();
        activeInput.setSelectionRange(cursorPos + placeholderKey.length, cursorPos + placeholderKey.length);
        
        // Trigger change event
        activeInput.dispatchEvent(new Event('change'));
        updateMobilePreview();
    } else {
        // Copiar para clipboard como fallback
        navigator.clipboard.writeText(placeholderKey);
        showNotification(`Placeholder ${placeholderKey} copiado!`, 'info');
    }
}

// ===== FAQs =====

async function loadFaqs() {
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/faq/faqs', {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            faqs = await response.json();
            renderFaqs();
        }
    } catch (error) {
        console.error('Erro ao carregar FAQs:', error);
    }
}

function renderFaqs() {
    const container = document.getElementById('faqsList');
    container.innerHTML = '';
    
    faqs.forEach(faq => {
        const faqDiv = document.createElement('div');
        faqDiv.className = 'faq-item';
        faqDiv.innerHTML = `
            <div class="faq-header" onclick="toggleFaqContent(${faq.id})">
                <div class="faq-emoji">${faq.emoji || '❓'}</div>
                <div class="faq-question">${faq.question}</div>
                <div class="faq-actions">
                    <button class="btn-edit" onclick="editFaq(${faq.id}); event.stopPropagation();">✏️</button>
                    <button class="btn-delete" onclick="deleteFaq(${faq.id}); event.stopPropagation();">🗑️</button>
                </div>
            </div>
            <div class="faq-content" id="faq-content-${faq.id}">
                <p><strong>Resposta:</strong> ${faq.answer}</p>
                <p><strong>Tempo de Reação:</strong> ${faq.reaction_time}s | <strong>Tempo de Resposta:</strong> ${faq.response_time}s</p>
                <p><strong>Horários:</strong> ${faq.schedule_hours} | <strong>Dias:</strong> ${faq.schedule_days}</p>
                <button class="btn-edit" onclick="testFaq(${faq.id})">🧪 Testar FAQ</button>
            </div>
        `;
        container.appendChild(faqDiv);
    });
}

function toggleFaqContent(faqId) {
    const content = document.getElementById(`faq-content-${faqId}`);
    content.classList.toggle('active');
}

function showAddFaqForm() {
    currentEditingFaq = null;
    document.getElementById('faqModalTitle').textContent = 'Adicionar FAQ';
    document.getElementById('faqForm').reset();
    document.getElementById('faqModal').style.display = 'block';
}

function editFaq(faqId) {
    const faq = faqs.find(f => f.id === faqId);
    if (!faq) return;
    
    currentEditingFaq = faq;
    document.getElementById('faqModalTitle').textContent = 'Editar FAQ';
    
    // Preencher formulário
    document.getElementById('faqQuestion').value = faq.question;
    document.getElementById('faqAnswer').value = faq.answer;
    document.getElementById('faqEmoji').value = faq.emoji || '';
    document.getElementById('faqOrder').value = faq.order_position || 1;
    document.getElementById('faqReactionTime').value = faq.reaction_time || 3;
    document.getElementById('faqResponseTime').value = faq.response_time || 7;
    document.getElementById('faqScheduleHours').value = faq.schedule_hours || '08:00-18:00';
    
    // Marcar dias da semana
    const days = (faq.schedule_days || '').split(',');
    ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'].forEach(day => {
        document.getElementById(day).checked = days.includes(day);
    });
    
    document.getElementById('faqModal').style.display = 'block';
}

async function handleFaqSubmit(e) {
    e.preventDefault();
    
    const selectedDays = [];
    ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'].forEach(day => {
        if (document.getElementById(day).checked) {
            selectedDays.push(day);
        }
    });
    
    const faqData = {
        question: document.getElementById('faqQuestion').value,
        answer: document.getElementById('faqAnswer').value,
        emoji: document.getElementById('faqEmoji').value,
        order_position: parseInt(document.getElementById('faqOrder').value),
        reaction_time: parseInt(document.getElementById('faqReactionTime').value),
        response_time: parseInt(document.getElementById('faqResponseTime').value),
        schedule_hours: document.getElementById('faqScheduleHours').value,
        schedule_days: selectedDays.join(',')
    };
    
    try {
        const token = localStorage.getItem('token');
        const url = currentEditingFaq ? `/api/faq/faqs/${currentEditingFaq.id}` : '/api/faq/faqs';
        const method = currentEditingFaq ? 'PUT' : 'POST';
        
        const response = await fetch(url, {
            method: method,
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify(faqData)
        });
        
        if (response.ok) {
            showNotification('FAQ salvo com sucesso!', 'success');
            closeFaqModal();
            loadFaqs();
        } else {
            throw new Error('Erro ao salvar FAQ');
        }
    } catch (error) {
        console.error('Erro ao salvar FAQ:', error);
        showNotification('Erro ao salvar FAQ', 'error');
    }
}

async function deleteFaq(faqId) {
    if (!confirm('Tem certeza que deseja excluir este FAQ?')) return;
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch(`/api/faq/faqs/${faqId}`, {
            method: 'DELETE',
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            showNotification('FAQ excluído com sucesso!', 'success');
            loadFaqs();
        } else {
            throw new Error('Erro ao excluir FAQ');
        }
    } catch (error) {
        console.error('Erro ao excluir FAQ:', error);
        showNotification('Erro ao excluir FAQ', 'error');
    }
}

function closeFaqModal() {
    document.getElementById('faqModal').style.display = 'none';
    currentEditingFaq = null;
}

// ===== PREVIEW MOBILE =====

function updateMobilePreview() {
    const chatContainer = document.getElementById('mobileChat');
    chatContainer.innerHTML = '';
    
    // Mostrar mensagens de boas-vindas
    welcomeMessages.forEach((message, index) => {
        if (message.message_text && message.message_text.trim() !== '') {
            setTimeout(() => {
                const messageDiv = document.createElement('div');
                messageDiv.className = 'chat-message bot';
                messageDiv.textContent = processPlaceholdersForPreview(message.message_text);
                chatContainer.appendChild(messageDiv);
                chatContainer.scrollTop = chatContainer.scrollHeight;
            }, index * 1000);
        }
    });
}

function processPlaceholdersForPreview(text) {
    let processed = text;
    
    // Substituir placeholders comuns para preview
    processed = processed.replace(/\{\{nome_cliente\}\}/g, 'João Silva');
    processed = processed.replace(/\{\{nome_loja\}\}/g, 'BeZap');
    processed = processed.replace(/\{\{horario_funcionamento\}\}/g, '08:00 às 18:00');
    processed = processed.replace(/\{\{telefone_loja\}\}/g, '(11) 99999-9999');
    processed = processed.replace(/\{\{endereco_loja\}\}/g, 'Rua das Flores, 123');
    processed = processed.replace(/\{\{data_atual\}\}/g, new Date().toLocaleDateString('pt-BR'));
    processed = processed.replace(/\{\{hora_atual\}\}/g, new Date().toLocaleTimeString('pt-BR'));
    
    return processed;
}

async function testFaq(faqId) {
    const faq = faqs.find(f => f.id === faqId);
    if (!faq) return;
    
    const chatContainer = document.getElementById('mobileChat');
    
    // Limpar chat
    chatContainer.innerHTML = '';
    
    // Simular pergunta do usuário
    const userMessage = document.createElement('div');
    userMessage.className = 'chat-message';
    userMessage.textContent = faq.question;
    chatContainer.appendChild(userMessage);
    
    // Mostrar reação (emoji)
    setTimeout(() => {
        if (faq.emoji) {
            const reactionDiv = document.createElement('div');
            reactionDiv.className = 'chat-message bot';
            reactionDiv.textContent = faq.emoji;
            reactionDiv.style.fontSize = '24px';
            reactionDiv.style.textAlign = 'center';
            chatContainer.appendChild(reactionDiv);
            chatContainer.scrollTop = chatContainer.scrollHeight;
        }
    }, faq.reaction_time * 1000);
    
    // Mostrar indicador de digitação
    setTimeout(() => {
        const typingDiv = document.createElement('div');
        typingDiv.className = 'typing-indicator';
        typingDiv.style.display = 'block';
        typingDiv.innerHTML = `
            <div class="typing-dots">
                <div class="typing-dot"></div>
                <div class="typing-dot"></div>
                <div class="typing-dot"></div>
            </div>
        `;
        chatContainer.appendChild(typingDiv);
        chatContainer.scrollTop = chatContainer.scrollHeight;
        
        // Remover indicador e mostrar resposta
        setTimeout(() => {
            typingDiv.remove();
            
            const responseDiv = document.createElement('div');
            responseDiv.className = 'chat-message bot';
            responseDiv.textContent = processPlaceholdersForPreview(faq.answer);
            chatContainer.appendChild(responseDiv);
            chatContainer.scrollTop = chatContainer.scrollHeight;
        }, faq.response_time * 1000);
        
    }, (faq.reaction_time + 1) * 1000);
}

// ===== UTILITÁRIOS =====

function showNotification(message, type = 'info') {
    // Criar elemento de notificação
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 5px;
        color: white;
        font-weight: bold;
        z-index: 10000;
        animation: slideIn 0.3s ease;
    `;
    
    // Cores por tipo
    const colors = {
        success: '#4CAF50',
        error: '#f44336',
        info: '#2196F3',
        warning: '#FF9800'
    };
    
    notification.style.backgroundColor = colors[type] || colors.info;
    
    document.body.appendChild(notification);
    
    // Remover após 3 segundos
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 3000);
}

// CSS para animações
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
    
    .modal {
        position: fixed;
        z-index: 1000;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0,0,0,0.5);
    }
    
    .modal-content {
        background-color: #fefefe;
        margin: 5% auto;
        padding: 20px;
        border-radius: 10px;
        width: 80%;
        max-width: 600px;
        max-height: 80vh;
        overflow-y: auto;
        position: relative;
    }
    
    .close {
        position: absolute;
        right: 15px;
        top: 15px;
        color: #aaa;
        font-size: 28px;
        font-weight: bold;
        cursor: pointer;
    }
    
    .close:hover {
        color: #000;
    }
`;
document.head.appendChild(style);
