// ===================================================================
//  EARLY INITIALIZATION - LOAD SETTINGS BEFORE DOM
// ===================================================================

/**
 * This script loads customization settings BEFORE the DOM
 * is fully loaded, avoiding the "flash" of default colors
 */

(function() {
    'use strict';
    
    const API_BASE = window.location.origin;
    const defaultColors = {
        menuBackground: '#8B4513',
        menuActive: '#A0522D',
        menuHover: '#9B5A3C',
        cardBackground: '#FFFFFF',
        cardBorder: '#EEEEEE',
        primaryColor: '#8B4513',
        titleColor: '#333333',
        textColor: '#666666',
        buttonPrimary: '#8B4513',
        buttonSuccess: '#28a745',
        buttonDanger: '#dc3545',
        buttonWarning: '#ffc107'
    };
    
    /**
     * Applies colors via CSS custom properties on root
     */
    function applyColorsToRoot(colors) {
        const root = document.documentElement;
        
        root.style.setProperty('--primary-color', colors.primaryColor);
        root.style.setProperty('--menu-background', colors.menuBackground);
        root.style.setProperty('--menu-active', colors.menuActive);
        root.style.setProperty('--menu-hover', colors.menuHover);
        root.style.setProperty('--card-background', colors.cardBackground);
        root.style.setProperty('--card-border', colors.cardBorder);
        root.style.setProperty('--title-color', colors.titleColor);
        root.style.setProperty('--text-color', colors.textColor);
        root.style.setProperty('--button-primary', colors.buttonPrimary);
        root.style.setProperty('--button-success', colors.buttonSuccess);
        root.style.setProperty('--button-danger', colors.buttonDanger);
        root.style.setProperty('--button-warning', colors.buttonWarning);
    }
    
    /**
     * Updates the favicon
     */
    function updateFavicon(faviconData) {
        if (!faviconData) return;
        
        let link = document.querySelector("link[rel~='icon']");
        if (!link) {
            link = document.createElement('link');
            link.rel = 'icon';
            document.head.appendChild(link);
        }
        link.href = faviconData;
    }
    
    /**
     * Updates the page title
     */
    function updateTitle(systemName) {
        if (systemName) {
            document.title = systemName;
        }
    }
    
    /**
     * Loads settings from server
     */
    async function loadSettings() {
        try {
            const token = localStorage.getItem('token');
            if (!token) {
                // If there's no token, apply default colors
                applyColorsToRoot(defaultColors);
                return;
            }
            
            const response = await fetch(`${API_BASE}/api/admin/profile-settings`, {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json'
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                
                // Apply colors
                let colors = defaultColors;
                if (data.customColors) {
                    try {
                        colors = JSON.parse(data.customColors);
                    } catch (e) {
                        console.error('Error parsing colors:', e);
                    }
                }
                applyColorsToRoot(colors);
                
                // Apply favicon
                if (data.favicon) {
                    updateFavicon(data.favicon);
                }
                
                // Apply system name
                if (data.systemName) {
                    updateTitle(data.systemName);
                }
                
                // Save to sessionStorage for quick access
                sessionStorage.setItem('profileSettings', JSON.stringify(data));
            } else {
                applyColorsToRoot(defaultColors);
            }
        } catch (error) {
            console.error('Error loading settings:', error);
            applyColorsToRoot(defaultColors);
        }
    }
    
    // Execute immediately
    loadSettings();
    
    // Also execute when DOM is ready to update dynamic elements
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function() {
            const cachedSettings = sessionStorage.getItem('profileSettings');
            if (cachedSettings) {
                try {
                    const data = JSON.parse(cachedSettings);
                    
                    // Update sidebar logo if it exists
                    if (data.systemLogo) {
                        const logoElement = document.querySelector('.sidebar .logo');
                        if (logoElement) {
                            logoElement.innerHTML = `<img src="${data.systemLogo}" alt="Logo" style="max-width: 210px; max-height: 70px;" />`;
                        }
                    } else if (data.systemName) {
                        const logoElement = document.querySelector('.sidebar .logo h2');
                        if (logoElement) {
                            logoElement.textContent = data.systemName;
                        }
                    }
                } catch (e) {
                    console.error('Error applying cached settings:', e);
                }
            }
        });
    }
})();

