// Background Sync System

class BackgroundSync {
    constructor() {
        this.syncInterval = 30000; // 30 seconds
        this.maxRetries = 3;
        this.retryDelay = 5000; // 5 seconds
        this.isOnline = navigator.onLine;
        this.syncInProgress = false;
        this.lastSyncTime = null;
        
        this.init();
    }
    
    init() {
        // Listen to connectivity changes
        window.addEventListener('online', () => this.handleOnline());
        window.addEventListener('offline', () => this.handleOffline());
        
        // Register service worker for background sync
        this.registerBackgroundSync();
        
        // Start periodic sync
        this.startPeriodicSync();
        
        // Sync on initialization if online
        if (this.isOnline) {
            setTimeout(() => this.sync(), 1000);
        }
        
        console.log('[BackgroundSync] Initialized');
    }
    
    // ===== CONNECTIVITY HANDLERS =====
    
    handleOnline() {
        console.log('[BackgroundSync] Connection restored');
        this.isOnline = true;
        
        // Sync immediately when back online
        this.sync();
        
        // Notify user
        this.showNotification('Connection restored', 'Syncing data...', 'success');
    }
    
    handleOffline() {
        console.log('[BackgroundSync] Connection lost');
        this.isOnline = false;
        
        // Notify user
        this.showNotification('No connection', 'Working offline', 'warning');
    }
    
    // ===== SERVICE WORKER REGISTRATION =====
    
    async registerBackgroundSync() {
        if ('serviceWorker' in navigator && 'sync' in window.ServiceWorkerRegistration.prototype) {
            try {
                const registration = await navigator.serviceWorker.ready;
                
                // Register sync events
                await registration.sync.register('background-sync-messages');
                await registration.sync.register('background-sync-contacts');
                await registration.sync.register('background-sync-media');
                
                console.log('[BackgroundSync] Service Worker sync registered');
            } catch (error) {
                console.error('[BackgroundSync] Failed to register SW sync:', error);
            }
        }
    }
    
    // ===== PERIODIC SYNC =====
    
    startPeriodicSync() {
        setInterval(() => {
            if (this.isOnline && !this.syncInProgress) {
                this.sync();
            }
        }, this.syncInterval);
    }
    
    // ===== MAIN SYNC FUNCTION =====
    
    async sync() {
        if (this.syncInProgress || !this.isOnline) {
            return;
        }
        
        this.syncInProgress = true;
        
        try {
            console.log('[BackgroundSync] Starting sync...');
            
            // Sync in priority order
            await this.syncMessages();
            await this.syncContacts();
            await this.syncMedia();
            await this.syncSettings();
            await this.syncFAQData();
            
            this.lastSyncTime = new Date();
            console.log('[BackgroundSync] Sync completed successfully');
            
            // Notify success if there was data to sync
            this.showSyncStatus('success');
            
        } catch (error) {
            console.error('[BackgroundSync] Sync failed:', error);
            this.showSyncStatus('error');
        } finally {
            this.syncInProgress = false;
        }
    }
    
    // ===== SYNC INDIVIDUAL COMPONENTS =====
    
    async syncMessages() {
        try {
            if (!window.offlineStorage) return;
            
            // Get unsynced messages
            const unsentMessages = await window.offlineStorage.getUnsentMessages();
            
            if (unsentMessages.length === 0) return;
            
            console.log(`[BackgroundSync] Syncing ${unsentMessages.length} messages`);
            
            for (const message of unsentMessages) {
                await this.syncSingleMessage(message);
            }
            
        } catch (error) {
            console.error('[BackgroundSync] Failed to sync messages:', error);
            throw error;
        }
    }
    
    async syncSingleMessage(message) {
        const maxRetries = this.maxRetries;
        let retryCount = 0;
        
        while (retryCount < maxRetries) {
            try {
                const token = localStorage.getItem('token');
                
                const response = await fetch('/api/chat/send', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${token}`
                    },
                    body: JSON.stringify({
                        chatId: message.chatId,
                        message: message.text,
                        type: message.type,
                        media: message.media,
                        timestamp: message.timestamp
                    })
                });
                
                if (response.ok) {
                    const result = await response.json();
                    
                    // Update message with server data
                    message.serverId = result.id;
                    message.syncStatus = 'sent';
                    message.status = 'sent';
                    
                    await window.offlineStorage.update('messages', message);
                    
                    console.log(`[BackgroundSync] Message ${message.id} synced successfully`);
                    return;
                    
                } else if (response.status === 401) {
                    // Token expired, try to renew
                    await this.refreshToken();
                    retryCount++;
                    continue;
                    
                } else {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                
            } catch (error) {
                retryCount++;
                
                if (retryCount >= maxRetries) {
                    // Mark as failed after exhausting attempts
                    message.syncStatus = 'failed';
                    message.lastSyncError = error.message;
                    await window.offlineStorage.update('messages', message);
                    
                    console.error(`[BackgroundSync] Failed to sync message ${message.id}:`, error);
                    throw error;
                }
                
                // Wait before retrying
                await this.delay(this.retryDelay * retryCount);
            }
        }
    }
    
    async syncContacts() {
        try {
            if (!window.offlineStorage) return;
            
            const token = localStorage.getItem('token');
            
            // Download updated contacts from server
            const response = await fetch('/api/chat/contacts', {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (response.ok) {
                const serverContacts = await response.json();
                
                // Update local contacts
                for (const contact of serverContacts) {
                    await window.offlineStorage.saveContact(contact);
                }
                
                console.log(`[BackgroundSync] Synced ${serverContacts.length} contacts`);
            }
            
        } catch (error) {
            console.error('[BackgroundSync] Failed to sync contacts:', error);
        }
    }
    
    async syncMedia() {
        try {
            if (!window.offlineStorage) return;
            
            // Implement pending media sync
            const pendingMedia = await window.offlineStorage.getAll('media');
            const unsynced = pendingMedia.filter(media => !media.synced);
            
            for (const media of unsynced) {
                await this.syncSingleMedia(media);
            }
            
            if (unsynced.length > 0) {
                console.log(`[BackgroundSync] Synced ${unsynced.length} media files`);
            }
            
        } catch (error) {
            console.error('[BackgroundSync] Failed to sync media:', error);
        }
    }
    
    async syncSingleMedia(media) {
        try {
            const token = localStorage.getItem('token');
            
            // Convert base64 back to blob
            const blob = window.offlineStorage.base64ToBlob(media.data, media.mimeType);
            
            const formData = new FormData();
            formData.append('file', blob, media.name);
            formData.append('messageId', media.messageId);
            
            const response = await fetch('/api/chat/upload-media', {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${token}`
                },
                body: formData
            });
            
            if (response.ok) {
                const result = await response.json();
                
                // Update media with server URL
                media.serverUrl = result.url;
                media.synced = true;
                
                await window.offlineStorage.update('media', media);
            }
            
        } catch (error) {
            console.error('[BackgroundSync] Failed to sync media:', error);
        }
    }
    
    async syncSettings() {
        try {
            if (!window.offlineStorage) return;
            
            const token = localStorage.getItem('token');
            
            // Download settings from server
            const response = await fetch('/api/settings', {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (response.ok) {
                const settings = await response.json();
                
                // Update local settings
                for (const [key, value] of Object.entries(settings)) {
                    await window.offlineStorage.saveSetting(key, value);
                }
                
                console.log('[BackgroundSync] Settings synced');
            }
            
        } catch (error) {
            console.error('[BackgroundSync] Failed to sync settings:', error);
        }
    }
    
    async syncFAQData() {
        try {
            if (!window.offlineStorage) return;
            
            const token = localStorage.getItem('token');
            
            // Download FAQ data from server
            const response = await fetch('/api/faq/items', {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (response.ok) {
                const faqItems = await response.json();
                
                // Update local FAQ
                for (const item of faqItems) {
                    await window.offlineStorage.saveFAQItem(item);
                }
                
                console.log(`[BackgroundSync] Synced ${faqItems.length} FAQ items`);
            }
            
        } catch (error) {
            console.error('[BackgroundSync] Failed to sync FAQ data:', error);
        }
    }
    
    // ===== TOKEN MANAGEMENT =====
    
    async refreshToken() {
        try {
            const refreshToken = localStorage.getItem('refreshToken');
            
            if (!refreshToken) {
                throw new Error('No refresh token available');
            }
            
            const response = await fetch('/api/auth/refresh', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    refreshToken: refreshToken
                })
            });
            
            if (response.ok) {
                const result = await response.json();
                
                localStorage.setItem('token', result.token);
                if (result.refreshToken) {
                    localStorage.setItem('refreshToken', result.refreshToken);
                }
                
                console.log('[BackgroundSync] Token refreshed successfully');
                return true;
                
            } else {
                throw new Error('Failed to refresh token');
            }
            
        } catch (error) {
            console.error('[BackgroundSync] Failed to refresh token:', error);
            
            // Redirect to login if unable to renew token
            this.handleAuthFailure();
            return false;
        }
    }
    
    handleAuthFailure() {
        // Clear authentication data
        localStorage.removeItem('token');
        localStorage.removeItem('refreshToken');
        
        // Notify user
        this.showNotification('Session expired', 'Please login again', 'error');
        
        // Redirect to login after a delay
        setTimeout(() => {
            window.location.href = '/login';
        }, 3000);
    }
    
    // ===== MANUAL SYNC TRIGGERS =====
    
    async forcSync() {
        if (this.syncInProgress) {
            console.log('[BackgroundSync] Sync already in progress');
            return;
        }
        
        if (!this.isOnline) {
            this.showNotification('No connection', 'Cannot sync offline', 'warning');
            return;
        }
        
        this.showNotification('Syncing', 'Updating data...', 'info');
        await this.sync();
    }
    
    async syncSpecificType(type) {
        if (!this.isOnline) return;
        
        try {
            switch (type) {
                case 'messages':
                    await this.syncMessages();
                    break;
                case 'contacts':
                    await this.syncContacts();
                    break;
                case 'media':
                    await this.syncMedia();
                    break;
                case 'settings':
                    await this.syncSettings();
                    break;
                case 'faq':
                    await this.syncFAQData();
                    break;
            }
            
            console.log(`[BackgroundSync] ${type} synced successfully`);
        } catch (error) {
            console.error(`[BackgroundSync] Failed to sync ${type}:`, error);
        }
    }
    
    // ===== SYNC STATUS =====
    
    getSyncStatus() {
        return {
            isOnline: this.isOnline,
            syncInProgress: this.syncInProgress,
            lastSyncTime: this.lastSyncTime,
            nextSyncIn: this.getNextSyncTime()
        };
    }
    
    getNextSyncTime() {
        if (!this.lastSyncTime) return 0;
        
        const nextSync = new Date(this.lastSyncTime.getTime() + this.syncInterval);
        const now = new Date();
        
        return Math.max(0, nextSync.getTime() - now.getTime());
    }
    
    // ===== UI NOTIFICATIONS =====
    
    showSyncStatus(status) {
        const statusElement = document.getElementById('syncStatus');
        if (statusElement) {
            statusElement.className = `sync-status ${status}`;
            
            switch (status) {
                case 'success':
                    statusElement.textContent = '✓ Synced';
                    break;
                case 'error':
                    statusElement.textContent = '⚠ Sync error';
                    break;
                case 'syncing':
                    statusElement.textContent = '⟳ Syncing...';
                    break;
            }
            
            // Hide after 3 seconds
            setTimeout(() => {
                statusElement.textContent = '';
                statusElement.className = 'sync-status';
            }, 3000);
        }
    }
    
    showNotification(title, message, type = 'info') {
        // Use PWA notification system if available
        if (window.pwaManager) {
            switch (type) {
                case 'success':
                    window.pwaManager.showNotificationSuccess(`${title}: ${message}`);
                    break;
                case 'error':
                    window.pwaManager.showNotificationError(`${title}: ${message}`);
                    break;
                case 'warning':
                    window.pwaManager.showNotificationWarning(`${title}: ${message}`);
                    break;
                default:
                    window.pwaManager.showNotificationInfo(`${title}: ${message}`);
            }
        } else {
            console.log(`[BackgroundSync] ${title}: ${message}`);
        }
    }
    
    // ===== UTILITIES =====
    
    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
    
    // ===== PUBLIC API =====
    
    async clearSyncData() {
        if (window.offlineStorage) {
            // Clear only sync data, keep offline data
            const messages = await window.offlineStorage.getAll('messages');
            
            for (const message of messages) {
                if (message.syncStatus === 'sent') {
                    message.syncStatus = 'pending';
                    await window.offlineStorage.update('messages', message);
                }
            }
        }
        
        console.log('[BackgroundSync] Sync data cleared');
    }
    
    setSyncInterval(interval) {
        this.syncInterval = Math.max(10000, interval); // Minimum 10 seconds
        console.log(`[BackgroundSync] Sync interval set to ${this.syncInterval}ms`);
    }
    
    async getUnsentCount() {
        if (!window.offlineStorage) return 0;
        
        const unsentMessages = await window.offlineStorage.getUnsentMessages();
        return unsentMessages.length;
    }
}

// ===== GLOBAL INSTANCE =====

let backgroundSync;

// Initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        backgroundSync = new BackgroundSync();
        window.backgroundSync = backgroundSync;
    });
} else {
    backgroundSync = new BackgroundSync();
    window.backgroundSync = backgroundSync;
}

// ===== CSS PARA STATUS =====

const style = document.createElement('style');
style.textContent = `
    .sync-status {
        position: fixed;
        top: 10px;
        right: 10px;
        padding: 8px 12px;
        border-radius: 6px;
        font-size: 12px;
        font-weight: bold;
        z-index: 1000;
        transition: all 0.3s ease;
    }
    
    .sync-status.success {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }
    
    .sync-status.error {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }
    
    .sync-status.syncing {
        background: #d1ecf1;
        color: #0c5460;
        border: 1px solid #bee5eb;
    }
    
    .connectivity-indicator {
        position: fixed;
        bottom: 10px;
        left: 10px;
        padding: 6px 10px;
        border-radius: 15px;
        font-size: 11px;
        font-weight: bold;
        text-transform: uppercase;
        z-index: 1000;
    }
    
    .connectivity-indicator.online {
        background: #d4edda;
        color: #155724;
    }
    
    .connectivity-indicator.offline {
        background: #f8d7da;
        color: #721c24;
    }
`;
document.head.appendChild(style);

// Add status indicators to DOM
document.addEventListener('DOMContentLoaded', () => {
    if (!document.getElementById('syncStatus')) {
        const syncStatus = document.createElement('div');
        syncStatus.id = 'syncStatus';
        syncStatus.className = 'sync-status';
        document.body.appendChild(syncStatus);
    }
    
    if (!document.getElementById('connectivityIndicator')) {
        const connectivityIndicator = document.createElement('div');
        connectivityIndicator.id = 'connectivityIndicator';
        connectivityIndicator.className = `connectivity-indicator ${navigator.onLine ? 'online' : 'offline'}`;
        connectivityIndicator.textContent = navigator.onLine ? 'Online' : 'Offline';
        document.body.appendChild(connectivityIndicator);
        
        // Update indicator when connectivity changes
        window.addEventListener('online', () => {
            connectivityIndicator.className = 'connectivity-indicator online';
            connectivityIndicator.textContent = 'Online';
        });
        
        window.addEventListener('offline', () => {
            connectivityIndicator.className = 'connectivity-indicator offline';
            connectivityIndicator.textContent = 'Offline';
        });
    }
});

// Exportar para uso em módulos
if (typeof module !== 'undefined' && module.exports) {
    module.exports = BackgroundSync;
}
