// ===================================================================
//  HELPER AND CONFIGURATION SECTION
// ===================================================================

// The base URL of your API (defined in auth.js)
// const API_BASE is already defined in auth.js

/**
 * Gets user data that was saved in localStorage during login.
 */
function getCurrentUser() {
  const userString = localStorage.getItem("currentUser");
  try {
    return userString ? JSON.parse(userString) : null;
  } catch (e) {
    console.error("Error reading user data:", e);
    return null;
  }
}

/**
 * Function that wraps 'fetch' to automatically add authentication token
 */
async function authenticatedFetch(url, options = {}) {
  const token = localStorage.getItem("authToken");
  const headers = {
    ...options.headers,
  };

  // Only add Content-Type if not FormData (let browser set automatically)
  if (!(options.body instanceof FormData)) {
    headers["Content-Type"] = "application/json";
  }

  if (token) {
    headers["Authorization"] = `Bearer ${token}`;
  }

  const response = await fetch(url, { ...options, headers });

  if (response.status === 401 || response.status === 403) {
    localStorage.removeItem("authToken");
    localStorage.removeItem("currentUser");
    window.location.href = "/";
    throw new Error("Unauthorized");
  }

  return response;
}

function formatDate(dateString) {
  if (!dateString) return "";
  return new Date(dateString).toLocaleString("en-US");
}

function showNotification(message, type = "info") {
  console.log(`[${type.toUpperCase()}] ${message}`);

  // Remove previous notification if it exists
  const existingNotification = document.querySelector(".notification");
  if (existingNotification) {
    existingNotification.remove();
  }

  const notification = document.createElement("div");
  notification.className = `notification ${type}`;
  notification.textContent = message;
  notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${
          type === "success"
            ? "#28a745"
            : type === "error"
            ? "#dc3545"
            : type === "warning"
            ? "#ffc107"
            : "#8B4513"
        };
        color: white;
        padding: 15px 20px;
        border-radius: 8px;
        box-shadow: 0 4px 15px rgba(0,0,0,0.2);
        z-index: 1001;
        display: block;
    `;

  document.body.appendChild(notification);
  setTimeout(() => {
    notification.remove();
  }, 3000);
}

function formatPhone(phone) {
  return phone || "N/A";
}

// ===================================================================
//  MAIN CODE
// ===================================================================

// Global variables
let socket;
let currentPage = "dashboard";
let currentConversationtionId = null;
let hourlyChart = null;

// Initialization
document.addEventListener("DOMContentLoaded", function () {
  const token = localStorage.getItem("authToken");
  if (!token) {
    window.location.href = "/";
    return;
  }

  const user = getCurrentUser();
  if (!user || user.role !== "admin") {
    localStorage.removeItem("authToken");
    localStorage.removeItem("currentUser");
    window.location.href = "/";
    return;
  }

  initializeAdmin();
});

// Initialize admin panel
async function initializeAdmin() {
  const user = getCurrentUser();
  const userDisplay = document.getElementById("currentUser");
  if (userDisplay) userDisplay.textContent = user.username;

  // Load active system language
  if (typeof i18n !== "undefined") {
    await i18n.loadLanguage();
  }

  initializeSocket();
  setupNavigation();
  setupModals();
  setupControlButtons();
  setupMobileMenu();
}

// Configure mobile menu
function setupMobileMenu() {
  // Add hamburger button if it doesn't exist
  const header = document.querySelector(".page-header");
  if (header && window.innerWidth <= 1024) {
    const menuBtn = document.createElement("button");
    menuBtn.className = "mobile-menu-btn";
    menuBtn.innerHTML = "☰";
    menuBtn.onclick = toggleSidebar;
    header.insertBefore(menuBtn, header.firstChild);
  }

  // Close sidebar when clicking item (mobile)
  document.querySelectorAll(".nav-item").forEach((item) => {
    item.addEventListener("click", () => {
      if (window.innerWidth <= 1024) {
        document.querySelector(".sidebar").classList.remove("open");
      }
    });
  });
}

function toggleSidebar() {
  const sidebar = document.querySelector(".sidebar");
  sidebar.classList.toggle("open");
}

// Inicializar Socket.IO
function initializeSocket() {
  socket = io();

  socket.on("connect", () => {
    console.log("Connected to server");
  });

  socket.on("qr-code", (qrCode) => {
    console.log("QR Code received:", qrCode ? "Yes" : "No");
    updateQRCode(qrCode);
  });

  socket.on("connection-status", (status) => {
    console.log("Connection status:", status);
    updateConnectionStatus(status.connected);
  });

  socket.on("new-message", (message) => {
    handleNewMessage(message);
    updateDashboard();
  });

  socket.on("conversation-attended", (data) => {
    showNotification(
      `${i18n.t("notification.conversation_attended")} ${data.attendedBy} (${
        data.store
      })`,
      "info"
    );
    loadConversationtions();
  });
}

// Configure navegação
function setupNavigation() {
  const initialPage = window.location.hash.substring(1) || "dashboard";

  document.querySelectorAll(".nav-item").forEach((item) => {
    item.addEventListener("click", (e) => {
      e.preventDefault();
      const page = item.dataset.page;
      if (page) {
        window.location.hash = page;
      }
    });
  });

  window.addEventListener("hashchange", () => {
    const page = window.location.hash.substring(1) || "dashboard";
    showPage(page);
  });

  showPage(initialPage);
}

// Função melhorada para showPage - evitar carregamentos desnecessários
async function showPage(page) {
  document.querySelectorAll(".nav-item").forEach((item) => {
    item.classList.remove("active");
  });
  const activeNavItem = document.querySelector(`[data-page="${page}"]`);
  if (activeNavItem) activeNavItem.classList.add("active");

  document.querySelectorAll(".page").forEach((p) => {
    p.style.display = "none";
  });
  const activePage = document.getElementById(`${page}-page`);
  if (activePage) activePage.style.display = "block";

  currentPage = page;

  // Atualizar título da página com tradução
  const pageTitle = document.getElementById("pageTitle");
  if (pageTitle) {
    const pageTitleKeys = {
      dashboard: "dashboard.title",
      users: "users.title",
      chat: "conversations.title",
      stores: "stores.title",
      departments: "departments.title",
      faqs: "faqs.title",
      "mass-messaging": "mass_messaging.title",
      payments: "payments.title",
      translations: "translations.title",
      "ai-config": "ai_config.title",
      "push-notifications": "push_notifications.title",
      contacts: "contacts.title",
      profile: "profile.title",
      settings: "settings.title",
    };

    const titleKey = pageTitleKeys[page];
    if (titleKey && typeof i18n !== "undefined") {
      pageTitle.setAttribute("data-i18n", titleKey);
      pageTitle.textContent = i18n.t(titleKey);
    } else {
      pageTitle.textContent = "Admin";
    }
  }

  // Carregar dados específicos da página
  try {
    switch (page) {
      case "dashboard":
        await loadDashboard();
        break;
      case "users":
        await loadUsers();
        break;
      case "chat":
        await loadConversationtions();
        break;
      case "stores":
        await loadStores();
        break;
      case "departments":
        await loadDepartments();
        break;
      case "faqs":
        await loadFaqs();
        break;
      case "mass-messaging":
        await loadMassMessaging();
        break;
      case "payments":
        await loadPayments();
        break;
      case "translations":
        await loadTranslations();
        break;
      case "ai-config":
        await loadAiConfig();
        break;
      case "push-notifications":
        await loadPushNotifications();
        break;
      case "contacts":
        await loadContacts();
        break;
      case "profile":
        await loadProfile();
        break;
      case "settings":
        await loadSettings();
        break;
    }
  } catch (error) {
    console.error(`Error loading page ${page}:`, error);
    showNotification(
      i18n.t("notification.page_load_error") + ` ${page}: ${error.message}`,
      "error"
    );
  }
}

// Carregar dashboard
async function loadDashboard() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/dashboard`
    );
    if (!response.ok) {
      throw new Error("Failed to load dashboard data");
    }

    const data = await response.json();

    // Função auxiliar para definir o texto de forma segura
    const setText = (id, value) => {
      const element = document.getElementById(id);
      if (element) {
        element.textContent = value || 0; // Usa 0 se o valor for nulo/undefined
      } else {
        // Aviso importante para depuração!
        console.warn(`Element with ID '${id}' not found in DOM.`);
      }
    };

    // Atualizar métricas de forma segura
    setText("todayMessages", data.todayMessages);
    setText("uniqueContacts", data.uniqueContacts);

    const waiting =
      data.conversationStats?.find((s) => s.status === "waiting")?.count || 0;
    const attended =
      data.conversationStats?.find((s) => s.status === "attended")?.count || 0;

    setText("waitingConversationtions", waiting);
    setText("activeConversationtions", attended);

    // Atualizar gráfico
    updateHourlyChart(data.hourlyStats || []);

    // Carregar status do bot e do WhatsApp
    await loadBotStatus();
    await loadWhatsAppStatus();

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  } catch (error) {
    console.error("Error loading dashboard:", error);
    showNotification(
      i18n.t("notification.dashboard_load_error") + ": " + error.message,
      "error"
    );
  }
}

// Carregar status do bot
async function loadBotStatus() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/bot-settings`
    );
    const settings = await response.json();

    const pauseBotBtn = document.getElementById("pauseBot");
    const pauseGroupsBtn = document.getElementById("pauseGroups");

    if (pauseBotBtn) {
      pauseBotBtn.innerHTML = `<span data-i18n="${
        settings.bot_active === "true"
          ? "dashboard.pause_bot"
          : "dashboard.activate_bot"
      }">${
        settings.bot_active === "true"
          ? i18n.t("dashboard.pause_bot")
          : i18n.t("dashboard.activate_bot")
      }</span>`;
      pauseBotBtn.className =
        settings.bot_active === "true" ? "btn btn-warning" : "btn btn-success";
    }

    if (pauseGroupsBtn) {
      pauseGroupsBtn.innerHTML = `<span data-i18n="${
        settings.auto_reply_groups === "true"
          ? "dashboard.pause_groups"
          : "dashboard.activate_groups"
      }">${
        settings.auto_reply_groups === "true"
          ? i18n.t("dashboard.pause_groups")
          : i18n.t("dashboard.activate_groups")
      }</span>`;
      pauseGroupsBtn.className =
        settings.auto_reply_groups === "true"
          ? "btn btn-secondary"
          : "btn btn-success";
    }
  } catch (error) {
    console.error("Error loading bot status:", error);
  }
}

// Atualizar gráfico de horários
function updateHourlyChart(hourlyStats) {
  const canvas = document.getElementById("hourlyChart");
  if (!canvas) return;

  const ctx = canvas.getContext("2d");

  if (hourlyChart) {
    hourlyChart.destroy();
  }

  const hours = Array.from({ length: 24 }, (_, i) => i);
  const data = hours.map((hour) => {
    const stat = hourlyStats.find((s) => s.hour === hour);
    return stat ? stat.message_count : 0;
  });

  hourlyChart = new Chart(ctx, {
    type: "bar",
    data: {
      labels: hours.map((h) => `${h}:00`),
      datasets: [
        {
          label: i18n.t("dashboard.messages"),
          data: data,
          backgroundColor: "#8B4513",
          borderColor: "#654321",
          borderWidth: 1,
        },
      ],
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      scales: {
        y: {
          beginAtZero: true,
        },
      },
    },
  });
}

// Atualizar QR Code
// Função para atualizar QR Code com melhor handling
function updateQRCode(qrCode) {
  const container = document.getElementById("qrCodeContainer");
  if (!container) return;

  if (qrCode) {
    container.innerHTML = `
            <div class="qr-code-wrapper">
                <img id="qrCodeImage" src="${qrCode}" alt="QR Code WhatsApp" style="max-width: 300px; height: auto; border: 1px solid #ddd; border-radius: 8px;">
                <p class="qr-instructions">Escaneie este QR Code com seu WhatsApp para conectar</p>
                <button onclick="refreshQRCode()" class="btn btn-secondary btn-sm" style="margin-top: 10px;">🔄 Update QR Code</button>
            </div>
        `;
  } else {
    container.innerHTML = `
            <div class="qr-placeholder" style="padding: 20px; text-align: center; background: #f8f9fa; border-radius: 8px; border: 2px solid #28a745;">
                <div style="color: #28a745; font-size: 24px; margin-bottom: 10px;">✅</div>
                <div style="color: #28a745; font-weight: bold;" data-i18n="dashboard.whatsapp_connected">${i18n.t(
                  "dashboard.whatsapp_connected"
                )}</div>
                <div style="color: #6c757d; font-size: 14px; margin-top: 5px;" data-i18n="dashboard.bot_ready">${i18n.t(
                  "dashboard.bot_ready"
                )}</div>
            </div>
        `;
  }
}

// Função para atualizar status de conexão com mais detalhes
function updateConnectionStatus(status) {
  const indicator = document.querySelector(".status-indicator");
  const text = document.querySelector(".status-text");

  if (indicator && text) {
    if (status.connected) {
      indicator.classList.remove("connecting", "disconnected");
      indicator.classList.add("connected");
      text.textContent = i18n.t("dashboard.connected");
      text.style.color = "#28a745";
    } else if (status.connecting) {
      indicator.classList.remove("connected", "disconnected");
      indicator.classList.add("connecting");
      text.textContent = "Connecting...";
      text.style.color = "#ffc107";
    } else {
      indicator.classList.remove("connected", "connecting");
      indicator.classList.add("disconnected");
      text.textContent = status.error || "Disconnected";
      text.style.color = "#dc3545";
    }
  }

  // Atualizar dashboard se necessário
  updateDashboardConnectionInfo(status);
}

// Função para atualizar informações de conexão no dashboard
function updateDashboardConnectionInfo(status) {
  const dashboardStatus = document.getElementById("dashboardConnectionStatus");
  if (dashboardStatus) {
    if (status.connected) {
      dashboardStatus.innerHTML = `
                <div class="alert alert-success">
                    <strong data-i18n="dashboard.whatsapp_connected">${i18n.t(
                      "dashboard.whatsapp_connected"
                    )}</strong><br>
                    <span data-i18n="dashboard.bot_operational">${i18n.t(
                      "dashboard.bot_operational"
                    )}</span>
                </div>
            `;
    } else {
      dashboardStatus.innerHTML = `
                <div class="alert alert-warning">
                    <strong data-i18n="dashboard.whatsapp_disconnected">${i18n.t(
                      "dashboard.whatsapp_disconnected"
                    )}</strong><br>
                    ${status.error || i18n.t("dashboard.checking")}
                    ${status.connecting ? "<br><em>Connecting...</em>" : ""}
                </div>
            `;
    }
  }
}

// Função para forçar refresh do QR Code
async function refreshQRCode() {
  try {
    showNotification(i18n.t("notification.updating_qr"), "info");

    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/whatsapp-reconnect`,
      {
        method: "POST",
      }
    );

    if (response.ok) {
      showNotification(i18n.t("notification.qr_updating"), "success");
    } else {
      const error = await response.json();
      showNotification(error.error || i18n.t("notification.qr_error"), "error");
    }
  } catch (error) {
    console.error("Error updating QR Code:", error);
    showNotification(i18n.t("notification.qr_error"), "error");
  }
}

// Função para desconectar WhatsApp
async function disconnectWhatsApp() {
  if (!confirm("Are you sure you want to disconnect WhatsApp?")) {
    return;
  }

  try {
    showNotification(i18n.t("notification.disconnecting_whatsapp"), "info");

    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/whatsapp-disconnect`,
      {
        method: "POST",
      }
    );

    if (response.ok) {
      showNotification(
        i18n.t("notification.whatsapp_disconnected_success"),
        "success"
      );
    } else {
      const error = await response.json();
      showNotification(
        error.error || i18n.t("notification.whatsapp_disconnect_error"),
        "error"
      );
    }
  } catch (error) {
    console.error("Error disconnecting WhatsApp:", error);
    showNotification(i18n.t("notification.whatsapp_disconnect_error"), "error");
  }
}

// Melhorar inicialização do Socket.IO
function initializeSocket() {
  // Configurações de reconexão mais agressivas
  socket = io({
    reconnection: true,
    reconnectionDelay: 1000,
    reconnectionAttempts: 10,
    timeout: 20000,
    forceNew: true,
  });

  socket.on("connect", () => {
    console.log("✅ Connected to server Socket.IO");
    showNotification(i18n.t("notification.connected_server"), "success");
  });

  socket.on("disconnect", (reason) => {
    console.log("❌ Disconnected do servidor:", reason);
    showNotification(i18n.t("notification.connection_lost"), "warning");
  });

  socket.on("reconnect", () => {
    console.log("🔄 Reconnected to the server");
    showNotification(i18n.t("notification.reconnected_server"), "success");
    // Recarregar dados se necessário
    if (currentPage === "dashboard") {
      loadDashboard();
    }
  });

  socket.on("qr-code", (qrCode) => {
    console.log("📱 QR Code received:", qrCode ? "Yes" : "No");
    updateQRCode(qrCode);
  });

  socket.on("connection-status", (status) => {
    console.log("📶 Connection status:", status);
    updateConnectionStatus(status);
  });

  socket.on("new-message", (message) => {
    console.log("📨 Nova mensagem recebida:", message.phoneNumber);
    handleNewMessage(message);
    updateDashboard();
  });

  socket.on("conversation-attended", (data) => {
    showNotification(
      `${i18n.t("notification.conversation_attended")} ${data.attendedBy} (${
        data.store
      })`,
      "info"
    );
    if (currentPage === "chat") {
      loadConversationtions();
    }
  });

  socket.on("message-sent", (data) => {
    console.log("📤 Message enviada via bot:", data.phoneNumber);
    // Atualizar interface se necessário
    if (currentPage === "chat") {
      loadConversationtions();
    }
  });

  // Lidar com erros de conexão
  socket.on("connect_error", (error) => {
    console.error("❌ Error conexão Socket.IO:", error);
    showNotification(i18n.t("notification.connection_error"), "error");
  });
}

// Carregar status do WhatsApp
async function loadWhatsAppStatus() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/whatsapp-status`
    );

    if (response.ok) {
      const status = await response.json();
      updateConnectionStatus(status);

      if (status.qrCode) {
        updateQRCode(status.qrCode);
      }
    }
  } catch (error) {
    console.error("Error loading WhatsApp status:", error);
  }
}

// Função para mostrar/esconder loading no dashboard
function showDashboardLoading(show) {
  const loadingElements = document.querySelectorAll(".dashboard-loading");
  const contentElements = document.querySelectorAll(".dashboard-content");

  if (show) {
    loadingElements.forEach((el) => (el.style.display = "block"));
    contentElements.forEach((el) => (el.style.opacity = "0.5"));
  } else {
    loadingElements.forEach((el) => (el.style.display = "none"));
    contentElements.forEach((el) => (el.style.opacity = "1"));
  }
}

// Add CSS para melhorar visual do QR Code
const qrCodeStyles = `
<style>
.qr-code-wrapper {
    text-align: center;
    padding: 20px;
    background: #f8f9fa;
    border-radius: 8px;
    border: 1px solid #dee2e6;
}

.qr-instructions {
    margin: 10px 0 5px 0;
    font-size: 14px;
    color: #6c757d;
}

.status-indicator {
    width: 12px;
    height: 12px;
    border-radius: 50%;
    display: inline-block;
    margin-right: 8px;
    background: #dc3545;
    animation: pulse 2s infinite;
}

.status-indicator.connected {
    background: #28a745;
    animation: none;
}

.status-indicator.connecting {
    background: #ffc107;
    animation: pulse 1s infinite;
}

@keyframes pulse {
    0% { opacity: 1; }
    50% { opacity: 0.5; }
    100% { opacity: 1; }
}

.dashboard-loading {
    display: none;
    text-align: center;
    padding: 20px;
    color: #6c757d;
}

.alert {
    padding: 15px;
    margin-bottom: 20px;
    border: 1px solid transparent;
    border-radius: 4px;
}

.alert-success {
    color: #155724;
    background-color: #d4edda;
    border-color: #c3e6cb;
}

.alert-warning {
    color: #856404;
    background-color: #fff3cd;
    border-color: #ffeeba;
}

.alert-danger {
    color: #721c24;
    background-color: #f8d7da;
    border-color: #f5c6cb;
}

.connection-actions {
    margin-top: 15px;
    text-align: center;
}

.connection-actions .btn {
    margin: 0 5px;
}
</style>
`;

// Add estilos ao head quando a página carregar
document.addEventListener("DOMContentLoaded", function () {
  if (!document.getElementById("qr-code-styles")) {
    const styleElement = document.createElement("div");
    styleElement.id = "qr-code-styles";
    styleElement.innerHTML = qrCodeStyles;
    document.head.appendChild(styleElement);
  }
});

// Função para adicionar botões de controle do WhatsApp
function addWhatsAppControls() {
  const container = document.getElementById("qrCodeContainer");
  if (!container) return;

  // Add botões de controle se não existirem
  let controlsContainer = document.querySelector(".whatsapp-controls");
  if (!controlsContainer) {
    controlsContainer = document.createElement("div");
    controlsContainer.className = "whatsapp-controls connection-actions";
    controlsContainer.innerHTML = `
            <button onclick="refreshQRCode()" class="btn btn-primary btn-sm">
                🔄 Reconectar WhatsApp
            </button>
            <button onclick="disconnectWhatsApp()" class="btn btn-danger btn-sm">
                🔌 Desconectar
            </button>
        `;
    container.parentNode.insertBefore(controlsContainer, container.nextSibling);
  }
}

// Função para monitorar o status de conexão periodicamente
function startConnectionMonitoring() {
  setInterval(async () => {
    if (currentPage === "dashboard") {
      try {
        await loadWhatsAppStatus();
      } catch (error) {
        console.error("Error in monitoramento:", error);
      }
    }
  }, 30000); // Verificar a cada 30 segundos
}

// Função para handle de erro mais robusto
function handleConnectionError(error) {
  console.error("Error conexão:", error);

  const container = document.getElementById("qrCodeContainer");
  if (container) {
    container.innerHTML = `
            <div class="alert alert-danger">
                <strong>❌ Error Conexão</strong><br>
                ${
                  error.message || "Errorsconhecido ao conectar com WhatsApp"
                }<br>
                <div class="connection-actions">
                <button onclick="refreshQRCode()" class="btn btn-primary btn-sm">
                    🔄 Tentar Novamente
                </button>
                </div>
            </div>
        `;
  }
}

// Melhorar a função de inicialização admin
async function initializeAdmin() {
  const user = getCurrentUser();
  const userDisplay = document.getElementById("currentUser");
  if (userDisplay) userDisplay.textContent = user.username;

  // Inicializar componentes
  initializeSocket();
  setupNavigation();
  setupModals();
  setupControlButtons();
  setupMobileMenu();

  // Add controles do WhatsApp
  setTimeout(() => {
    addWhatsAppControls();
    startConnectionMonitoring();
  }, 1000);

  // Carregar página inicial
  const initialPage = window.location.hash.substring(1) || "dashboard";
  await showPage(initialPage);
}

// Função para debug do Socket.IO
function debugSocketConnection() {
  if (socket) {
    console.log("Socket connected:", socket.connected);
    console.log("Socket ID:", socket.id);
    console.log("Socket transport:", socket.io.engine.transport.name);
    console.log("Socket readyState:", socket.io.engine.readyState);
  }
}

// Add função para testar conexão
async function testWhatsAppConnection() {
  try {
    showNotification(i18n.t("notification.testing_connection"), "info");

    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/whatsapp-status`
    );

    if (response.ok) {
      const status = await response.json();
      console.log("WhatsApp Status:", status);

      if (status.connected) {
        showNotification(i18n.t("notification.whatsapp_working"), "success");
      } else {
        showNotification(
          i18n.t("notification.whatsapp_not_connected"),
          "warning"
        );
      }

      updateConnectionStatus(status);
    } else {
      throw new Error("Error checking status");
    }
  } catch (error) {
    console.error("Error testing connection:", error);
    showNotification(i18n.t("notification.test_connection_error"), "error");
  }
}

// Função para limpar dados de sessão e reiniciar
async function resetWhatsAppSession() {
  if (
    !confirm(
      "This will clear all session data and require new authentication. Continue?"
    )
  ) {
    return;
  }

  try {
    showNotification(i18n.t("notification.cleaning_session"), "info");

    // Primeiro desconectar
    await authenticatedFetch(`${API_BASE}/api/admin/whatsapp-disconnect`, {
      method: "POST",
    });

    // Aguardar um pouco
    await new Promise((resolve) => setTimeout(resolve, 2000));

    // Reconectar
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/whatsapp-reconnect`,
      {
        method: "POST",
      }
    );

    if (response.ok) {
      showNotification(i18n.t("notification.session_cleaned"), "success");
    } else {
      throw new Error("Error reinitializing session");
    }
  } catch (error) {
    console.error("Error resetting session:", error);
    showNotification(i18n.t("notification.reset_session_error"), "error");
  }
}

// Add ao objeto global para debug
window.whatsappDebug = {
  debugSocket: debugSocketConnection,
  testConnection: testWhatsAppConnection,
  resetSession: resetWhatsAppSession,
  refreshQR: refreshQRCode,
};

// Atualizar status de conexão
function updateConnectionStatus(connected) {
  const indicator = document.querySelector(".status-indicator");
  const text = document.querySelector(".status-text");

  if (indicator && text) {
    if (connected) {
      indicator.classList.add("connected");
      text.textContent = "Connected";
    } else {
      indicator.classList.remove("connected");
      text.textContent = "Disconnected";
    }
  }
}

// Configure botões de controle
function setupControlButtons() {
  const pauseBotBtn = document.getElementById("pauseBot");
  const pauseGroupsBtn = document.getElementById("pauseGroups");

  if (pauseBotBtn) pauseBotBtn.addEventListener("click", toggleBot);
  if (pauseGroupsBtn) pauseGroupsBtn.addEventListener("click", toggleGroups);
}

// Alternar bot
async function toggleBot() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/bot-settings`
    );
    const settings = await response.json();

    const newStatus = settings.bot_active === "true" ? "false" : "true";

    await authenticatedFetch(`${API_BASE}/api/admin/bot-settings`, {
      method: "PUT",
      body: JSON.stringify({ bot_active: newStatus }),
    });

    const btn = document.getElementById("pauseBot");
    btn.innerHTML = `<span data-i18n="${
      newStatus === "true" ? "dashboard.pause_bot" : "dashboard.activate_bot"
    }">${
      newStatus === "true"
        ? i18n.t("dashboard.pause_bot")
        : i18n.t("dashboard.activate_bot")
    }</span>`;
    btn.className =
      newStatus === "true" ? "btn btn-warning" : "btn btn-success";

    showNotification(
      i18n.t(
        newStatus === "true"
          ? "notification.bot_activated"
          : "notification.bot_paused"
      ),
      "success"
    );
  } catch (error) {
    console.error("Error alternar bot:", error);
    showNotification(i18n.t("notification.bot_status_error"), "error");
  }
}

// Alternar grupos
async function toggleGroups() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/bot-settings`
    );
    const settings = await response.json();

    const newStatus = settings.auto_reply_groups === "true" ? "false" : "true";

    await authenticatedFetch(`${API_BASE}/api/admin/bot-settings`, {
      method: "PUT",
      body: JSON.stringify({ auto_reply_groups: newStatus }),
    });

    const btn = document.getElementById("pauseGroups");
    btn.innerHTML = `<span data-i18n="${
      newStatus === "true"
        ? "dashboard.pause_groups"
        : "dashboard.activate_groups"
    }">${
      newStatus === "true"
        ? i18n.t("dashboard.pause_groups")
        : i18n.t("dashboard.activate_groups")
    }</span>`;
    btn.className =
      newStatus === "true" ? "btn btn-secondary" : "btn btn-success";

    showNotification(
      i18n.t(
        newStatus === "true"
          ? "notification.groups_activated"
          : "notification.groups_paused"
      ),
      "success"
    );
  } catch (error) {
    console.error("Error alternar grupos:", error);
    showNotification(i18n.t("notification.groups_status_error"), "error");
  }
}

// Função corrigida para loadUsers - evitar carregar lojas desnecessariamente
async function loadUsers() {
  try {
    const response = await authenticatedFetch(`${API_BASE}/api/admin/users`);
    const users = await response.json();

    const tbody = document.querySelector("#usersTable tbody");
    if (!tbody) return;

    tbody.innerHTML = "";

    users.forEach((user) => {
      const row = document.createElement("tr");
      row.innerHTML = `
                <td>${user.username}</td>
                <td>${user.role}</td>
                <td>${user.store || "-"}</td>
                <td>${user.department || "-"}</td>
                <td>${formatDate(user.created_at)}</td>
                <td>
                    <button onclick="editUser(${
                      user.id
                    })" class="btn btn-secondary btn-sm">
                        <span data-i18n="btn.edit">${i18n.t("btn.edit")}</span>
                    </button>
                    <button onclick="deleteUser(${
                      user.id
                    })" class="btn btn-danger btn-sm">
                        <span data-i18n="btn.delete">${i18n.t(
                          "btn.delete"
                        )}</span>
                    </button>
                </td>
            `;
      tbody.appendChild(row);
    });
  } catch (error) {
    console.error("Error loading users:", error);
    showNotification(i18n.t("notification.users_load_error"), "error");

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Carregar lojas para modal - CORRIGIDO
async function loadStoresForModal() {
  try {
    // CORREÇÃO: Verificar se o modal está aberto antes de tentar carregar
    const modalContainer = document.getElementById("modalContainer");
    if (!modalContainer || !modalContainer.innerHTML.includes("newUserStore")) {
      console.log(
        "Modal de usuário não está aberto, pulando carregamento de lojas"
      );
      return;
    }

    console.log("Loading lojas para modal...");
    const response = await authenticatedFetch(`${API_BASE}/api/admin/stores`);

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }

    const stores = await response.json();
    console.log("Stores carregadas para modal:", stores);

    // CORREÇÃO: Usar requestAnimationFrame para garantir que o DOM foi atualizado
    return new Promise((resolve) => {
      requestAnimationFrame(async () => {
        let select = document.getElementById("newUserStore");
        let attempts = 0;
        const maxAttempts = 20; // Aumentar tentativas

        while (!select && attempts < maxAttempts) {
          await new Promise((resolve) => setTimeout(resolve, 50)); // Reduzir intervalo
          select = document.getElementById("newUserStore");
          attempts++;
          console.log(
            `Tentativa ${attempts}: elemento ${
              select ? "encontrado" : "not found"
            }`
          );
        }

        if (!select) {
          console.error(
            "Select newUserStore not found após múltiplas tentativas"
          );
          console.log(
            "DOM atual:",
            document.getElementById("modalContainer")?.innerHTML
          );
          resolve();
          return;
        }

        select.innerHTML = '<option value="">Select a store</option>';

        stores.forEach((store) => {
          const option = document.createElement("option");
          // USAR O NOME DA LOJA como value (não o ID)
          option.value = store.name;
          option.textContent = store.name;
          select.appendChild(option);
        });

        console.log("Store options loaded:", select.children.length);
        resolve();
      });
    });
  } catch (error) {
    console.error("Error loading stores para modal:", error);
    showNotification(
      i18n.t("notification.stores_load_error") + ": " + error.message,
      "error"
    );

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Carregar setores para modal
async function loadDepartmentsForModal() {
  try {
    const modalContainer = document.getElementById("modalContainer");
    if (
      !modalContainer ||
      !modalContainer.innerHTML.includes("newUserDepartment")
    ) {
      console.log(
        "User modal not open, skipping sector loading"
      );
      return;
    }

    console.log("Loading setores para modal...");
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/departments`
    );

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }

    const departments = await response.json();
    console.log("Departments loaded for modal:", departments);

    return new Promise((resolve) => {
      requestAnimationFrame(async () => {
        let select = document.getElementById("newUserDepartment");
        let attempts = 0;
        const maxAttempts = 20;

        while (!select && attempts < maxAttempts) {
          await new Promise((resolve) => setTimeout(resolve, 50));
          select = document.getElementById("newUserDepartment");
          attempts++;
          console.log(
            `Tentativa ${attempts}: elemento ${
              select ? "encontrado" : "not found"
            }`
          );
        }

        if (!select) {
          console.error(
            "Select newUserDepartment not found após múltiplas tentativas"
          );
          resolve();
          return;
        }

        select.innerHTML = '<option value="">Select a department</option>';

        departments.forEach((dept) => {
          const option = document.createElement("option");
          option.value = dept.name;
          option.textContent = dept.name;
          select.appendChild(option);
        });

        console.log("Loaded sector options:", select.children.length);
        resolve();
      });
    });
  } catch (error) {
    console.error("Error loading departments para modal:", error);
    showNotification(
      i18n.t("notification.departments_load_error") + ": " + error.message,
      "error"
    );

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Configure modais - MELHORADO
function setupModals() {
  const userModal = document.getElementById("userModal");
  const userForm = document.getElementById("userForm");

  if (userForm) {
    // Remover listener anterior para evitar duplicatas
    userForm.onsubmit = null;

    userForm.addEventListener("submit", async (e) => {
      e.preventDefault();

      const username = document.getElementById("newUsername").value.trim();
      const password = document.getElementById("newPassword").value;
      const storeSelect = document.getElementById("newUserStore");
      const store = storeSelect.value;

      console.log("Dados do formulário:", { username, password: "***", store });

      // Validações no frontend
      if (!username) {
        showNotification(i18n.t("notification.username_required"), "error");
        return;
      }

      if (!password) {
        showNotification(i18n.t("notification.password_required"), "error");
        return;
      }

      if (password.length < 6) {
        showNotification(i18n.t("notification.password_min_length"), "error");
        return;
      }

      if (!store) {
        showNotification(i18n.t("notification.store_required"), "error");
        return;
      }

      // Desabilitar botão durante envio
      const submitBtn = userForm.querySelector('button[type="submit"]');
      const originalText = submitBtn.textContent;
      submitBtn.disabled = true;
      submitBtn.textContent = "Saving...";

      try {
        console.log("Enviando dados:", { username, password: "***", store });

        const response = await authenticatedFetch(
          `${API_BASE}/api/admin/users`,
          {
            method: "POST",
            body: JSON.stringify({ username, password, store }),
          }
        );

        const result = await response.json();

        if (response.ok) {
          showNotification(i18n.t("notification.user_created"), "success");
          if (userModal) userModal.style.display = "none";
          userForm.reset();
          await loadUsers(); // Recarregar lista de usuários
        } else {
          console.error("Server error:", result);
          showNotification(
            result.error || i18n.t("notification.user_create_error"),
            "error"
          );
        }
      } catch (error) {
        console.error("Error creating user:", error);
        showNotification(i18n.t("notification.user_connection_error"), "error");
      } finally {
        // Reabilitar botão
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
      }
    });
  }

  // FAQ Modal
  const faqModal = document.getElementById("faqModal");
  const faqForm = document.getElementById("faqForm");

  if (faqForm) {
    faqForm.onsubmit = null;

    faqForm.addEventListener("submit", async (e) => {
      e.preventDefault();

      const question = document.getElementById("faqQuestion").value.trim();
      const answer = document.getElementById("faqAnswer").value.trim();
      const emoji = document.getElementById("faqEmoji").value.trim();
      const placeholder_key = document
        .getElementById("faqPlaceholder")
        .value.trim();

      if (!question || !answer) {
        showNotification(i18n.t("notification.faq_fields_required"), "error");
        return;
      }

      const submitBtn = faqForm.querySelector('button[type="submit"]');
      const originalText = submitBtn.textContent;
      submitBtn.disabled = true;
      submitBtn.textContent = "Saving...";

      try {
        const response = await authenticatedFetch(
          `${API_BASE}/api/admin/faqs`,
          {
            method: "POST",
            body: JSON.stringify({ question, answer, emoji, placeholder_key }),
          }
        );

        const result = await response.json();

        if (response.ok) {
          showNotification(i18n.t("notification.faq_created"), "success");
          if (faqModal) faqModal.style.display = "none";
          faqForm.reset();
          await loadFaqs();
        } else {
          showNotification(
            result.error || i18n.t("notification.faq_create_error"),
            "error"
          );
        }
      } catch (error) {
        console.error("Error creating FAQ:", error);
        showNotification("Error creating FAQ", "error");
      } finally {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
      }
    });
  }

  // Handlers para fechar modais
  document.querySelectorAll(".close").forEach((closeBtn) => {
    closeBtn.addEventListener("click", (e) => {
      e.target.closest(".modal").style.display = "none";
    });
  });

  window.addEventListener("click", (e) => {
    if (e.target.classList.contains("modal")) {
      e.target.style.display = "none";
    }
  });
}

// Inicializar Socket.IO - MELHORADO
function initializeSocket() {
  // Configurações de reconexão mais robustas
  socket = io({
    reconnection: true,
    reconnectionDelay: 2000,
    reconnectionAttempts: 10,
    timeout: 20000,
    forceNew: true,
  });

  socket.on("connect", () => {
    console.log("✅ Connected to server Socket.IO");
    showNotification(i18n.t("notification.connected_server"), "success");
  });

  socket.on("disconnect", (reason) => {
    console.log("❌ Disconnected do servidor:", reason);
    if (reason !== "io client disconnect") {
      showNotification(i18n.t("notification.connection_lost"), "warning");
    }
  });

  socket.on("reconnect", (attemptNumber) => {
    console.log("🔄 Reconnected to the server after", attemptNumber, "tentativas");
    showNotification(i18n.t("notification.reconnected_server"), "success");
    // Recarregar dados se necessário
    if (currentPage === "dashboard") {
      loadDashboard();
    }
  });

  socket.on("connect_error", (error) => {
    console.error("❌ Socket.IO connection error:", error.message);
    showNotification(i18n.t("notification.connection_error"), "error");
  });

  socket.on("qr-code", (qrCode) => {
    console.log("📱 QR Code received:", qrCode ? "Yes" : "No");
    updateQRCode(qrCode);
  });

  socket.on("connection-status", (status) => {
    console.log("📶 Connection status WhatsApp:", status);
    updateConnectionStatus(status);
  });

  socket.on("new-message", (message) => {
    console.log("📨 Nova mensagem recebida:", message.phoneNumber);
    handleNewMessage(message);
    if (currentPage === "dashboard") {
      // Atualizar métricas do dashboard
      setTimeout(() => loadDashboard(), 1000);
    }
  });

  socket.on("conversation-attended", (data) => {
    showNotification(
      `${i18n.t("notification.conversation_attended")} ${data.attendedBy} (${
        data.store
      })`,
      "info"
    );
    if (currentPage === "chat") {
      loadConversationtions();
    }
  });
}

// Função melhorada para atualizar status de conexão WhatsApp
function updateConnectionStatus(status) {
  console.log("📶 Atualizando status de conexão:", status);

  const indicator = document.querySelector(".status-indicator");
  const text = document.querySelector(".status-text");

  if (indicator && text) {
    // Limpar classes anteriores
    indicator.classList.remove("connected", "connecting", "disconnected");

    if (status.connected) {
      indicator.classList.add("connected");
      text.textContent = "Connected";
      text.style.color = "#28a745";
    } else if (status.connecting) {
      indicator.classList.add("connecting");
      text.textContent = "Connecting...";
      text.style.color = "#ffc107";
    } else {
      indicator.classList.add("disconnected");
      text.textContent = status.error || "Disconnected";
      text.style.color = "#dc3545";
    }
  }

  // Atualizar dashboard se necessário
  updateDashboardConnectionInfo(status);
}

// Função para atualizar informações de conexão no dashboard
function updateDashboardConnectionInfo(status) {
  const dashboardStatus = document.getElementById("dashboardConnectionStatus");
  if (dashboardStatus) {
    if (status.connected) {
      dashboardStatus.innerHTML = `
                <div class="alert alert-success">
                    <strong data-i18n="dashboard.whatsapp_connected">${i18n.t(
                      "dashboard.whatsapp_connected"
                    )}</strong><br>
                    <span data-i18n="dashboard.bot_operational">${i18n.t(
                      "dashboard.bot_operational"
                    )}</span>
                </div>
            `;
    } else if (status.connecting) {
      dashboardStatus.innerHTML = `
                <div class="alert alert-info">
                    <strong>🔄 ${i18n.t("dashboard.checking")}</strong><br>
                    ${i18n.t("dashboard.loading_connection")} ${
        status.qrCode ? "Escaneie o QR Code." : ""
      }
                </div>
            `;
    } else {
      dashboardStatus.innerHTML = `
                <div class="alert alert-warning">
                    <strong data-i18n="dashboard.whatsapp_disconnected">${i18n.t(
                      "dashboard.whatsapp_disconnected"
                    )}</strong><br>
                    ${status.error || i18n.t("dashboard.checking")}
                    <div class="connection-actions" style="margin-top: 10px;">
                        <button onclick="refreshQRCode()" class="btn btn-primary btn-sm">
                            🔄 ${i18n.t("dashboard.refresh_qr")}
                        </button>
                    </div>
                </div>
            `;
    }
  }
}

// Função melhorada para atualizar QR Code
function updateQRCode(qrCode) {
  const container = document.getElementById("qrCodeContainer");
  if (!container) return;

  if (qrCode) {
    container.innerHTML = `
            <div class="qr-code-wrapper">
                <img id="qrCodeImage" src="${qrCode}" alt="QR Code WhatsApp" 
                     style="max-width: 300px; height: auto; border: 1px solid #ddd; border-radius: 8px;">
                <p class="qr-instructions">Escaneie este QR Code com seu WhatsApp para conectar</p>
                <div class="connection-actions">
                    <button onclick="refreshQRCode()" class="btn btn-secondary btn-sm">
                        🔄 Update QR Code
                    </button>
                </div>
            </div>
        `;
  } else {
    container.innerHTML = `
            <div class="qr-placeholder" style="padding: 20px; text-align: center; background: #f8f9fa; border-radius: 8px; border: 2px solid #28a745;">
                <div style="color: #28a745; font-size: 24px; margin-bottom: 10px;">✅</div>
                <div style="color: #28a745; font-weight: bold;" data-i18n="dashboard.whatsapp_connected">${i18n.t(
                  "dashboard.whatsapp_connected"
                )}</div>
                <div style="color: #6c757d; font-size: 14px; margin-top: 5px;" data-i18n="dashboard.bot_ready">${i18n.t(
                  "dashboard.bot_ready"
                )}</div>
                <div class="connection-actions" style="margin-top: 15px;">
                    <button onclick="disconnectWhatsApp()" class="btn btn-warning btn-sm">
                        🔌 <span data-i18n="dashboard.disconnect_whatsapp">${i18n.t(
                          "dashboard.disconnect_whatsapp"
                        )}</span>
                    </button>
                </div>
            </div>
        `;
  }
}

// Função melhorada para refresh do QR Code
async function refreshQRCode() {
  try {
    const refreshBtn = document.querySelector(
      'button[onclick="refreshQRCode()"]'
    );
    if (refreshBtn) {
      refreshBtn.disabled = true;
      refreshBtn.textContent = "🔄 Reconnecting...";
    }

    showNotification("Starting WhatsApp reconnection...", "info");

    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/whatsapp-reconnect`,
      {
        method: "POST",
      }
    );

    const result = await response.json();

    if (response.ok) {
      showNotification(
        "Reconnection started. New QR Code will be generated shortly.",
        "success"
      );
    } else {
      showNotification(result.error || "Error reconnecting WhatsApp", "error");
    }
  } catch (error) {
    console.error("Error reconnecting WhatsApp:", error);
    showNotification("Connection error when reconnecting WhatsApp", "error");
  } finally {
    // Reabilitar botão após 5 segundos
    setTimeout(() => {
      const refreshBtn = document.querySelector(
        'button[onclick="refreshQRCode()"]'
      );
      if (refreshBtn) {
        refreshBtn.disabled = false;
        refreshBtn.textContent = "🔄 Update QR Code";
      }
    }, 5000);
  }
}

// Função para desconectar WhatsApp
async function disconnectWhatsApp() {
  if (
    !confirm(
      "Are you sure you want to disconnect WhatsApp? This will stop all automatic responses."
    )
  ) {
    return;
  }

  try {
    const disconnectBtn = document.querySelector(
      'button[onclick="disconnectWhatsApp()"]'
    );
    if (disconnectBtn) {
      disconnectBtn.disabled = true;
      disconnectBtn.textContent = "🔌 Disconnecting...";
    }

    showNotification("Desconectando WhatsApp...", "info");

    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/whatsapp-disconnect`,
      {
        method: "POST",
      }
    );

    const result = await response.json();

    if (response.ok) {
      showNotification("WhatsApp disconnected successfully", "success");
    } else {
      showNotification(result.error || "Error disconnecting WhatsApp", "error");
    }
  } catch (error) {
    console.error("Error disconnecting WhatsApp:", error);
    showNotification("Error disconnecting WhatsApp", "error");
  }
}

// Função para carregar status do WhatsApp
async function loadWhatsAppStatus() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/whatsapp-status`
    );

    if (response.ok) {
      const status = await response.json();
      updateConnectionStatus(status);

      if (status.qrCode) {
        updateQRCode(status.qrCode);
      } else if (status.connected) {
        updateQRCode(null); // Mostrar status conectado
      }
    }
  } catch (error) {
    console.error("Error loading WhatsApp status:", error);
  }
}

// Função para inicializar monitoramento
function startConnectionMonitoring() {
  setInterval(async () => {
    if (currentPage === "dashboard") {
      try {
        await loadWhatsAppStatus();
      } catch (error) {
        console.error("Error in monitoramento:", error);
      }
    }
  }, 30000); // Verificar a cada 30 segundos
}

// Add estilos CSS para melhor visual
const additionalStyles = `
<style>
.qr-code-wrapper {
    text-align: center;
    padding: 20px;
    background: #f8f9fa;
    border-radius: 8px;
    border: 1px solid #dee2e6;
}

.qr-instructions {
    margin: 15px 0 10px 0;
    font-size: 14px;
    color: #6c757d;
}

.status-indicator {
    width: 12px;
    height: 12px;
    border-radius: 50%;
    display: inline-block;
    margin-right: 8px;
    background: #dc3545;
    animation: pulse 2s infinite;
}

.status-indicator.connected {
    background: #28a745;
    animation: none;
}

.status-indicator.connecting {
    background: #ffc107;
    animation: pulse 1s infinite;
}

.status-indicator.disconnected {
    background: #dc3545;
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0% { opacity: 1; }
    50% { opacity: 0.5; }
    100% { opacity: 1; }
}

.connection-actions {
    margin-top: 15px;
    text-align: center;
}

.connection-actions .btn {
    margin: 0 5px;
}

.alert {
    padding: 15px;
    margin-bottom: 20px;
    border: 1px solid transparent;
    border-radius: 4px;
}

.alert-success {
    color: #155724;
    background-color: #d4edda;
    border-color: #c3e6cb;
}

.alert-info {
    color: #0c5460;
    background-color: #d1ecf1;
    border-color: #bee5eb;
}

.alert-warning {
    color: #856404;
    background-color: #fff3cd;
    border-color: #ffeeba;
}

.alert-danger {
    color: #721c24;
    background-color: #f8d7da;
    border-color: #f5c6cb;
}

.notification {
    z-index: 1001;
}

.btn:disabled {
    opacity: 0.6;
    cursor: not-allowed;
}

@media (max-width: 768px) {
    .qr-code-wrapper {
        padding: 15px;
    }
    
    .qr-code-wrapper img {
        max-width: 250px;
    }
}
</style>
`;

// Add estilos ao head quando a página carregar
document.addEventListener("DOMContentLoaded", function () {
  if (!document.getElementById("admin-additional-styles")) {
    const styleElement = document.createElement("div");
    styleElement.id = "admin-additional-styles";
    styleElement.innerHTML = additionalStyles;
    document.head.appendChild(styleElement);
  }
});

// Melhorar a função de inicialização admin
async function initializeAdmin() {
  const user = getCurrentUser();
  const userDisplay = document.getElementById("currentUser");
  if (userDisplay) userDisplay.textContent = user.username;

  // Inicializar componentes
  initializeSocket();
  setupNavigation();
  setupModals();
  setupControlButtons();
  setupMobileMenu();

  // Inicializar monitoramento
  setTimeout(() => {
    startConnectionMonitoring();
  }, 2000);

  // Carregar página inicial
  const initialPage = window.location.hash.substring(1) || "dashboard";
  await showPage(initialPage);
}

// Add ao objeto global para debug
window.whatsappAdmin = {
  refreshQR: refreshQRCode,
  disconnect: disconnectWhatsApp,
  loadStatus: loadWhatsAppStatus,
  testConnection: async () => {
    try {
      const response = await authenticatedFetch(
        `${API_BASE}/api/admin/whatsapp-test`,
        {
          method: "POST",
          body: JSON.stringify({
            phoneNumber: "5527999999999",
            message: "Message de teste do painel admin",
          }),
        }
      );

      const result = await response.json();
      console.log("Resultado teste:", result);
      showNotification(
        result.message || result.error,
        response.ok ? "success" : "error"
      );
    } catch (error) {
      console.error("Error in teste:", error);
      showNotification("Error in connection test", "error");
    }
  },
};

// Abrir modal de usuário
function openUserModal() {
  const modal = document.getElementById("userModal");
  if (modal) modal.style.display = "block";
}

// Abrir modal de FAQ
function openFaqModal() {
  const modal = document.getElementById("faqModal");
  if (modal) modal.style.display = "block";
}

// Carregar conversas
async function loadConversationtions() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/chat/conversations`
    );
    const conversations = await response.json();

    const container = document.getElementById("conversationsList");
    const countElement = document.getElementById("conversationsCount");

    if (!container) return;

    // Atualizar contador
    if (countElement) {
      countElement.textContent = conversations.length || 0;
    }

    container.innerHTML = "";

    if (conversations.length === 0) {
      container.innerHTML =
        '<div style="padding: 20px; text-align: center; color: #999;">No conversations found</div>';
      return;
    }

    conversations.forEach((conversation) => {
      // Debug: ver estrutura completa da conversa
      console.log("Conversation completa:", conversation);

      const item = document.createElement("div");
      item.className = "conversation-item";
      item.onclick = (e) => selectConversationtion(conversation.id, e);

      // Determinar status da conversa - verificar múltiplos campos possíveis
      let statusBadge = "";
      let statusClass = "";

      // Verificar se está fechada
      if (
        conversation.status === "closed" ||
        conversation.status === "finalizada"
      ) {
        statusBadge =
          '<span class="status-badge status-closed">Finalizada</span>';
        statusClass = "conversation-closed";
      }
      // Verificar se está atendida - múltiplas possibilidades de campos
      else if (
        conversation.assigned_user_id ||
        conversation.assigned_username ||
        conversation.assignedUserId ||
        conversation.assignedUsername ||
        conversation.attended_by ||
        conversation.attendedBy ||
        conversation.status === "attended" ||
        conversation.status === "em_atendimento"
      ) {
        const attendantName =
          conversation.assigned_username ||
          conversation.assignedUsername ||
          conversation.attended_by ||
          conversation.attendedBy ||
          "Operador";
        statusBadge = `<span class="status-badge status-attended" data-i18n="conversations.attended_by">${i18n.t(
          "conversations.attended_by"
        )} ${attendantName}</span>`;
        statusClass = "conversation-attended";
      }
      // Caso contrário, está aguardando
      else {
        statusBadge =
          '<span class="status-badge status-waiting">Waiting</span>';
        statusClass = "conversation-waiting";
      }

      item.classList.add(statusClass);

      item.innerHTML = `
                <div class="conversation-contact">${formatPhone(
                  conversation.phone_number
                )}</div>
                <div class="conversation-preview">${
                  conversation.last_message || "No messages"
                }</div>
                <div class="conversation-meta">
                    <div class="conversation-time">${formatDate(
                      conversation.last_message_time
                    )}</div>
                    ${statusBadge}
                </div>
            `;

      container.appendChild(item);
    });
  } catch (error) {
    console.error("Error loading conversations:", error);
    showNotification("Error loading conversations", "error");

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Voltar para lista de conversas (mobile)
function backToConversationtionsList() {
  const conversationsList = document.querySelector(".conversations-list");
  const chatArea = document.querySelector(".chat-area");
  const backButton = document.querySelector(".back-to-conversations");

  if (conversationsList) conversationsList.style.display = "flex";
  if (chatArea) chatArea.classList.remove("mobile-show");
  if (backButton) backButton.style.display = "none";
}

// Selecionar conversa
async function selectConversationtion(conversationId, event) {
  currentConversationtionId = conversationId;

  document.querySelectorAll(".conversation-item").forEach((item) => {
    item.classList.remove("active");
  });
  event.target.closest(".conversation-item").classList.add("active");

  // No mobile, ocultar lista e mostrar chat
  if (window.innerWidth <= 1024) {
    const conversationsList = document.querySelector(".conversations-list");
    const chatArea = document.querySelector(".chat-area");
    const backButton = document.querySelector(".back-to-conversations");

    if (conversationsList) conversationsList.style.display = "none";
    if (chatArea) chatArea.classList.add("mobile-show");
    if (backButton) backButton.style.display = "inline-block";
  }

  await loadMessages(conversationId);
}

// Carregar mensagens (modo somente leitura para admin)
async function loadMessages(conversationId) {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/chat/conversations/${conversationId}/messages`
    );
    const data = await response.json();

    const messages = data.messages || data;
    const conversationInfo = data.conversation || {};

    const container = document.getElementById("messagesContainer");
    if (!container) {
      console.error("Container de mensagens not found!");
      return;
    }

    // Atualizar cabeçalho com informações da conversa
    const chatHeaderTitle = document.getElementById("chatHeaderTitle");
    const chatHeaderAttendant = document.getElementById("chatHeaderAttendant");

    if (chatHeaderTitle) {
      chatHeaderTitle.textContent = conversationInfo.phone_number || "Conversation";
    }

    if (chatHeaderAttendant) {
      if (conversationInfo.assigned_username) {
        chatHeaderAttendant.innerHTML = `
                    <span style="color: #28a745; font-size: 14px;">
                        👤 <span data-i18n="conversations.attended_by">${i18n.t(
                          "conversations.attended_by"
                        )}</span> <strong>${
          conversationInfo.assigned_username
        }</strong>
                    </span>
                `;
      } else {
        chatHeaderAttendant.innerHTML = `
                    <span style="color: #ffc107; font-size: 14px;">
                        ⏳ Waiting atendimento
                    </span>
                `;
      }
    }

    // Limpar container
    container.innerHTML = "";

    if (messages.length === 0) {
      container.innerHTML =
        '<div class="no-messages"><p>Nenhuma mensagem nesta conversa</p></div>';
      return;
    }

    messages.forEach((message) => {
      const messageDiv = document.createElement("div");
      messageDiv.className = `message ${
        message.is_from_bot || message.sender_user_id ? "sent" : "received"
      }`;

      let messageContent = message.message_text || "";

      // Tratar mensagens com mídia
      if (message.media_url) {
        if (message.message_type === "image") {
          messageContent = `<img src="${message.media_url}" alt="Imagem" style="max-width: 300px; border-radius: 8px;">`;
        } else if (message.message_type === "video") {
          messageContent = `<video controls style="max-width: 300px; border-radius: 8px;"><source src="${message.media_url}"></video>`;
        } else if (message.message_type === "audio") {
          messageContent = `<audio controls><source src="${message.media_url}"></audio>`;
        } else if (message.message_type === "document") {
          messageContent = `<a href="${message.media_url}" target="_blank">📄 Documento</a>`;
        }
      }

      // Identificar quem enviou
      let senderLabel = "";
      if (message.is_from_bot) {
        senderLabel = '<span class="message-label bot-label">🤖 Bot</span>';
      } else if (message.sender_username) {
        senderLabel = `<span class="message-label operator-label">👤 ${message.sender_username}</span>`;
      } else {
        senderLabel =
          '<span class="message-label client-label">👤 Cliente</span>';
      }

      messageDiv.innerHTML = `
                <div class="message-bubble">
                    ${senderLabel}
                    <div class="message-text">${messageContent}</div>
                    <div class="message-time">${formatDate(
                      message.timestamp
                    )}</div>
                </div>
            `;

      container.appendChild(messageDiv);
    });

    // Scroll para o final
    setTimeout(() => {
      container.scrollTop = container.scrollHeight;
    }, 100);
  } catch (error) {
    console.error("Error loading messages:", error);
    showNotification("Error loading messages", "error");

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Enviar mensagem (DESABILITADO PARA ADMIN - MODO SOMENTE LEITURA)
async function sendMessage() {
  showNotification(
    "Admin cannot send messages. Read-only mode.",
    "warning"
  );
  return;
}

// Função loadStores removida - versão duplicada (usar a versão corrigida abaixo)

// Atualizar loja
async function updateStore(event, storeId) {
  event.preventDefault();

  const form = event.target;
  const formData = new FormData(form);

  const data = {
    name: form.closest(".store-card").querySelector("h3").textContent,
    address: formData.get("address"),
    hours: formData.get("hours"),
    promotions: formData.get("promotions"),
  };

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/stores/${storeId}`,
      {
        method: "PUT",
        body: JSON.stringify(data),
      }
    );

    if (response.ok) {
      showNotification("Store updated successfully", "success");
    } else {
      const error = await response.json();
      showNotification(error.error || "Error updating store", "error");
    }
  } catch (error) {
    console.error("Error updating store:", error);
    showNotification("Error updating store", "error");
  }
}

// Carregar FAQs
async function loadFaqs() {
  try {
    // Carregar mensagens de boas-vindas
    await loadWelcomeMessages();

    const response = await authenticatedFetch(`${API_BASE}/api/admin/faqs`);
    const faqs = await response.json();

    const tbody = document.querySelector("#faqsTable tbody");
    if (!tbody) return;

    tbody.innerHTML = "";

    faqs.forEach((faq) => {
      const row = document.createElement("tr");
      row.innerHTML = `
                <td>${faq.question}</td>
                <td>${faq.answer.substring(0, 50)}...</td>
                <td>${faq.emoji || "-"}</td>
                <td>
                    <button onclick="editFaq(${
                      faq.id
                    })" class="btn btn-secondary btn-sm">
                        <span data-i18n="btn.edit">${i18n.t("btn.edit")}</span>
                    </button>
                    <button onclick="deleteFaq(${
                      faq.id
                    })" class="btn btn-danger btn-sm">
                        <span data-i18n="btn.delete">${i18n.t(
                          "btn.delete"
                        )}</span>
                    </button>
                </td>
            `;
      tbody.appendChild(row);
    });
  } catch (error) {
    console.error("Error loading FAQs:", error);
    showNotification("Error loading FAQs", "error");

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Carregar configurações
async function loadSettings() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/bot-settings`
    );
    const settings = await response.json();

    const botNameInput = document.getElementById("botName");
    const botPersonaInput = document.getElementById("botPersona");

    if (botNameInput) botNameInput.value = settings.bot_name || "";
    if (botPersonaInput) botPersonaInput.value = settings.bot_persona || "";

    const settingsForm = document.getElementById("settingsForm");
    if (settingsForm) {
      // Remover listener anterior se existir
      settingsForm.onsubmit = null;

      settingsForm.onsubmit = async (e) => {
        e.preventDefault();

        const formData = new FormData(settingsForm);
        const data = Object.fromEntries(formData);

        try {
          const response = await authenticatedFetch(
            `${API_BASE}/api/admin/bot-settings`,
            {
              method: "PUT",
              body: JSON.stringify(data),
            }
          );

          if (response.ok) {
            showNotification("Settings saved successfully", "success");
          } else {
            const error = await response.json();
            showNotification(
              error.error || "Error saving settings",
              "error"
            );
          }
        } catch (error) {
          console.error("Error saving settings:", error);
          showNotification("Error saving settings", "error");
        }
      };
    }
  } catch (error) {
    console.error("Error loading settings:", error);
    showNotification("Error loading settings", "error");

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Deletar usuário
async function deleteUser(userId) {
  if (!confirm(i18n.t("notification.confirm_delete_user"))) {
    return;
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/users/${userId}`,
      {
        method: "DELETE",
      }
    );

    if (response.ok) {
      showNotification(i18n.t("notification.user_deleted"), "success");
      loadUsers();
    } else {
      const error = await response.json();
      showNotification(
        error.error || i18n.t("notification.user_delete_error"),
        "error"
      );
    }
  } catch (error) {
    console.error("Error deleting user:", error);
    showNotification(i18n.t("notification.user_delete_error"), "error");
  }
}

async function editUser(userId) {
  try {
    // Search dados do usuário
    const response = await authenticatedFetch(`${API_BASE}/api/admin/users`);
    const users = await response.json();
    const user = users.find((u) => u.id === userId);

    if (!user) {
      showNotification(i18n.t("notification.error"), "error");
      return;
    }

    const modalContainer = document.getElementById("modalContainer");
    if (!modalContainer) {
      console.error("Elemento modalContainer not found no DOM");
      showNotification("Error: Modal container not found", "error");
      return;
    }

    const hasStore = user.store && user.store !== "";
    const hasDepartment = user.department && user.department !== "";

    const modalHtml = `
            <div id="editUserModal" class="modal" style="display: block;">
                <div class="modal-content">
                    <div class="modal-header">
                        <h3 data-i18n="users.edit">Edit User</h3>
                        <span class="close" onclick="closeEditUserModal()">&times;</span>
                    </div>
                    <form id="editUserForm">
                        <div class="form-group">
                            <label>Username:</label>
                            <input type="text" id="editUsername" value="${
                              user.username
                            }" required>
                        </div>
                        <div class="form-group">
                            <label>New Password (leave blank to keep current):</label>
                            <input type="password" id="editPassword" placeholder="New password (optional)">
                        </div>
                        <div class="form-group">
                            <label>Link to:</label>
                            <div style="margin-bottom: 10px;">
                                <label style="margin-right: 20px;">
                                    <input type="radio" name="editLinkType" value="store" ${
                                      hasStore ? "checked" : ""
                                    } onchange="toggleEditUserLinkType()"> Store
                                </label>
                                <label>
                                    <input type="radio" name="editLinkType" value="department" ${
                                      hasDepartment ? "checked" : ""
                                    } onchange="toggleEditUserLinkType()"> Department
                                </label>
                            </div>
                        </div>
                        <div class="form-group" id="editStoreSelectGroup" style="display: ${
                          hasStore ? "block" : "none"
                        };">
                            <label>Store:</label>
                            <select id="editUserStore">
                                <option value="">Selecione uma loja</option>
                            </select>
                        </div>
                        <div class="form-group" id="editDepartmentSelectGroup" style="display: ${
                          hasDepartment ? "block" : "none"
                        };">
                            <label>Department:</label>
                            <select id="editUserDepartment">
                                <option value="">Select a department</option>
                            </select>
                        </div>
                        <div class="modal-footer">
                            <button type="button" onclick="closeEditUserModal()" class="btn btn-secondary">
                                <span data-i18n="btn.cancel">Cancel</span>
                            </button>
                            <button type="submit" class="btn btn-primary">
                                <span data-i18n="btn.save">Save</span>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        `;

    modalContainer.innerHTML = modalHtml;

    // Atualizar traduções
    i18n.updatePageTranslations();

    // Carregar lojas e setores
    setTimeout(async () => {
      await loadStoresForEditModal(user.store);
      await loadDepartmentsForEditModal(user.department);
    }, 0);

    // Configure formulário
    setTimeout(() => {
      const form = document.getElementById("editUserForm");
      if (form) {
        form.onsubmit = async (e) => {
          e.preventDefault();
          await updateUser(userId);
        };
      }
    }, 10);
  } catch (error) {
    console.error("Error editar usuário:", error);
    showNotification(i18n.t("notification.error"), "error");
  }
}

function closeEditUserModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

async function updateUser(userId) {
  const username = document.getElementById("editUsername").value;
  const password = document.getElementById("editPassword").value;
  const linkType = document.querySelector(
    'input[name="editLinkType"]:checked'
  ).value;

  let store = null;
  let department = null;

  if (linkType === "store") {
    store = document.getElementById("editUserStore").value;
    if (!store) {
      showNotification(i18n.t("notification.store_required"), "error");
      return;
    }
  } else {
    department = document.getElementById("editUserDepartment").value;
    if (!department) {
      showNotification("Select a department", "error");
      return;
    }
  }

  try {
    const body = { username, store, department };
    if (password && password.trim() !== "") {
      body.password = password;
    }

    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/users/${userId}`,
      {
        method: "PUT",
        body: JSON.stringify(body),
      }
    );

    if (response.ok) {
      showNotification(i18n.t("notification.updated"), "success");
      closeEditUserModal();
      await loadUsers();
    } else {
      const error = await response.json();
      showNotification(
        error.error || i18n.t("notification.user_create_error"),
        "error"
      );
    }
  } catch (error) {
    console.error("Error updating user:", error);
    showNotification(i18n.t("notification.user_create_error"), "error");
  }
}

function toggleEditUserLinkType() {
  const linkType = document.querySelector(
    'input[name="editLinkType"]:checked'
  ).value;
  const storeGroup = document.getElementById("editStoreSelectGroup");
  const departmentGroup = document.getElementById("editDepartmentSelectGroup");

  if (linkType === "store") {
    storeGroup.style.display = "block";
    departmentGroup.style.display = "none";
    document.getElementById("editUserDepartment").value = "";
  } else {
    storeGroup.style.display = "none";
    departmentGroup.style.display = "block";
    document.getElementById("editUserStore").value = "";
  }
}

async function loadStoresForEditModal(currentStore) {
  try {
    const response = await authenticatedFetch(`${API_BASE}/api/admin/stores`);
    if (!response.ok) {
      throw new Error("Error loading stores");
    }

    const stores = await response.json();
    const select = document.getElementById("editUserStore");

    if (!select) return;

    select.innerHTML = '<option value="">Selecione uma loja</option>';
    stores.forEach((store) => {
      const option = document.createElement("option");
      option.value = store.name;
      option.textContent = store.name;
      if (store.name === currentStore) {
        option.selected = true;
      }
      select.appendChild(option);
    });
  } catch (error) {
    console.error("Error loading stores:", error);
  }
}

async function loadDepartmentsForEditModal(currentDepartment) {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/departments`
    );
    if (!response.ok) {
      throw new Error("Error loading departments");
    }

    const departments = await response.json();
    const select = document.getElementById("editUserDepartment");

    if (!select) return;

    select.innerHTML = '<option value="">Select a department</option>';
    departments.forEach((dept) => {
      const option = document.createElement("option");
      option.value = dept.name;
      option.textContent = dept.name;
      if (dept.name === currentDepartment) {
        option.selected = true;
      }
      select.appendChild(option);
    });
  } catch (error) {
    console.error("Error loading departments:", error);
  }
}

async function editUser(userId) {
  try {
    // Search dados do usuário
    const response = await authenticatedFetch(`${API_BASE}/api/admin/users`);
    const users = await response.json();
    const user = users.find((u) => u.id === userId);

    if (!user) {
      showNotification(i18n.t("notification.error"), "error");
      return;
    }

    const modalContainer = document.getElementById("modalContainer");
    if (!modalContainer) {
      console.error("Elemento modalContainer not found no DOM");
      showNotification("Error: Modal container not found", "error");
      return;
    }

    const hasStore = user.store && user.store !== "";
    const hasDepartment = user.department && user.department !== "";

    const modalHtml = `
            <div id="editUserModal" class="modal" style="display: block;">
                <div class="modal-content">
                    <div class="modal-header">
                        <h3 data-i18n="users.edit">Edit User</h3>
                        <span class="close" onclick="closeEditUserModal()">&times;</span>
                    </div>
                    <form id="editUserForm">
                        <div class="form-group">
                            <label>Username:</label>
                            <input type="text" id="editUsername" value="${
                              user.username
                            }" required>
                        </div>
                        <div class="form-group">
                            <label>New Password (leave blank to keep current):</label>
                            <input type="password" id="editPassword" placeholder="New password (optional)">
                        </div>
                        <div class="form-group">
                            <label>Link to:</label>
                            <div style="margin-bottom: 10px;">
                                <label style="margin-right: 20px;">
                                    <input type="radio" name="editLinkType" value="store" ${
                                      hasStore ? "checked" : ""
                                    } onchange="toggleEditUserLinkType()"> Store
                                </label>
                                <label>
                                    <input type="radio" name="editLinkType" value="department" ${
                                      hasDepartment ? "checked" : ""
                                    } onchange="toggleEditUserLinkType()"> Department
                                </label>
                            </div>
                        </div>
                        <div class="form-group" id="editStoreSelectGroup" style="display: ${
                          hasStore ? "block" : "none"
                        };">
                            <label>Store:</label>
                            <select id="editUserStore">
                                <option value="">Selecione uma loja</option>
                            </select>
                        </div>
                        <div class="form-group" id="editDepartmentSelectGroup" style="display: ${
                          hasDepartment ? "block" : "none"
                        };">
                            <label>Department:</label>
                            <select id="editUserDepartment">
                                <option value="">Select a department</option>
                            </select>
                        </div>
                        <div class="modal-footer">
                            <button type="button" onclick="closeEditUserModal()" class="btn btn-secondary">
                                <span data-i18n="btn.cancel">Cancel</span>
                            </button>
                            <button type="submit" class="btn btn-primary">
                                <span data-i18n="btn.save">Save</span>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        `;

    modalContainer.innerHTML = modalHtml;

    // Atualizar traduções
    i18n.updatePageTranslations();

    // Carregar lojas e setores
    setTimeout(async () => {
      await loadStoresForEditModal(user.store);
      await loadDepartmentsForEditModal(user.department);
    }, 0);

    // Configure formulário
    setTimeout(() => {
      const form = document.getElementById("editUserForm");
      if (form) {
        form.onsubmit = async (e) => {
          e.preventDefault();
          await updateUser(userId);
        };
      }
    }, 10);
  } catch (error) {
    console.error("Error editar usuário:", error);
    showNotification(i18n.t("notification.error"), "error");
  }
}

function closeEditUserModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

async function updateUser(userId) {
  const username = document.getElementById("editUsername").value;
  const password = document.getElementById("editPassword").value;
  const linkType = document.querySelector(
    'input[name="editLinkType"]:checked'
  ).value;

  let store = null;
  let department = null;

  if (linkType === "store") {
    store = document.getElementById("editUserStore").value;
    if (!store) {
      showNotification(i18n.t("notification.store_required"), "error");
      return;
    }
  } else {
    department = document.getElementById("editUserDepartment").value;
    if (!department) {
      showNotification("Select a department", "error");
      return;
    }
  }

  try {
    const body = { username, store, department };
    if (password && password.trim() !== "") {
      body.password = password;
    }

    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/users/${userId}`,
      {
        method: "PUT",
        body: JSON.stringify(body),
      }
    );

    if (response.ok) {
      showNotification(i18n.t("notification.updated"), "success");
      closeEditUserModal();
      await loadUsers();
    } else {
      const error = await response.json();
      showNotification(
        error.error || i18n.t("notification.user_create_error"),
        "error"
      );
    }
  } catch (error) {
    console.error("Error updating user:", error);
    showNotification(i18n.t("notification.user_create_error"), "error");
  }
}

function toggleEditUserLinkType() {
  const linkType = document.querySelector(
    'input[name="editLinkType"]:checked'
  ).value;
  const storeGroup = document.getElementById("editStoreSelectGroup");
  const departmentGroup = document.getElementById("editDepartmentSelectGroup");

  if (linkType === "store") {
    storeGroup.style.display = "block";
    departmentGroup.style.display = "none";
    document.getElementById("editUserDepartment").value = "";
  } else {
    storeGroup.style.display = "none";
    departmentGroup.style.display = "block";
    document.getElementById("editUserStore").value = "";
  }
}

async function loadStoresForEditModal(currentStore) {
  try {
    const response = await authenticatedFetch(`${API_BASE}/api/admin/stores`);
    if (!response.ok) {
      throw new Error("Error loading stores");
    }

    const stores = await response.json();
    const select = document.getElementById("editUserStore");

    if (!select) return;

    select.innerHTML = '<option value="">Selecione uma loja</option>';
    stores.forEach((store) => {
      const option = document.createElement("option");
      option.value = store.name;
      option.textContent = store.name;
      if (store.name === currentStore) {
        option.selected = true;
      }
      select.appendChild(option);
    });
  } catch (error) {
    console.error("Error loading stores:", error);
  }
}

async function loadDepartmentsForEditModal(currentDepartment) {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/departments`
    );
    if (!response.ok) {
      throw new Error("Error loading departments");
    }

    const departments = await response.json();
    const select = document.getElementById("editUserDepartment");

    if (!select) return;

    select.innerHTML = '<option value="">Select a department</option>';
    departments.forEach((dept) => {
      const option = document.createElement("option");
      option.value = dept.name;
      option.textContent = dept.name;
      if (dept.name === currentDepartment) {
        option.selected = true;
      }
      select.appendChild(option);
    });
  } catch (error) {
    console.error("Error loading departments:", error);
  }
}

// Deletar FAQ
async function deleteFaq(faqId) {
  if (!confirm("Are you sure you want to delete this FAQ?")) {
    return;
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/faqs/${faqId}`,
      {
        method: "DELETE",
      }
    );

    if (response.ok) {
      showNotification("FAQ deleted successfully", "success");
      loadFaqs();
    } else {
      const error = await response.json();
      showNotification(error.error, "error");
    }
  } catch (error) {
    console.error("Error deleting FAQ:", error);
    showNotification("Error deleting FAQ", "error");
  }
}

// Edit FAQ
function editFaq(faqId) {
  // Implementar lógica de edição se necessário
  showNotification("Edit FAQ function not implemented", "warning");
}

// Manipular nova mensagem
function handleNewMessage(message) {
  if (currentPage === "chat") {
    loadConversationtions();
  }

  if (
    currentConversationtionId &&
    message.conversationId === currentConversationtionId
  ) {
    loadMessages(currentConversationtionId);
  }
}

// Atualizar dashboard automaticamente
function updateDashboard() {
  if (currentPage === "dashboard") {
    loadDashboard();
  }
}

// Função logout global
function logout() {
  localStorage.removeItem("authToken");
  localStorage.removeItem("currentUser");
  window.location.href = "/";
}
// Carregar página de envio em massa
// Função corrigida para loadMassMessaging - sem tentar carregar campanhas se endpoint não existir
async function loadMassMessaging() {
  try {
    console.log("Loading página de envio em massa...");

    // Primeiro verificar se o endpoint de campanhas existe
    let campaignsLoaded = false;
    try {
      const response = await authenticatedFetch(
        `${API_BASE}/api/admin/campaigns`,
        {
          method: "HEAD", // Só verificar se existe, sem carregar dados
        }
      );
      if (response.ok) {
        await loadCampaigns();
        campaignsLoaded = true;
      }
    } catch (error) {
      console.warn("Campaign endpoint not available:", error.message);
    }

    if (!campaignsLoaded) {
      // Se não conseguiu carregar campanhas, mostrar mensagem na tabela
      const tbody = document.getElementById("campaignsTableBody");
      if (tbody) {
        tbody.innerHTML = `
                    <tr>
                        <td colspan="7" class="text-center">
                            <div style="padding: 20px;">
                                <p><strong>Funcionalidade de Campanhas em Desenvolvimento</strong></p>
                                <p>Esta funcionalidade será implementada em breve.</p>
                            </div>
                        </td>
                    </tr>
                `;
      }
    }

    // Carregar grupos de contatos (isso deve funcionar)
    await loadContactGroups();

    // Atualizar selects
    await updateGroupSelects();

    console.log("Página de envio em massa carregada");
  } catch (error) {
    console.error("Error loading mass messaging:", error);
    showNotification(
      "Error loading mass messaging page: " + error.message,
      "error"
    );

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Carregar página de pagamentos
async function loadPayments() {
  try {
    showNotification(i18n.t("payments.loading"), "info");
    // Implementar carregamento de pagamentos
    console.log("Página de pagamentos carregada");
  } catch (error) {
    console.error("Error loading payments:", error);
    showNotification("Error loading payments", "error");

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Carregar página de traduções
async function loadTranslations() {
  try {
    // Verificar se o sistema i18n está disponível
    if (typeof i18n === "undefined") {
      console.error("Sistema i18n not found");
      showNotification("Translation system not initialized", "error");
      return;
    }

    // Inicializar sistema de traduções se a função existir
    if (typeof initTranslationSystem === "function") {
      initTranslationSystem();
    } else {
      // Implementação básica se a função não existir
      if (typeof loadTranslationData === "function") {
        loadTranslationData();
      }
      if (typeof loadLanguageSelectors === "function") {
        loadLanguageSelectors();
      }
      if (typeof loadLanguagesTable === "function") {
        loadLanguagesTable();
      }
      if (typeof loadStringsTable === "function") {
        loadStringsTable();
      }
      if (typeof updateTranslationStats === "function") {
        updateTranslationStats();
      }
    }

    // Carregar traduções salvas do localStorage
    const savedTranslations = localStorage.getItem("i18nTranslations");
    if (savedTranslations) {
      try {
        const parsed = JSON.parse(savedTranslations);
        // Mesclar traduções salvas com as padrão
        Object.keys(parsed).forEach((lang) => {
          if (!i18n.translations[lang]) {
            i18n.translations[lang] = {};
          }
          Object.assign(i18n.translations[lang], parsed[lang]);
        });

        // Atualizar interface se necessário
        if (typeof loadLanguageSelectors === "function") {
          loadLanguageSelectors();
          loadLanguagesTable();
          loadStringsTable();
          updateTranslationStats();
        }
      } catch (e) {
        console.error("Error loading saved translations:", e);
      }
    }

    console.log("Translation system loaded successfully");
  } catch (error) {
    console.error("Error loading translations:", error);
    showNotification("Error loading translations: " + error.message, "error");

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Carregar página de configuração IA
async function loadAiConfig() {
  try {
    showNotification(i18n.t("ai_config.loading"), "info");

    // Carregar configurações de IA do backend
    const response = await authenticatedFetch(
      `${API_BASE}/api/ai-config/settings`
    );

    if (response.ok) {
      const configs = await response.json();

      // O backend retorna um array de configurações
      if (Array.isArray(configs) && configs.length > 0) {
        // Search configuração ativa ou a primeira
        const activeConfig = configs.find((c) => c.active) || configs[0];

        // Save ID da configuração para edição
        window.currentAiConfigId = activeConfig.id;

        // Preencher formulário baseado no provider
        if (activeConfig.provider === "deepseek") {
          // No preencher API Key por segurança (ela é criptografada)
          if (document.getElementById("deepseekModel")) {
            document.getElementById("deepseekModel").value =
              activeConfig.model_name || "deepseek-chat";
          }
          if (document.getElementById("deepseekPersonaName")) {
            document.getElementById("deepseekPersonaName").value =
              activeConfig.persona_name || "Assistente";
          }
          if (document.getElementById("deepseekPersonaDescription")) {
            document.getElementById("deepseekPersonaDescription").value =
              activeConfig.persona_description ||
              "Intelligent virtual assistant";
          }
          if (document.getElementById("deepseekSystemPrompt")) {
            document.getElementById("deepseekSystemPrompt").value =
              activeConfig.system_prompt || "";
          }
          if (document.getElementById("deepseekWorkingHours")) {
            const hours = `${activeConfig.business_hours_start || "00:00"}-${
              activeConfig.business_hours_end || "23:59"
            }`;
            document.getElementById("deepseekWorkingHours").value = hours;
          }
          if (document.getElementById("deepseekTemperature")) {
            document.getElementById("deepseekTemperature").value =
              activeConfig.temperature || 0.7;
          }
          if (document.getElementById("deepseekMaxTokens")) {
            document.getElementById("deepseekMaxTokens").value =
              activeConfig.max_tokens || 1000;
          }
        }

        console.log(
          "AI settings loaded successfully",
          activeConfig
        );
      } else {
        console.log("Nenhuma configuração de IA encontrada");
        window.currentAiConfigId = null;
      }
    } else {
      console.log("Error loading AI settings");
      window.currentAiConfigId = null;
    }
  } catch (error) {
    console.error("Error loading AI configuration:", error);
    showNotification("Error loading AI configuration", "error");
  }

  // Atualizar traduções
  if (typeof i18n !== "undefined") {
    i18n.updatePageTranslations();
  }
}

// Carregar página de notificações push
async function loadPushNotifications() {
  try {
    showNotification("Loading push notifications...", "info");
    // Implementar carregamento de notificações push
    console.log("Página de notificações push carregada");
  } catch (error) {
    console.error("Error loading push notifications:", error);
    showNotification("Error loading push notifications", "error");

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Funções auxiliares para controles do bot
async function toggleBot() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/toggle-bot`,
      {
        method: "POST",
      }
    );

    if (response.ok) {
      const result = await response.json();
      showNotification(result.message, "success");
      await loadBotStatus();
    } else {
      const error = await response.json();
      showNotification(error.error || "Error changing bot status", "error");
    }
  } catch (error) {
    console.error("Error changing bot status:", error);
    showNotification(i18n.t("notification.bot_status_error"), "error");
  }
}

async function toggleGroups() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/toggle-groups`,
      {
        method: "POST",
      }
    );

    if (response.ok) {
      const result = await response.json();
      showNotification(result.message, "success");
      await loadBotStatus();
    } else {
      const error = await response.json();
      showNotification(
        error.error || "Error changing groups status",
        "error"
      );
    }
  } catch (error) {
    console.error("Error changing groups status:", error);
    showNotification(i18n.t("notification.groups_status_error"), "error");
  }
}

// Função para atualizar dashboard
async function updateDashboard() {
  if (currentPage === "dashboard") {
    await loadDashboard();
  }
}

// ===================================================================
//  FUNÇÕES FALTANTES - CORREÇÕES DOS ERROS IDENTIFICADOS
// ===================================================================

// Funções para FAQs
function showAddFaqModal() {
  const modalHtml = `
        <div id="faqModal" class="modal" style="display: block;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3>Add FAQ</h3>
                    <span class="close" onclick="closeFaqModal()">&times;</span>
                </div>
                <form id="faqForm">
                    <div class="form-group">
                        <label>Question(s):</label>
                        <textarea id="faqQuestion" rows="3" required placeholder="Enter one or multiple questions separated by commas..."></textarea>
                        <small style="color: #666; font-size: 12px;">
                            💡 Tip: You can add multiple ways to ask by separating them with commas.<br>
                            Example: What time?, What time does it open?, Opening hours?
                        </small>
                    </div>
                    <div class="form-group">
                        <label>Answer:</label>
                        <textarea id="faqAnswer" rows="4" required></textarea>
                    </div>
                    <div class="form-group">
                        <label>Reaction Emoji:</label>
                        <input type="text" id="faqEmoji" readonly placeholder="Select an emoji" style="cursor: pointer; background: #f5f5f5;">
                        <div class="emoji-selector" style="display: flex; flex-wrap: wrap; gap: 8px; margin-top: 10px; padding: 10px; background: #f9f9f9; border-radius: 5px;">
                            <span class="emoji-option" onclick="selectEmoji('❓')">❓</span><span class="emoji-option" onclick="selectEmoji('👋')">👋</span><span class="emoji-option" onclick="selectEmoji('📍')">📍</span><span class="emoji-option" onclick="selectEmoji('⏰')">⏰</span><span class="emoji-option" onclick="selectEmoji('💳')">💳</span><span class="emoji-option" onclick="selectEmoji('📦')">📦</span><span class="emoji-option" onclick="selectEmoji('✅')">✅</span><span class="emoji-option" onclick="selectEmoji('❌')">❌</span><span class="emoji-option" onclick="selectEmoji('💬')">💬</span><span class="emoji-option" onclick="selectEmoji('📞')">📞</span><span class="emoji-option" onclick="selectEmoji('🏪')">🏪</span><span class="emoji-option" onclick="selectEmoji('🎉')">🎉</span><span class="emoji-option" onclick="selectEmoji('⚠️')">⚠️</span><span class="emoji-option" onclick="selectEmoji('🔔')">🔔</span><span class="emoji-option" onclick="selectEmoji('💡')">💡</span><span class="emoji-option" onclick="selectEmoji('🚚')">🚚</span><span class="emoji-option" onclick="selectEmoji('🛒')">🛒</span><span class="emoji-option" onclick="selectEmoji('💰')">💰</span>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" onclick="closeFaqModal()" class="btn btn-secondary">Cancel</button>
                        <button type="submit" class="btn btn-primary">Save</button>
                    </div>
                </form>
            </div>
        </div>
    `;

  document.getElementById("modalContainer").innerHTML = modalHtml;

  // Configure formulário
  document.getElementById("faqForm").onsubmit = async (e) => {
    e.preventDefault();
    await saveFaq();
  };
}

function closeFaqModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

async function saveFaq() {
  const question = document.getElementById("faqQuestion").value;
  const answer = document.getElementById("faqAnswer").value;
  const emoji = document.getElementById("faqEmoji").value;

  if (!emoji) {
    showNotification("Please select a reaction emoji", "warning");
    return;
  }

  try {
    const response = await authenticatedFetch(`${API_BASE}/api/admin/faqs`, {
      method: "POST",
      body: JSON.stringify({ question, answer, emoji }),
    });

    if (response.ok) {
      showNotification("FAQ added successfully", "success");
      closeFaqModal();
      await loadFaqs();
    } else {
      const error = await response.json();
      showNotification(error.error || "Error saving FAQ", "error");
    }
  } catch (error) {
    console.error("Error saving FAQ:", error);
    showNotification("Error saving FAQ", "error");
  }
}

function editFaq(faqId) {
  showNotification("Edit FAQ function not implemented", "warning");
  // TODO: Implementar edição de FAQ
}

function filterFaqs() {
  const category = document.getElementById("faqCategoryFilter").value;
  const search = document.getElementById("faqSearch").value.toLowerCase();

  const rows = document.querySelectorAll("#faqsTable tbody tr");
  rows.forEach((row) => {
    const question = row.cells[0].textContent.toLowerCase();
    const answer = row.cells[1].textContent.toLowerCase();

    const matchesSearch =
      !search || question.includes(search) || answer.includes(search);
    const matchesCategory = !category; // TODO: Implementar filtro por categoria quando tiver campo

    row.style.display = matchesSearch && matchesCategory ? "" : "none";
  });
}

function searchFaqs() {
  filterFaqs();
}

async function deleteFaq(faqId) {
  if (!confirm("Are you sure you want to delete this FAQ?")) {
    return;
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/faqs/${faqId}`,
      {
        method: "DELETE",
      }
    );

    if (response.ok) {
      showNotification("FAQ deleted successfully", "success");
      await loadFaqs();
    } else {
      const error = await response.json();
      showNotification(error.error || "Error deleting FAQ", "error");
    }
  } catch (error) {
    console.error("Error deleting FAQ:", error);
    showNotification("Error deleting FAQ", "error");
  }
}

// Funções para Campanhas/Envio em Massa
// Função corrigida para mostrar modal de criar campanha
function showCreateCampaignModal() {
  const modalHtml = `
        <div id="campaignModal" class="modal" style="display: block;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3>New Campaign</h3>
                    <span class="close" onclick="closeCampaignModal()">&times;</span>
                </div>
                <form id="campaignForm">
                    <div class="form-group">
                        <label>Campaign Name:</label>
                        <input type="text" id="campaignName" required>
                    </div>
                    <div class="form-group">
                        <label>Message:</label>
                        <textarea id="campaignMessage" rows="4" required placeholder="Type your message here..."></textarea>
                    </div>
                    <div class="form-group">
                        <label>Contact Group:</label>
                        <select id="campaignGroup" required>
                            <option value="">Loading grupos...</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>Schedule for (optional):</label>
                        <input type="datetime-local" id="campaignSchedule">
                    </div>
                    <div class="modal-footer">
                        <button type="button" onclick="closeCampaignModal()" class="btn btn-secondary">Cancel</button>
                        <button type="submit" class="btn btn-primary">Criar Campanha</button>
                    </div>
                </form>
            </div>
        </div>
    `;

  document.getElementById("modalContainer").innerHTML = modalHtml;

  // Carregar grupos no select após criar o modal
  setTimeout(async () => {
    await updateGroupSelects();

    // Add bibliotecas ao campo de mensagem
    addLibrariesToField("campaignMessage", "#campaignModal .form-group");
  }, 100);

  // Configure formulário
  document.getElementById("campaignForm").onsubmit = async (e) => {
    e.preventDefault();
    await saveCampaign();
  };
}

function closeCampaignModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

// Função corrigida para salvar campanha
async function saveCampaign() {
  const name = document.getElementById("campaignName").value.trim();
  const message = document.getElementById("campaignMessage").value.trim();
  const groupId = document.getElementById("campaignGroup").value;
  const schedule = document.getElementById("campaignSchedule").value;

  if (!name || !message || !groupId) {
    showNotification("Fill in all required fields", "error");
    return;
  }

  const submitBtn = document.querySelector(
    '#campaignForm button[type="submit"]'
  );
  if (submitBtn) {
    const originalText = submitBtn.textContent;
    submitBtn.disabled = true;
    submitBtn.textContent = "Creating...";

    try {
      console.log("Salvando campanha:", { name, message, groupId, schedule });

      const response = await authenticatedFetch(
        `${API_BASE}/api/admin/campaigns`,
        {
          method: "POST",
          body: JSON.stringify({
            name,
            message,
            group_id: groupId,
            schedule: schedule || null,
          }),
        }
      );

      const result = await response.json();

      if (response.ok) {
        showNotification("Campaign created successfully", "success");
        closeCampaignModal();
        await loadCampaigns();
      } else {
        console.error("Server error:", result);
        showNotification(result.error || "Error creating campaign", "error");
      }
    } catch (error) {
      console.error("Error creating campaign:", error);
      showNotification("Connection error when creating campaign", "error");
    } finally {
      if (submitBtn) {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
      }
    }
  }
}

// Função corrigida para mostrar modal de importar contatos
function showImportContactsModal() {
  const modalHtml = `
        <div id="importContactsModal" class="modal" style="display: block;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3>Import Contacts</h3>
                    <span class="close" onclick="closeImportContactsModal()">&times;</span>
                </div>
                <form id="importContactsForm">
                    <div class="form-group">
                        <label>CSV File:</label>
                        <input type="file" id="contactsFile" accept=".csv" required>
                        <small>Expected format: name,phone,email (header in first line)</small>
                    </div>
                    <div class="form-group">
                        <label>Destination Group (optional):</label>
                        <select id="importGroup">
                            <option value="">No group</option>
                        </select>
                    </div>
                    <div class="modal-footer">
                        <button type="button" onclick="closeImportContactsModal()" class="btn btn-secondary">Cancel</button>
                        <button type="submit" class="btn btn-primary">Importar</button>
                    </div>
                </form>
            </div>
        </div>
    `;

  document.getElementById("modalContainer").innerHTML = modalHtml;

  // Carregar grupos no select após criar o modal
  setTimeout(async () => {
    await updateGroupSelects();
  }, 100);

  // Configure formulário
  document.getElementById("importContactsForm").onsubmit = async (e) => {
    e.preventDefault();
    await importContacts();
  };
}

function closeImportContactsModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

async function importContacts() {
  const file = document.getElementById("contactsFile").files[0];
  const group = document.getElementById("importGroup").value;

  if (!file) {
    showNotification("Select a CSV file", "error");
    return;
  }

  const formData = new FormData();
  formData.append("file", file);
  if (group) {
    formData.append("group_id", group);
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/contacts/import`,
      {
        method: "POST",
        body: formData,
      }
    );

    const result = await response.json();

    if (response.ok) {
      showNotification(
        `✓ ${result.imported} importados | ${result.duplicates} duplicados`,
        "success"
      );
      closeImportContactsModal();
      await loadContactsList();
    } else {
      console.error("Error response:", result);
      showNotification(
        `Erro: ${result.error}\nDetalhes: ${
          result.errors?.join(" | ") || "N/A"
        }`,
        "error"
      );
    }
  } catch (error) {
    console.error("Connection error:", error);
    showNotification("Connection error when importing", "error");
  }
}

function showCreateGroupModal() {
  const modalHtml = `
        <div id="groupModal" class="modal" style="display: block;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3>New Contact Group</h3>
                    <span class="close" onclick="closeGroupModal()">&times;</span>
                </div>
                <form id="groupForm">
                    <div class="form-group">
                        <label>Group Name:</label>
                        <input type="text" id="groupName" required>
                    </div>
                    <div class="form-group">
                        <label>Description:</label>
                        <textarea id="groupDescription" rows="3"></textarea>
                    </div>
                    <div class="modal-footer">
                        <button type="button" onclick="closeGroupModal()" class="btn btn-secondary">Cancel</button>
                        <button type="submit" class="btn btn-primary">Criar Grupo</button>
                    </div>
                </form>
            </div>
        </div>
    `;

  document.getElementById("modalContainer").innerHTML = modalHtml;

  // Configure formulário
  document.getElementById("groupForm").onsubmit = async (e) => {
    e.preventDefault();
    await saveGroup();
  };
}

function closeGroupModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

// Função corrigida para salvar grupo
async function saveGroup() {
  const name = document.getElementById("groupName").value.trim();
  const description = document.getElementById("groupDescription").value.trim();

  if (!name) {
    showNotification("Group name is required", "error");
    return;
  }

  const submitBtn = document.querySelector('#groupForm button[type="submit"]');
  if (submitBtn) {
    const originalText = submitBtn.textContent;
    submitBtn.disabled = true;
    submitBtn.textContent = "Saving...";

    try {
      console.log("Salvando grupo:", { name, description });

      const response = await authenticatedFetch(
        `${API_BASE}/api/admin/contact-groups`,
        {
          method: "POST",
          body: JSON.stringify({ name, description }),
        }
      );

      const result = await response.json();

      if (response.ok) {
        showNotification("Group created successfully", "success");
        closeGroupModal();

        // Recarregar dados
        await loadContactGroups();
        await loadContactGroupsFilter();

        // Se estivermos na página de envio em massa, também atualizar
        if (currentPage === "mass-messaging") {
          await loadMassMessaging();
        }
      } else {
        console.error("Server error:", result);
        showNotification(result.error || "Error creating group", "error");
      }
    } catch (error) {
      console.error("Error creating group:", error);
      showNotification("Connection error when creating group", "error");
    } finally {
      if (submitBtn) {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
      }
    }
  }
}

// Funções para Pagamentos
function showPaymentConfigModal() {
  const modalHtml = `
        <div id="paymentConfigModal" class="modal" style="display: block;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3>Configure Payment Gateways</h3>
                    <span class="close" onclick="closePaymentConfigModal()">&times;</span>
                </div>
                <div class="tabs">
                    <button class="tab-btn active" onclick="showPaymentTab('pagseguro')">PagBank</button>
                    <button class="tab-btn" onclick="showPaymentTab('paypal')">PayPal</button>
                </div>
                <div id="pagseguro-tab" class="tab-content active">
                    <form id="pagseguroForm">
                        <div class="form-group">
                            <label>PagBank Email:</label>
                            <input type="email" id="pagseguroEmail" placeholder="your-email@example.com">
                        </div>
                        <div class="form-group">
                            <label>PagBank Token:</label>
                            <input type="text" id="pagseguroToken" placeholder="Integration token">
                        </div>
                        <div class="form-group">
                            <label>
                                <input type="checkbox" id="pagseguroSandbox" checked> Sandbox Mode (Testing)
                            </label>
                        </div>
                        <div class="form-group">
                            <label>
                                <input type="checkbox" id="pagseguroActive"> Active
                            </label>
                        </div>
                        <button type="button" onclick="savePaymentConfig('pagseguro')" class="btn btn-primary">Save PagBank</button>
                    </form>
                </div>
                <div id="paypal-tab" class="tab-content">
                    <form id="paypalForm">
                        <div class="form-group">
                            <label>PayPal Client ID:</label>
                            <input type="text" id="paypalClientId">
                        </div>
                        <div class="form-group">
                            <label>PayPal Client Secret:</label>
                            <input type="password" id="paypalClientSecret">
                        </div>
                        <div class="form-group">
                            <label>
                                <input type="checkbox" id="paypalSandbox"> Sandbox Mode
                            </label>
                        </div>
                        <div class="form-group">
                            <label>
                                <input type="checkbox" id="paypalActive"> Active
                            </label>
                        </div>
                        <button type="button" onclick="savePaymentConfig('paypal')" class="btn btn-primary">Save PayPal</button>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" onclick="closePaymentConfigModal()" class="btn btn-secondary">Close</button>
                </div>
            </div>
        </div>
    `;

  document.getElementById("modalContainer").innerHTML = modalHtml;

  // Carregar configurações existentes
  setTimeout(() => {
    loadPaymentConfig("pagseguro");
    loadPaymentConfig("paypal");
  }, 100);
}

function closePaymentConfigModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

function showPaymentTab(tab) {
  document
    .querySelectorAll("#paymentConfigModal .tab-btn")
    .forEach((btn) => btn.classList.remove("active"));
  document
    .querySelectorAll("#paymentConfigModal .tab-content")
    .forEach((content) => content.classList.remove("active"));

  document
    .querySelector(`#paymentConfigModal [onclick="showPaymentTab('${tab}')"]`)
    .classList.add("active");
  document.getElementById(`${tab}-tab`).classList.add("active");
}

async function savePaymentConfig(gateway) {
  const formData = {};

  if (gateway === "pagseguro") {
    const emailInput = document.getElementById("pagseguroEmail");
    const tokenInput = document.getElementById("pagseguroToken");
    const sandboxInput = document.getElementById("pagseguroSandbox");
    const activeInput = document.getElementById("pagseguroActive");

    if (!emailInput || !tokenInput) {
      console.error("Elementos do formulário PagBank not founds");
      showNotification("Error in form", "error");
      return;
    }

    formData.email = emailInput.value.trim();
    formData.token = tokenInput.value.trim();
    formData.sandbox = sandboxInput ? sandboxInput.checked : true;
    formData.active = activeInput ? activeInput.checked : false;

    console.log("PagBank formData:", formData);

    if (!formData.email || !formData.token) {
      showNotification("Fill in PagBank email and token", "warning");
      return;
    }
  } else if (gateway === "paypal") {
    const clientIdInput = document.getElementById("paypalClientId");
    const clientSecretInput = document.getElementById("paypalClientSecret");
    const sandboxInput = document.getElementById("paypalSandbox");
    const activeInput = document.getElementById("paypalActive");

    if (!clientIdInput || !clientSecretInput) {
      console.error("Elementos do formulário PayPal not founds");
      showNotification("Error in form", "error");
      return;
    }

    formData.clientId = clientIdInput.value.trim();
    formData.clientSecret = clientSecretInput.value.trim();
    formData.sandbox = sandboxInput ? sandboxInput.checked : true;
    formData.active = activeInput ? activeInput.checked : false;

    console.log("PayPal formData:", formData);

    if (!formData.clientId || !formData.clientSecret) {
      showNotification(
        "Fill in PayPal Client ID and Client Secret",
        "warning"
      );
      return;
    }
  }

  try {
    console.log(
      "Enviando para:",
      `${API_BASE}/api/admin/payment-config/${gateway}`
    );
    console.log("Dados:", formData);

    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/payment-config/${gateway}`,
      {
        method: "POST",
        body: JSON.stringify(formData),
      }
    );

    const result = await response.json();
    console.log("Answer:", result);

    if (response.ok) {
      showNotification(
        `${gateway.toUpperCase()} configuration saved successfully`,
        "success"
      );
      setTimeout(() => loadPaymentConfig(gateway), 500);
    } else {
      showNotification(
        result.error || `Error saving ${gateway} configuration`,
        "error"
      );
    }
  } catch (error) {
    console.error(`Error saving ${gateway} configuration:`, error);
    showNotification(`Error saving ${gateway} configuration`, "error");
  }
}

async function loadPaymentConfig(gateway) {
  try {
    console.log("Loading configuração:", gateway);
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/payment-config/${gateway}`
    );

    if (response.ok) {
      const data = await response.json();
      console.log("Dados carregados:", data);

      if (data.configured) {
        if (gateway === "pagseguro") {
          const sandboxCheck = document.getElementById("pagseguroSandbox");
          const activeCheck = document.getElementById("pagseguroActive");

          // MANTER OS CHECKBOXES MARCADOS!
          if (sandboxCheck) sandboxCheck.checked = data.sandbox;
          if (activeCheck) activeCheck.checked = data.active;

          // Mostrar que está configurado
          const emailInput = document.getElementById("pagseguroEmail");
          const tokenInput = document.getElementById("pagseguroToken");

          if (emailInput) {
            emailInput.value = ""; // Limpar por segurança
            emailInput.placeholder = "✓ Email configured";
          }
          if (tokenInput) {
            tokenInput.value = ""; // Limpar por segurança
            tokenInput.placeholder = "✓ Token configured";
          }
        } else if (gateway === "paypal") {
          const sandboxCheck = document.getElementById("paypalSandbox");
          const activeCheck = document.getElementById("paypalActive");

          // MANTER OS CHECKBOXES MARCADOS!
          if (sandboxCheck) sandboxCheck.checked = data.sandbox;
          if (activeCheck) activeCheck.checked = data.active;

          const clientIdInput = document.getElementById("paypalClientId");
          const clientSecretInput =
            document.getElementById("paypalClientSecret");

          if (clientIdInput) {
            clientIdInput.value = "";
            clientIdInput.placeholder = "✓ Client ID configurado";
          }
          if (clientSecretInput) {
            clientSecretInput.value = "";
            clientSecretInput.placeholder = "✓ Client Secret configurado";
          }
        }
      }
    }
  } catch (error) {
    console.error(`Error loading configuration ${gateway}:`, error);

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Carregar configurações ao mostrar a aba de pagamentos
function showPaymentTab(gateway) {
  // Remover active de todas as tabs
  const modal = document.getElementById("paymentConfigModal");
  if (!modal) return;

  modal
    .querySelectorAll(".tab-btn")
    .forEach((btn) => btn.classList.remove("active"));
  modal.querySelectorAll(".tab-content").forEach((content) => {
    content.classList.remove("active");
    content.style.display = "none";
  });

  // Activer tab selecionada
  event.target.classList.add("active");
  const tabContent = document.getElementById(`${gateway}-tab`);
  if (tabContent) {
    tabContent.classList.add("active");
    tabContent.style.display = "block";
  }

  // Carregar configuração
  loadPaymentConfig(gateway);
}

function filterPayments() {
  const status = document.getElementById("paymentStatusFilter").value;
  const dateFrom = document.getElementById("paymentDateFrom").value;
  const dateTo = document.getElementById("paymentDateTo").value;

  // TODO: Implementar filtro de pagamentos
  showNotification("Payment filter applied", "info");
}

// Funções para Traduções
function showImportTranslationModal() {
  const modalHtml = `
        <div id="importTranslationModal" class="modal" style="display: block;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3>Import Translations</h3>
                    <span class="close" onclick="closeImportTranslationModal()">&times;</span>
                </div>
                <form id="importTranslationForm">
                    <div class="form-group">
                        <label>Language:</label>
                        <select id="translationLanguage" required>
                            <option value="">Selecione um idioma</option>
                            <option value="en">Inglês</option>
                            <option value="es">Espanhol</option>
                            <option value="fr">Francês</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>JSON File:</label>
                        <input type="file" id="translationFile" accept=".json" required>
                    </div>
                    <div class="modal-footer">
                        <button type="button" onclick="closeImportTranslationModal()" class="btn btn-secondary">Cancel</button>
                        <button type="submit" class="btn btn-primary">Importar</button>
                    </div>
                </form>
            </div>
        </div>
    `;

  document.getElementById("modalContainer").innerHTML = modalHtml;

  // Configure formulário
  document.getElementById("importTranslationForm").onsubmit = async (e) => {
    e.preventDefault();
    await importTranslations();
  };
}

function closeImportTranslationModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

async function importTranslations() {
  const language = document.getElementById("translationLanguage").value;
  const file = document.getElementById("translationFile").files[0];

  if (!language || !file) {
    showNotification("Select a language and a file", "error");
    return;
  }

  const formData = new FormData();
  formData.append("language", language);
  formData.append("file", file);

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/import-translations`,
      {
        method: "POST",
        body: formData,
        headers: {}, // Remove Content-Type para FormData
      }
    );

    if (response.ok) {
      const result = await response.json();
      showNotification(`Translations imported successfully`, "success");
      closeImportTranslationModal();
      await loadTranslations();
    } else {
      const error = await response.json();
      showNotification(error.error || "Error importing translations", "error");
    }
  } catch (error) {
    console.error("Error importing translations:", error);
    showNotification("Error importing translations", "error");
  }
}

function downloadTranslationTemplate() {
  const template = {
    welcome_message: "Bem-vindo!",
    goodbye_message: "Até logo!",
    help_message: "Como posso ajudar?",
  };

  const blob = new Blob([JSON.stringify(template, null, 2)], {
    type: "application/json",
  });
  const url = URL.createObjectURL(blob);
  const a = document.createElement("a");
  a.href = url;
  a.download = "translation_template.json";
  a.click();
  URL.revokeObjectURL(url);
}

function showTranslationTab(tab) {
  document
    .querySelectorAll(".translation-tab-btn")
    .forEach((btn) => btn.classList.remove("active"));
  document
    .querySelectorAll(".translation-tab-content")
    .forEach((content) => content.classList.remove("active"));

  document
    .querySelector(`[onclick="showTranslationTab('${tab}')"]`)
    .classList.add("active");
  document.getElementById(`${tab}-translation-tab`).classList.add("active");
}

function filterTranslationStrings() {
  const language = document.getElementById("translationLanguageFilter").value;
  const search = document
    .getElementById("translationSearch")
    .value.toLowerCase();

  // TODO: Implementar filtro de strings de tradução
  showNotification("Translation filter applied", "info");
}

// Funções para Configuração de IA
function saveDeepSeekConfig() {
  const apiKey = document.getElementById("deepseekApiKey")?.value || "";
  const model =
    document.getElementById("deepseekModel")?.value || "deepseek-chat";
  const personaName =
    document.getElementById("deepseekPersonaName")?.value || "Assistente";
  const personaDescription =
    document.getElementById("deepseekPersonaDescription")?.value ||
    "Intelligent virtual assistant";
  const systemPrompt =
    document.getElementById("deepseekSystemPrompt")?.value ||
    "Você é um assistente útil.";
  const workingHours =
    document.getElementById("deepseekWorkingHours")?.value || "00:00-23:59";

  const [start, end] = workingHours.split("-");

  const config = {
    provider: "deepseek",
    api_key: apiKey,
    model_name: model,
    persona_name: personaName,
    persona_description: personaDescription,
    system_prompt: systemPrompt,
    business_hours_start: start?.trim() || "08:00",
    business_hours_end: end?.trim() || "18:00",
    active: true,
  };

  saveAiConfig(config);
}

function saveOpenAIConfig() {
  const apiKey = document.getElementById("openaiApiKey")?.value || "";
  const model =
    document.getElementById("openaiModel")?.value || "gpt-3.5-turbo";
  const personaName =
    document.getElementById("openaiPersonaName")?.value || "Assistente";
  const personaDescription =
    document.getElementById("openaiPersonaDescription")?.value ||
    "Intelligent virtual assistant";
  const systemPrompt =
    document.getElementById("openaiSystemPrompt")?.value ||
    "Você é um assistente útil.";
  const workingHours =
    document.getElementById("openaiWorkingHours")?.value || "00:00-23:59";

  const [start, end] = workingHours.split("-");

  const config = {
    provider: "gpt",
    api_key: apiKey,
    model_name: model,
    persona_name: personaName,
    persona_description: personaDescription,
    system_prompt: systemPrompt,
    business_hours_start: start?.trim() || "08:00",
    business_hours_end: end?.trim() || "18:00",
    active: true,
  };

  saveAiConfig(config);
}

async function saveAiConfig(config) {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/ai-config/settings`,
      {
        method: "POST",
        body: JSON.stringify(config),
      }
    );

    if (response.ok) {
      showNotification(
        `${config.provider} configuration saved successfully`,
        "success"
      );
    } else {
      const error = await response.json();
      showNotification(
        error.error || `Error saving ${config.provider} configuration`,
        "error"
      );
    }
  } catch (error) {
    console.error(`Error saving ${config.provider} configuration:`, error);
    showNotification(`Error saving ${config.provider} configuration`, "error");
  }
}

// Funções auxiliares para usuários
function showAddUserModal() {
  // CORREÇÃO: Verificar se modalContainer existe
  const modalContainer = document.getElementById("modalContainer");
  if (!modalContainer) {
    console.error("Elemento modalContainer not found no DOM");
    showNotification("Error: Modal container not found", "error");
    return;
  }

  const modalHtml = `
        <div id="userModal" class="modal" style="display: block;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3>Add User</h3>
                    <span class="close" onclick="closeUserModal()">&times;</span>
                </div>
                <form id="userForm">
                    <div class="form-group">
                        <label>Username:</label>
                        <input type="text" id="newUsername" required>
                    </div>
                    <div class="form-group">
                        <label>Password:</label>
                        <input type="password" id="newPassword" required>
                    </div>
                    <div class="form-group">
                        <label>Link to:</label>
                        <div style="margin-bottom: 10px;">
                            <label style="margin-right: 20px;">
                                <input type="radio" name="linkType" value="store" checked onchange="toggleUserLinkType()"> Store
                            </label>
                            <label>
                                <input type="radio" name="linkType" value="department" onchange="toggleUserLinkType()"> Department
                            </label>
                        </div>
                    </div>
                    <div class="form-group" id="storeSelectGroup">
                        <label>Store:</label>
                        <select id="newUserStore">
                            <option value="">Selecione uma loja</option>
                        </select>
                    </div>
                    <div class="form-group" id="departmentSelectGroup" style="display: none;">
                        <label>Department:</label>
                        <select id="newUserDepartment">
                            <option value="">Select a department</option>
                        </select>
                    </div>
                    <div class="modal-footer">
                        <button type="button" onclick="closeUserModal()" class="btn btn-secondary">Cancel</button>
                        <button type="submit" class="btn btn-primary">Save</button>
                    </div>
                </form>
            </div>
        </div>
    `;

  document.getElementById("modalContainer").innerHTML = modalHtml;

  // Carregar lojas e setores
  setTimeout(async () => {
    await loadStoresForModal();
    await loadDepartmentsForModal();
  }, 0);

  // Configure formulário após um pequeno delay
  setTimeout(() => {
    const form = document.getElementById("userForm");
    if (form) {
      form.onsubmit = async (e) => {
        e.preventDefault();
        await saveUser();
      };
    }
  }, 10);
}

function closeUserModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

async function saveUser() {
  const username = document.getElementById("newUsername").value;
  const password = document.getElementById("newPassword").value;
  const linkType = document.querySelector(
    'input[name="linkType"]:checked'
  ).value;

  let store = null;
  let department = null;

  if (linkType === "store") {
    store = document.getElementById("newUserStore").value;
    if (!store) {
      showNotification(i18n.t("notification.store_required"), "error");
      return;
    }
  } else {
    department = document.getElementById("newUserDepartment").value;
    if (!department) {
      showNotification("Select a department", "error");
      return;
    }
  }

  try {
    const response = await authenticatedFetch(`${API_BASE}/api/admin/users`, {
      method: "POST",
      body: JSON.stringify({ username, password, store, department }),
    });

    if (response.ok) {
      showNotification(i18n.t("notification.user_created"), "success");
      closeUserModal();
      await loadUsers();
    } else {
      const error = await response.json();
      showNotification(error.error || "Error creating user", "error");
    }
  } catch (error) {
    console.error("Error creating user:", error);
    showNotification("Error creating user", "error");
  }
}

// Função para alternar entre loja e setor
function toggleUserLinkType() {
  const linkType = document.querySelector(
    'input[name="linkType"]:checked'
  ).value;
  const storeGroup = document.getElementById("storeSelectGroup");
  const departmentGroup = document.getElementById("departmentSelectGroup");

  if (linkType === "store") {
    storeGroup.style.display = "block";
    departmentGroup.style.display = "none";
    document.getElementById("newUserDepartment").value = "";
  } else {
    storeGroup.style.display = "none";
    departmentGroup.style.display = "block";
    document.getElementById("newUserStore").value = "";
  }
}

// Funções auxiliares para lojas
function showAddStoreModal() {
  const modalHtml = `
        <div id="storeModal" class="modal" style="display: block;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3>Add Store</h3>
                    <span class="close" onclick="closeStoreModal()">&times;</span>
                </div>
                <form id="storeForm">
                    <div class="form-group">
                        <label>Store Name:</label>
                        <input type="text" id="storeName" required>
                    </div>
                    <div class="form-group">
                        <label>Description:</label>
                        <textarea id="storeDescription" rows="3"></textarea>
                    </div>
                    <div class="form-group">
                        <label>Address:</label>
                        <textarea id="storeAddress" rows="3"></textarea>
                    </div>
                    <div class="form-group">
                        <label>Hours:</label>
                        <textarea id="storeHours" rows="2"></textarea>
                    </div>
                    <div class="form-group">
                        <label>Promotions:</label>
                        <textarea id="storePromotions" rows="3"></textarea>
                    </div>
                    <div class="modal-footer">
                        <button type="button" onclick="closeStoreModal()" class="btn btn-secondary">Cancel</button>
                        <button type="submit" class="btn btn-primary">Save</button>
                    </div>
                </form>
            </div>
        </div>
    `;

  document.getElementById("modalContainer").innerHTML = modalHtml;

  // Configure formulário
  document.getElementById("storeForm").onsubmit = async (e) => {
    e.preventDefault();
    await saveStore();
  };
}

function closeStoreModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

async function saveStore() {
  const name = document.getElementById("storeName").value;
  const description = document.getElementById("storeDescription").value;
  const address = document.getElementById("storeAddress").value;
  const hours = document.getElementById("storeHours").value;
  const promotions = document.getElementById("storePromotions").value;

  try {
    const response = await authenticatedFetch(`${API_BASE}/api/admin/stores`, {
      method: "POST",
      body: JSON.stringify({ name, description, address, hours, promotions }),
    });

    if (response.ok) {
      showNotification(i18n.t("notification.store_created"), "success");
      closeStoreModal();
      await loadStores();
    } else {
      const error = await response.json();
      showNotification(
        error.error || i18n.t("notification.store_create_error"),
        "error"
      );
    }
  } catch (error) {
    console.error("Error creating store:", error);
    showNotification(i18n.t("notification.store_create_error"), "error");
  }
}

// Função para corrigir o problema do storesList
async function loadStores() {
  try {
    console.log("Loading lojas...");
    const response = await authenticatedFetch(`${API_BASE}/api/admin/stores`);

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }

    const stores = await response.json();
    console.log("Stores carregadas:", stores);

    // Buscar contagem de usuários por loja
    const usersResponse = await authenticatedFetch(
      `${API_BASE}/api/admin/users`
    );
    const users = await usersResponse.json();

    // Tentar encontrar o container na tabela primeiro
    let container = document.getElementById("storesTableBody");
    if (!container) {
      // Se não encontrar, tentar o container de cards
      container = document.getElementById("storesList");
      if (!container) {
        // Se ainda não encontrar, criar o container
        const storesPage = document.getElementById("stores-page");
        if (storesPage) {
          const tableContainer = storesPage.querySelector(".table-container");
          if (tableContainer) {
            container = document.createElement("div");
            container.id = "storesList";
            tableContainer.appendChild(container);
          }
        }
      }
    }

    if (!container) {
      console.error("Container storesList not found e could not be created");
      return;
    }

    container.innerHTML = "";

    if (stores.length === 0) {
      container.innerHTML =
        '<tr><td colspan="5">No stores found</td></tr>';
      return;
    }

    // Se for tabela, usar formato de tabela
    if (container.tagName === "TBODY") {
      stores.forEach((store) => {
        const userCount = users.filter((u) => u.store === store.name).length;
        
        const row = document.createElement("tr");
        row.dataset.storeId = store.id;
        row.dataset.storeName = store.name;
        row.dataset.storeDescription = store.description || "";
        row.innerHTML = `
                    <td>${store.name}</td>
                    <td>${userCount}</td>
                    <td>${formatDate(store.created_at)}</td>
                    <td>
                        <button onclick="editStore(${
                          store.id
                        })" class="btn btn-secondary btn-sm">
                            <span data-i18n="btn.edit">${i18n.t(
                              "btn.edit"
                            )}</span>
                        </button>
                        <button onclick="deleteStore(${
                          store.id
                        })" class="btn btn-danger btn-sm">
                            <span data-i18n="btn.delete">${i18n.t(
                              "btn.delete"
                            )}</span>
                        </button>
                    </td>
                `;
        container.appendChild(row);
      });
    } else {
      // Se for div, usar formato de cards
      stores.forEach((store) => {
        const storeCard = document.createElement("div");
        storeCard.className = "store-card";

        storeCard.innerHTML = `
                    <h3>${store.name}</h3>
                    <form onsubmit="updateStore(event, ${store.id})">
                        <div class="form-group">
                            <label>Address</label>
                            <textarea name="address" rows="3">${
                              store.address || ""
                            }</textarea>
                        </div>
                        <div class="form-group">
                            <label>Hours</label>
                            <textarea name="hours" rows="2">${
                              store.hours || ""
                            }</textarea>
                        </div>
                        <div class="form-group">
                            <label>Promotions</label>
                            <textarea name="promotions" rows="3">${
                              store.promotions || ""
                            }</textarea>
                        </div>
                        <button type="submit" class="btn btn-primary">Save</button>
                    </form>
                `;

        container.appendChild(storeCard);
      });
    }
  } catch (error) {
    console.error("Error loading stores:", error);
    showNotification(
      i18n.t("notification.stores_load_error") + ": " + error.message,
      "error"
    );

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

async function editStore(storeId) {
  try {
    // Search dados completos da loja do servidor
    const response = await authenticatedFetch(`${API_BASE}/api/admin/stores`);
    if (!response.ok) {
      throw new Error("Error fetching store data");
    }

    const stores = await response.json();
    const store = stores.find((s) => s.id === storeId);

    if (!store) {
      showNotification(i18n.t("notification.error"), "error");
      return;
    }

    // Criar modal de edição com todos os campos (igual ao modal de criar)
    const modalHtml = `
            <div id="editStoreModal" class="modal" style="display: block;">
                <div class="modal-content">
                    <div class="modal-header">
                        <h3 data-i18n="stores.edit">Edit Store</h3>
                        <span class="close" onclick="closeEditStoreModal()">&times;</span>
                    </div>
                    <form id="editStoreForm">
                        <div class="form-group">
                            <label>Store Name:</label>
                            <input type="text" id="editStoreName" value="${
                              store.name || ""
                            }" required>
                        </div>
                        <div class="form-group">
                            <label>Address:</label>
                            <textarea id="editStoreAddress" rows="3">${
                              store.address || ""
                            }</textarea>
                        </div>
                        <div class="form-group">
                            <label>Hours:</label>
                            <textarea id="editStoreHours" rows="2">${
                              store.hours || ""
                            }</textarea>
                        </div>
                        <div class="form-group">
                            <label>Promotions:</label>
                            <textarea id="editStorePromotions" rows="3">${
                              store.promotions || ""
                            }</textarea>
                        </div>
                        <div class="modal-footer">
                            <button type="button" onclick="closeEditStoreModal()" class="btn btn-secondary">
                                <span data-i18n="btn.cancel">Cancel</span>
                            </button>
                            <button type="submit" class="btn btn-primary">
                                <span data-i18n="btn.save">Save</span>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        `;

    document.getElementById("modalContainer").innerHTML = modalHtml;

    // Configure formulário
    document.getElementById("editStoreForm").onsubmit = async (e) => {
      e.preventDefault();
      await saveStoreEdit(storeId);
    };

    // Atualizar traduções
    i18n.updatePageTranslations();
  } catch (error) {
    console.error("Error editar loja:", error);
    showNotification(i18n.t("notification.error"), "error");
  }
}

function closeEditStoreModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

async function saveStoreEdit(storeId) {
  const name = document.getElementById("editStoreName").value;
  const address = document.getElementById("editStoreAddress").value;
  const hours = document.getElementById("editStoreHours").value;
  const promotions = document.getElementById("editStorePromotions").value;

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/stores/${storeId}`,
      {
        method: "PUT",
        body: JSON.stringify({ name, address, hours, promotions }),
      }
    );

    if (response.ok) {
      showNotification(i18n.t("notification.updated"), "success");
      closeEditStoreModal();
      await loadStores();
    } else {
      const error = await response.json();
      showNotification(error.error || i18n.t("notification.error"), "error");
    }
  } catch (error) {
    console.error("Error saving store:", error);
    showNotification(i18n.t("notification.error"), "error");
  }
}

async function deleteStore(storeId) {
  if (!confirm(i18n.t("notification.confirm_delete_store"))) {
    return;
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/stores/${storeId}`,
      {
        method: "DELETE",
      }
    );

    if (response.ok) {
      showNotification(i18n.t("notification.store_deleted"), "success");
      await loadStores();
    } else {
      const error = await response.json();
      showNotification(
        error.error || i18n.t("notification.store_delete_error"),
        "error"
      );
    }
  } catch (error) {
    console.error("Error deleting store:", error);
    showNotification("Error deleting store", "error");
  }
}

// Funções auxiliares para filtros e busca
function searchUsers() {
  const search = document.getElementById("userSearch").value.toLowerCase();
  const rows = document.querySelectorAll("#usersTable tbody tr");

  rows.forEach((row) => {
    const username = row.cells[0].textContent.toLowerCase();
    const email = row.cells[1].textContent.toLowerCase();
    const store = row.cells[3].textContent.toLowerCase();

    const matches =
      username.includes(search) ||
      email.includes(search) ||
      store.includes(search);
    row.style.display = matches ? "" : "none";
  });
}

function filterUsers() {
  const role = document.getElementById("userRoleFilter").value;
  const rows = document.querySelectorAll("#usersTable tbody tr");

  rows.forEach((row) => {
    const userRole = row.cells[2].textContent.toLowerCase();
    const matches = !role || userRole === role;
    row.style.display = matches ? "" : "none";
  });
}

async function filterConversationtions() {
  const statusFilter = document.getElementById(
    "conversationStatusFilter"
  ).value;
  const storeFilter = document.getElementById("conversationStoreFilter").value;

  try {
    // Construir query string com filtros
    let url = `${API_BASE}/api/chat/conversations`;
    const params = new URLSearchParams();

    if (statusFilter) {
      params.append("status", statusFilter);
    }
    if (storeFilter) {
      params.append("store_id", storeFilter);
    }

    if (params.toString()) {
      url += "?" + params.toString();
    }

    const response = await authenticatedFetch(url);
    const conversations = await response.json();

    const container = document.getElementById("conversationsList");
    const countElement = document.getElementById("conversationsCount");

    if (!container) return;

    // Atualizar contador
    if (countElement) {
      countElement.textContent = conversations.length || 0;
    }

    container.innerHTML = "";

    if (conversations.length === 0) {
      container.innerHTML =
        '<div style="padding: 20px; text-align: center; color: #999;">No conversations found com os filtros aplicados</div>';
      return;
    }

    conversations.forEach((conversation) => {
      // Debug: ver estrutura completa da conversa
      console.log("Conversation completa:", conversation);

      const item = document.createElement("div");
      item.className = "conversation-item";
      item.onclick = (e) => selectConversationtion(conversation.id, e);

      // Determinar status da conversa - verificar múltiplos campos possíveis
      let statusBadge = "";
      let statusClass = "";

      // Verificar se está fechada
      if (
        conversation.status === "closed" ||
        conversation.status === "finalizada"
      ) {
        statusBadge =
          '<span class="status-badge status-closed">Finalizada</span>';
        statusClass = "conversation-closed";
      }
      // Verificar se está atendida - múltiplas possibilidades de campos
      else if (
        conversation.assigned_user_id ||
        conversation.assigned_username ||
        conversation.assignedUserId ||
        conversation.assignedUsername ||
        conversation.attended_by ||
        conversation.attendedBy ||
        conversation.status === "attended" ||
        conversation.status === "em_atendimento"
      ) {
        const attendantName =
          conversation.assigned_username ||
          conversation.assignedUsername ||
          conversation.attended_by ||
          conversation.attendedBy ||
          "Operador";
        statusBadge = `<span class="status-badge status-attended" data-i18n="conversations.attended_by">${i18n.t(
          "conversations.attended_by"
        )} ${attendantName}</span>`;
        statusClass = "conversation-attended";
      }
      // Caso contrário, está aguardando
      else {
        statusBadge =
          '<span class="status-badge status-waiting">Waiting</span>';
        statusClass = "conversation-waiting";
      }

      item.classList.add(statusClass);

      item.innerHTML = `
                <div class="conversation-contact">${formatPhone(
                  conversation.phone_number
                )}</div>
                <div class="conversation-preview">${
                  conversation.last_message || "Sem mensagens"
                }</div>
                <div class="conversation-meta">
                    <div class="conversation-time">${formatDate(
                      conversation.last_message_time
                    )}</div>
                    ${statusBadge}
                </div>
            `;

      container.appendChild(item);
    });

    showNotification(
      `${conversations.length} conversa(s) encontrada(s)`,
      "success"
    );
  } catch (error) {
    console.error("Error filtrar conversas:", error);
    showNotification("Error filtrar conversas", "error");
  }
}

// Função para mostrar loading no dashboard

// Função para mostrar abas
function showTab(tabName) {
  document
    .querySelectorAll(".tab-btn")
    .forEach((btn) => btn.classList.remove("active"));
  document
    .querySelectorAll(".tab-content")
    .forEach((content) => content.classList.remove("active"));

  document
    .querySelector(`[onclick="showTab('${tabName}')"]`)
    .classList.add("active");
  document.getElementById(`${tabName}-tab`).classList.add("active");
}

console.log("Funções faltantes adicionadas ao admin.js");

// ===================================================================
//  BIBLIOTECA DE EMOJIS E PLACEHOLDERS CLICÁVEIS
// ===================================================================

// Lista de emojis mais usados
const EMOJI_LIBRARY = [
  "😀",
  "😂",
  "😍",
  "😎",
  "😭",
  "😡",
  "👍",
  "🙏",
  "💪",
  "🎉",
  "❤️",
  "🔥",
  "✅",
  "❌",
  "⚠️",
  "📞",
  "📍",
  "🕐",
  "💰",
  "🛒",
  "🚚",
  "📦",
  "🎯",
  "💡",
  "🔔",
  "📱",
  "💻",
  "🌟",
  "👋",
  "🤝",
];

// Lista de placeholders disponíveis
const PLACEHOLDER_LIBRARY = [
  "{{nome_cliente}}",
  "{{data_atual}}",
  "{{hora_atual}}",
];

// Função para criar biblioteca de emojis
function createEmojiLibrary(targetInputId) {
  const emojiContainer = document.createElement("div");
  emojiContainer.className = "emoji-library";
  emojiContainer.innerHTML = `
        <div class="emoji-header">
            <span>Emojis:</span>
            <button type="button" class="emoji-toggle" onclick="toggleEmojiLibrary('${targetInputId}')">😀</button>
        </div>
        <div class="emoji-grid" id="emojiGrid_${targetInputId}" style="display: none;">
            ${EMOJI_LIBRARY.map(
              (emoji) =>
                `<button type="button" class="emoji-btn" onclick="insertEmoji('${targetInputId}', '${emoji}')">${emoji}</button>`
            ).join("")}
        </div>
    `;
  return emojiContainer;
}

// Função para criar biblioteca de placeholders
function createPlaceholderLibrary(targetInputId) {
  const placeholderContainer = document.createElement("div");
  placeholderContainer.className = "placeholder-library";
  placeholderContainer.innerHTML = `
        <div class="placeholder-header">
            <span>Placeholders:</span>
        </div>
        <div class="placeholder-grid" id="placeholderGrid_${targetInputId}" style="display: block;">
            ${PLACEHOLDER_LIBRARY.map(
              (placeholder) =>
                `<button type="button" class="placeholder-btn" data-placeholder="${placeholder}">${placeholder}</button>`
            ).join("")}
        </div>
    `;
  return placeholderContainer;
}

// Função para alternar visibilidade da biblioteca de emojis
function toggleEmojiLibrary(targetInputId) {
  const grid = document.getElementById(`emojiGrid_${targetInputId}`);
  if (grid) {
    grid.style.display = grid.style.display === "none" ? "block" : "none";
  }
}

// Função para alternar visibilidade da biblioteca de placeholders
function togglePlaceholderLibrary(targetInputId) {
  const grid = document.getElementById(`placeholderGrid_${targetInputId}`);
  if (grid) {
    grid.style.display = grid.style.display === "none" ? "block" : "none";
  }
}

// Função para inserir emoji no campo de texto
function insertEmoji(targetInputId, emoji) {
  const input = document.getElementById(targetInputId);
  if (input) {
    const cursorPos = input.selectionStart;
    const textBefore = input.value.substring(0, cursorPos);
    const textAfter = input.value.substring(input.selectionEnd);

    input.value = textBefore + emoji + textAfter;
    input.focus();
    input.setSelectionRange(cursorPos + emoji.length, cursorPos + emoji.length);

    // Fechar a biblioteca após inserir
    toggleEmojiLibrary(targetInputId);
  }
}

// Função para inserir placeholder no campo de texto - VERSÃO CORRIGIDA
function insertPlaceholder(targetInputId, placeholder) {
  let input = null;

  // 1. Se foi passado um ID específico, tentar encontrar
  if (targetInputId && typeof targetInputId === "string") {
    input = document.getElementById(targetInputId);
  }

  // 2. Se não encontrou ou não foi passado ID, usar lógica inteligente
  if (!input) {
    // Procurar por textarea com foco
    const focusedElement = document.activeElement;
    if (focusedElement && focusedElement.tagName === "TEXTAREA") {
      input = focusedElement;
    }
  }

  // 3. Se ainda não encontrou, procurar por campos específicos de FAQ
  if (!input) {
    const faqFields = ["faqAnswer", "editFaqAnswer", "faqResponse"];
    for (const fieldId of faqFields) {
      const field = document.getElementById(fieldId);
      if (field && field.offsetParent !== null) {
        input = field;
        break;
      }
    }
  }

  // 4. Procurar por textarea em modal aberto
  if (!input) {
    const openModal = document.querySelector(
      '.modal[style*="display: block"], .modal.show'
    );
    if (openModal) {
      const modalTextarea = openModal.querySelector("textarea");
      if (modalTextarea && modalTextarea.offsetParent !== null) {
        input = modalTextarea;
      }
    }
  }

  // 5. Como último recurso, procurar qualquer textarea visível
  if (!input) {
    const textareas = document.querySelectorAll("textarea");
    for (const textarea of textareas) {
      if (
        textarea.offsetParent !== null &&
        !textarea.disabled &&
        !textarea.readOnly
      ) {
        input = textarea;
        break;
      }
    }
  }

  // Inserir o placeholder se encontrou um campo
  if (input) {
    const cursorPos = input.selectionStart || 0;
    const textBefore = input.value.substring(0, cursorPos);
    const textAfter = input.value.substring(input.selectionEnd || cursorPos);

    input.value = textBefore + placeholder + " " + textAfter;

    // Posicionar cursor após o placeholder
    const newPos = cursorPos + placeholder.length + 1;
    input.setSelectionRange(newPos, newPos);
    input.focus();

    console.log("Placeholder inserted successfully:", placeholder);
  } else {
    console.warn(
      "Nenhum campo de entrada encontrado para inserir placeholder:",
      placeholder
    );
    // Mostrar notificação para o usuário
    if (typeof showNotification === "function") {
      showNotification(
        "Click on a text field before using the placeholder",
        "warning"
      );
    }
  }
}

// Função para adicionar bibliotecas a um campo de texto
function addLibrariesToField(fieldId, containerSelector) {
  const container = document.querySelector(containerSelector);
  if (container && document.getElementById(fieldId)) {
    // Criar container para as bibliotecas
    const librariesContainer = document.createElement("div");
    librariesContainer.className = "input-libraries";

    // Add bibliotecas
    librariesContainer.appendChild(createEmojiLibrary(fieldId));
    librariesContainer.appendChild(createPlaceholderLibrary(fieldId));

    // Inserir após o campo
    const field = document.getElementById(fieldId);
    field.parentNode.insertBefore(librariesContainer, field.nextSibling);
  }
}

// Função para adicionar apenas emojis a um campo
function addEmojisToField(fieldId, containerSelector) {
  const container = document.querySelector(containerSelector);
  if (container && document.getElementById(fieldId)) {
    // Criar container para emojis
    const emojiContainer = document.createElement("div");
    emojiContainer.className = "input-libraries";

    // Add apenas emojis
    emojiContainer.appendChild(createEmojiLibrary(fieldId));

    // Inserir após o campo
    const field = document.getElementById(fieldId);
    field.parentNode.insertBefore(emojiContainer, field.nextSibling);
  }
}

// Função para adicionar apenas placeholders a um campo
function addPlaceholdersToField(fieldId, containerSelector) {
  const container = document.querySelector(containerSelector);
  if (container && document.getElementById(fieldId)) {
    // Criar container para placeholders
    const placeholderContainer = document.createElement("div");
    placeholderContainer.className = "input-libraries";

    // Add apenas placeholders
    placeholderContainer.appendChild(createPlaceholderLibrary(fieldId));

    // Inserir após o campo
    const field = document.getElementById(fieldId);
    field.parentNode.insertBefore(placeholderContainer, field.nextSibling);
  }
}

// ===================================================================
//  CORREÇÕES PARA PÁGINAS ESPECÍFICAS
// ===================================================================

// Função para mostrar modal de notificação push
function showCreateNotificationModal() {
  const modalHtml = `
        <div id="notificationModal" class="modal" style="display: block;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3>New Push Notification</h3>
                    <span class="close" onclick="closeNotificationModal()">&times;</span>
                </div>
                <form id="notificationForm">
                    <div class="form-group">
                        <label>Title:</label>
                        <input type="text" id="notificationTitle" required>
                    </div>
                    <div class="form-group">
                        <label>Message:</label>
                        <textarea id="notificationMessage" rows="4" required></textarea>
                    </div>
                    <div class="form-group">
                        <label>URL (optional):</label>
                        <input type="url" id="notificationUrl">
                    </div>
                    <div class="form-group">
                        <label>Icon (optional):</label>
                        <input type="text" id="notificationIcon" placeholder="Icon URL">
                    </div>
                    <div class="form-group">
                        <label>Schedule for:</label>
                        <input type="datetime-local" id="notificationSchedule">
                    </div>
                    <div class="modal-footer">
                        <button type="button" onclick="closeNotificationModal()" class="btn btn-secondary">Cancel</button>
                        <button type="submit" class="btn btn-primary">Enviar Notificação</button>
                    </div>
                </form>
            </div>
        </div>
    `;

  document.getElementById("modalContainer").innerHTML = modalHtml;

  // Add bibliotecas ao campo de mensagem
  setTimeout(() => {
    addLibrariesToField(
      "notificationMessage",
      "#notificationModal .form-group"
    );
  }, 100);

  // Configure formulário
  document.getElementById("notificationForm").onsubmit = async (e) => {
    e.preventDefault();
    await saveNotification();
  };
}

function closeNotificationModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

async function saveNotification() {
  const title = document.getElementById("notificationTitle").value;
  const message = document.getElementById("notificationMessage").value;
  const url = document.getElementById("notificationUrl").value;
  const icon = document.getElementById("notificationIcon").value;
  const schedule = document.getElementById("notificationSchedule").value;

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/push-notifications`,
      {
        method: "POST",
        body: JSON.stringify({ title, message, url, icon, schedule }),
      }
    );

    if (response.ok) {
      showNotification("Notification created successfully", "success");
      closeNotificationModal();
      await loadPushNotifications();
    } else {
      const error = await response.json();
      showNotification(error.error || "Error creating notification", "error");
    }
  } catch (error) {
    console.error("Error creating notification:", error);
    showNotification("Error creating notification", "error");
  }
}

// Função para testar conexão IA
async function testAiConnection() {
  try {
    showNotification("Testing AI connection...", "info");

    // Search configurações atuais
    const response = await authenticatedFetch(
      `${API_BASE}/api/ai-config/settings`
    );

    if (!response.ok) {
      showNotification("Error loading AI settings", "error");
      return;
    }

    const configs = await response.json();

    // Verificar se há configuração do DeepSeek
    if (!configs.deepseek || !configs.deepseek.apiKey) {
      showNotification("Configure the DeepSeek API Key first", "warning");
      return;
    }

    // Testar conexão com DeepSeek
    const testResponse = await authenticatedFetch(
      `${API_BASE}/api/ai-config/test/1`,
      {
        method: "POST",
        body: JSON.stringify({
          provider: "deepseek",
          apiKey: configs.deepseek.apiKey,
          model: configs.deepseek.model || "deepseek-chat",
          personaName: configs.deepseek.personaName || "Assistente",
          personaDescription:
            configs.deepseek.personaDescription || "Misayan Bot",
        }),
      }
    );

    const testResult = await testResponse.json();

    if (testResult.success) {
      showNotification(
        `✅ Connection successful! Response: "${testResult.response}"`,
        "success"
      );
    } else {
      showNotification(`❌ Connection failed: ${testResult.error}`, "error");
    }
  } catch (error) {
    console.error("Error testar conexão:", error);
    showNotification("Error testing AI connection", "error");
  }
}

// Corrigir função showTranslationTab
function showTranslationTab(tab) {
  // Remover classe active de todos os botões
  const buttons = document.querySelectorAll(".translation-tab-btn");
  buttons.forEach((btn) => btn.classList.remove("active"));

  // Remover classe active de todos os conteúdos
  const contents = document.querySelectorAll(".translation-tab-content");
  contents.forEach((content) => content.classList.remove("active"));

  // Add classe active ao botão clicado
  const activeButton = document.querySelector(
    `[onclick="showTranslationTab('${tab}')"]`
  );
  if (activeButton) {
    activeButton.classList.add("active");
  }

  // Add classe active ao conteúdo correspondente
  const activeContent = document.getElementById(`${tab}-translation-tab`);
  if (activeContent) {
    activeContent.classList.add("active");
  }
}

// Corrigir função filterTranslationStrings
function filterTranslationStrings() {
  const languageFilter = document.getElementById("translationLanguageFilter");
  const searchFilter = document.getElementById("translationSearch");

  if (!languageFilter || !searchFilter) {
    console.warn("Elementos de filtro de tradução not founds");
    return;
  }

  const language = languageFilter.value.toLowerCase();
  const search = searchFilter.value.toLowerCase();

  const rows = document.querySelectorAll("#translationsTable tbody tr");
  rows.forEach((row) => {
    const langCell = row.cells[0]?.textContent.toLowerCase() || "";
    const keyCell = row.cells[1]?.textContent.toLowerCase() || "";
    const valueCell = row.cells[2]?.textContent.toLowerCase() || "";

    const matchesLanguage = !language || langCell.includes(language);
    const matchesSearch =
      !search || keyCell.includes(search) || valueCell.includes(search);

    row.style.display = matchesLanguage && matchesSearch ? "" : "none";
  });
}

// ===================================================================
//  EDIÇÃO DE FAQs
// ===================================================================

// Variável global para armazenar FAQ sendo editada
let currentEditingFaq = null;

// Função para editar FAQ (implementação completa)
async function editFaq(faqId) {
  try {
    // Search dados da FAQ
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/faqs/${faqId}`
    );
    if (!response.ok) {
      throw new Error("FAQ not found");
    }

    const faq = await response.json();
    currentEditingFaq = faq;

    const modalHtml = `
            <div id="editFaqModal" class="modal" style="display: block;">
                <div class="modal-content">
                    <div class="modal-header">
                        <h3>Edit FAQ</h3>
                        <span class="close" onclick="closeEditFaqModal()">&times;</span>
                    </div>
                    <form id="editFaqForm">
                        <div class="form-group">
                            <label>Question(s):</label>
                            <textarea id="editFaqQuestion" rows="3" required placeholder="Enter one or multiple questions separated by commas...">${
                              faq.question || ""
                            }</textarea>
                            <small style="color: #666; font-size: 12px;">
                                💡 Dica: Você pode adicionar várias formas de perguntar separando por vírgula.<br>
                                Exemplo: Qual o horário?, Que horas abre?, Horário de funcionamento?
                            </small>
                        </div>
                        <div class="form-group">
                            <label>Answer:</label>
                            <textarea id="editFaqAnswer" rows="4" required>${
                              faq.answer || ""
                            }</textarea>
                        </div>
                        <div class="form-group">
                            <label>Reaction Emoji:</label>
                            <input type="text" id="editFaqEmoji" value="${
                              faq.emoji || ""
                            }" readonly placeholder="Select an emoji" style="cursor: pointer; background: #f5f5f5;">
                            <div class="emoji-selector" style="display: flex; flex-wrap: wrap; gap: 8px; margin-top: 10px; padding: 10px; background: #f9f9f9; border-radius: 5px;">
                                <span class="emoji-option" onclick="selectEmoji('❓')">❓</span><span class="emoji-option" onclick="selectEmoji('👋')">👋</span><span class="emoji-option" onclick="selectEmoji('📍')">📍</span><span class="emoji-option" onclick="selectEmoji('⏰')">⏰</span><span class="emoji-option" onclick="selectEmoji('💳')">💳</span><span class="emoji-option" onclick="selectEmoji('📦')">📦</span><span class="emoji-option" onclick="selectEmoji('✅')">✅</span><span class="emoji-option" onclick="selectEmoji('❌')">❌</span><span class="emoji-option" onclick="selectEmoji('💬')">💬</span><span class="emoji-option" onclick="selectEmoji('📞')">📞</span><span class="emoji-option" onclick="selectEmoji('🏪')">🏪</span><span class="emoji-option" onclick="selectEmoji('🎉')">🎉</span><span class="emoji-option" onclick="selectEmoji('⚠️')">⚠️</span><span class="emoji-option" onclick="selectEmoji('🔔')">🔔</span><span class="emoji-option" onclick="selectEmoji('💡')">💡</span><span class="emoji-option" onclick="selectEmoji('🚚')">🚚</span><span class="emoji-option" onclick="selectEmoji('🛒')">🛒</span><span class="emoji-option" onclick="selectEmoji('💰')">💰</span>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" onclick="closeEditFaqModal()" class="btn btn-secondary">Cancel</button>
                            <button type="submit" class="btn btn-primary">Save Alterações</button>
                        </div>
                    </form>
                </div>
            </div>
        `;

    document.getElementById("modalContainer").innerHTML = modalHtml;

    // Configure formulário
    document.getElementById("editFaqForm").onsubmit = async (e) => {
      e.preventDefault();
      await updateFaq();
    };
  } catch (error) {
    console.error("Error loading FAQ for editing:", error);
    showNotification(
      "Error loading FAQ for editing: " + error.message,
      "error"
    );
  }
}

function closeEditFaqModal() {
  document.getElementById("modalContainer").innerHTML = "";
  currentEditingFaq = null;
}

async function updateFaq() {
  if (!currentEditingFaq) {
    showNotification("Error: FAQ not loaded", "error");
    return;
  }

  const question = document.getElementById("editFaqQuestion").value;
  const answer = document.getElementById("editFaqAnswer").value;
  const emoji = document.getElementById("editFaqEmoji").value;

  if (!emoji) {
    showNotification("Please select a reaction emoji", "warning");
    return;
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/faqs/${currentEditingFaq.id}`,
      {
        method: "PUT",
        body: JSON.stringify({ question, answer, emoji }),
      }
    );

    if (response.ok) {
      showNotification("FAQ updated successfully", "success");
      closeEditFaqModal();
      await loadFaqs();
    } else {
      const error = await response.json();
      showNotification(error.error || "Error updating FAQ", "error");
    }
  } catch (error) {
    console.error("Error updating FAQ:", error);
    showNotification("Error updating FAQ", "error");
  }
}

// Modificar a função showAddFaqModal para incluir as bibliotecas
const originalShowAddFaqModal = showAddFaqModal;
showAddFaqModal = function () {
  originalShowAddFaqModal();

  // Add bibliotecas após criar o modal
  setTimeout(() => {
    // Emojis apenas no campo emoji
    addEmojisToField("faqEmoji", "#faqModal .form-group");
    // Placeholders apenas no campo de resposta
    addPlaceholdersToField("faqAnswer", "#faqModal .form-group");
  }, 100);
};

// Função para selecionar emoji
function selectEmoji(emoji) {
  const addInput = document.getElementById("faqEmoji");
  const editInput = document.getElementById("editFaqEmoji");

  if (addInput) {
    addInput.value = emoji;
  }
  if (editInput) {
    editInput.value = emoji;
  }

  // Destacar emoji selecionado
  document.querySelectorAll(".emoji-option").forEach((opt) => {
    opt.style.background = "transparent";
    opt.style.transform = "scale(1)";
  });
  event.target.style.background = "#e3f2fd";
  event.target.style.transform = "scale(1.2)";
}

console.log(
  "Biblioteca de emojis, placeholders e correções adicionadas ao admin.js"
);

// ===================================================================
//  GERENCIAMENTO DE CONTATOS E GRUPOS
// ===================================================================

// Função para carregar página de contatos
async function loadContacts() {
  try {
    await loadContactsList();
    await loadContactGroups();
    await loadContactGroupsFilter();
  } catch (error) {
    console.error("Error loading contacts:", error);
    showNotification("Error loading contacts", "error");

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Função para alternar entre abas de contatos
function showContactTab(tab) {
  // Remover classe active de todos os botões
  document
    .querySelectorAll(".tab-btn")
    .forEach((btn) => btn.classList.remove("active"));

  // Ocultar todos os conteúdos
  document
    .querySelectorAll("#contacts-page .tab-content")
    .forEach((content) => {
      content.classList.remove("active");
      content.style.display = "none";
    });

  // Add classe active ao botão clicado
  const activeButton = document.querySelector(
    `[onclick="showContactTab('${tab}')"]`
  );
  if (activeButton) {
    activeButton.classList.add("active");
  }

  // Mostrar o conteúdo correspondente
  const activeContent = document.getElementById(`${tab}-tab`);
  if (activeContent) {
    activeContent.classList.add("active");
    activeContent.style.display = "block";
  }
}

// Função para carregar lista de contatos
async function loadContactsList() {
  try {
    const response = await authenticatedFetch(`${API_BASE}/api/admin/contacts`);
    if (!response.ok) {
      throw new Error("Error loading contacts");
    }

    const contacts = await response.json();
    const tbody = document.getElementById("contactsTableBody");

    if (contacts.length === 0) {
      tbody.innerHTML =
        '<tr><td colspan="7" class="text-center">No contacts found</td></tr>';
      return;
    }

    tbody.innerHTML = contacts
      .map(
        (contact) => `
            <tr>
                <td>${contact.name || "N/A"}</td>
                <td>${contact.phone || "N/A"}</td>
                <td>${contact.email || "N/A"}</td>
                <td>${contact.group_name || "No group"}</td>
                <td>${contact.tags || ""}</td>
                <td>${new Date(contact.created_at).toLocaleDateString(
                  "pt-BR"
                )}</td>
                <td>
                    <button onclick="editContact(${
                      contact.id
                    })" class="btn btn-sm btn-primary">
                        <span data-i18n="btn.edit">${i18n.t("btn.edit")}</span>
                    </button>
                    <button onclick="deleteContact(${
                      contact.id
                    })" class="btn btn-sm btn-danger">
                        <span data-i18n="btn.delete">${i18n.t(
                          "btn.delete"
                        )}</span>
                    </button>
                </td>
            </tr>
        `
      )
      .join("");
  } catch (error) {
    console.error("Error loading contacts:", error);
    document.getElementById("contactsTableBody").innerHTML =
      '<tr><td colspan="7" class="text-center text-error">Error loading contacts</td></tr>';

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Função para carregar grupos de contatos
async function loadContactGroups() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/contact-groups`
    );
    if (!response.ok) {
      throw new Error("Error loading groups");
    }

    const groups = await response.json();
    console.log("Grupos recebidos:", groups);

    const tbody = document.getElementById("contactGroupsTableBody");

    if (!tbody) {
      console.error("Elemento contactGroupsTableBody not found no DOM");
      return;
    }

    if (groups.length === 0) {
      tbody.innerHTML =
        '<tr><td colspan="5" class="text-center">No groups found</td></tr>';
      return;
    }

    const html = groups
      .map(
        (group) => `
            <tr>
                <td>${group.name || "N/A"}</td>
                <td>${group.description || "N/A"}</td>
                <td>${group.contact_count || 0}</td>
                <td>${
                  group.created_at
                    ? new Date(group.created_at).toLocaleDateString("pt-BR")
                    : "N/A"
                }</td>
                <td>
                    <button onclick="editContactGroup(${
                      group.id
                    })" class="btn btn-sm btn-primary">
                        <span data-i18n="btn.edit">${i18n.t("btn.edit")}</span>
                    </button>
                    <button onclick="deleteContactGroup(${
                      group.id
                    })" class="btn btn-sm btn-danger">
                        <span data-i18n="btn.delete">${i18n.t(
                          "btn.delete"
                        )}</span>
                    </button>
                </td>
            </tr>
        `
      )
      .join("");

    console.log("HTML gerado:", html);
    tbody.innerHTML = html;
    console.log("Grupos carregados na tabela:", groups.length);
    console.log("tbody.innerHTML após atribuição:", tbody.innerHTML);
  } catch (error) {
    console.error("Error loading groups:", error);
    const tbody = document.getElementById("contactGroupsTableBody");
    if (tbody) {
      tbody.innerHTML =
        '<tr><td colspan="5" class="text-center text-error">Error loading groups</td></tr>';
    }

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Função para carregar filtro de grupos
async function loadContactGroupsFilter() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/contact-groups`
    );
    if (!response.ok) return;

    const groups = await response.json();
    const select = document.getElementById("contactGroupFilter");

    select.innerHTML =
      `<option value="" data-i18n="contacts.all_groups">${i18n.t(
        "contacts.all_groups"
      )}</option>` +
      groups
        .map((group) => `<option value="${group.id}">${group.name}</option>`)
        .join("");
  } catch (error) {
    console.error("Error loading group filter:", error);

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Função para buscar contatos
function searchContacts() {
  const search = document.getElementById("contactSearch").value.toLowerCase();
  const rows = document.querySelectorAll("#contactsTable tbody tr");

  rows.forEach((row) => {
    const text = row.textContent.toLowerCase();
    row.style.display = text.includes(search) ? "" : "none";
  });
}

// Função para filtrar contatos por grupo
function filterContactsByGroup() {
  const groupId = document.getElementById("contactGroupFilter").value;
  const rows = document.querySelectorAll("#contactsTable tbody tr");

  if (!groupId) {
    rows.forEach((row) => (row.style.display = ""));
    return;
  }

  // TODO: Implementar filtro por grupo quando tiver o ID do grupo na tabela
  showNotification("Group filter will be implemented soon", "info");
}

// Função para mostrar modal de adicionar contato
function showAddContactModal() {
  const modalHtml = `
        <div id="addContactModal" class="modal" style="display: block;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3>Add Contact</h3>
                    <span class="close" onclick="closeAddContactModal()">&times;</span>
                </div>
                <form id="addContactForm">
                    <div class="form-group">
                        <label>Name:</label>
                        <input type="text" id="contactName" required>
                    </div>
                    <div class="form-group">
                        <label>Phone:</label>
                        <input type="tel" id="contactPhone" required>
                    </div>
                    <div class="form-group">
                        <label>Email (optional):</label>
                        <input type="email" id="contactEmail">
                    </div>
                    <div class="form-group">
                        <label>Group:</label>
                        <select id="contactGroupSelect">
                            <option value="">No group</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>Tags (comma separated):</label>
                        <input type="text" id="contactTags" placeholder="customer, vip, discount">
                    </div>
                    <div class="modal-footer">
                        <button type="button" onclick="closeAddContactModal()" class="btn btn-secondary">Cancel</button>
                        <button type="submit" class="btn btn-primary">Add Contact</button>
                    </div>
                </form>
            </div>
        </div>
    `;

  document.getElementById("modalContainer").innerHTML = modalHtml;

  // Carregar grupos no select
  loadGroupsInSelect("contactGroupSelect");

  // Configure validação do campo de telefone com + fixo
  const phoneInput = document.getElementById("contactPhone");

  // Inicializar com +
  if (!phoneInput.value || phoneInput.value === "") {
    phoneInput.value = "+";
  }

  phoneInput.addEventListener("input", (e) => {
    let value = e.target.value;

    // Se o usuário apagar o +, restaura
    if (!value.startsWith("+")) {
      value = "+" + value.replace(/\+/g, "");
    }

    // Remove tudo que não é número após o +
    const numbers = value.substring(1).replace(/[^0-9]/g, "");
    e.target.value = "+" + numbers;
  });

  phoneInput.addEventListener("keydown", (e) => {
    // Impede apagar o + se o cursor estiver no início
    if (
      (e.key === "Backspace" || e.key === "Delete") &&
      e.target.selectionStart <= 1
    ) {
      e.preventDefault();
    }
  });

  phoneInput.addEventListener("click", (e) => {
    // Se clicar antes do +, move o cursor para depois do +
    if (e.target.selectionStart < 1) {
      e.target.setSelectionRange(1, 1);
    }
  });

  // Configure formulário
  document.getElementById("addContactForm").onsubmit = async (e) => {
    e.preventDefault();
    await saveContact();
  };
}

function closeAddContactModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

// Função para salvar contato
async function saveContact() {
  const name = document.getElementById("contactName").value;
  const phoneRaw = document.getElementById("contactPhone").value;
  // Limpar telefone: remover espaços, traços e o sinal de +
  const phone = phoneRaw.replace(/[\s\-\+]/g, "");
  const email = document.getElementById("contactEmail").value;
  const groupId = document.getElementById("contactGroupSelect").value;
  const tags = document.getElementById("contactTags").value;

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/contacts`,
      {
        method: "POST",
        body: JSON.stringify({
          name,
          phone,
          email,
          group_id: groupId || null,
          tags,
        }),
      }
    );

    if (response.ok) {
      showNotification("Contact added successfully", "success");
      closeAddContactModal();
      await loadContactsList();
    } else {
      const error = await response.json();
      showNotification(error.error || "Error adding contact", "error");
    }
  } catch (error) {
    console.error("Error adding contact:", error);
    showNotification("Error adding contact", "error");
  }
}

// Função para editar contato
async function editContact(contactId) {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/contacts/${contactId}`
    );
    if (!response.ok) {
      throw new Error("Contact not found");
    }

    const contact = await response.json();

    const modalHtml = `
            <div id="editContactModal" class="modal" style="display: block;">
                <div class="modal-content">
                    <div class="modal-header">
                        <h3>Edit Contact</h3>
                        <span class="close" onclick="closeEditContactModal()">&times;</span>
                    </div>
                    <form id="editContactForm">
                        <div class="form-group">
                            <label>Name:</label>
                            <input type="text" id="editContactName" value="${
                              contact.name || ""
                            }" required>
                        </div>
                        <div class="form-group">
                            <label>Phone:</label>
                            <input type="tel" id="editContactPhone" value="${
                              contact.phone || ""
                            }" required>
                        </div>
                        <div class="form-group">
                            <label>Email (optional):</label>
                            <input type="email" id="editContactEmail" value="${
                              contact.email || ""
                            }">
                        </div>
                        <div class="form-group">
                            <label>Group:</label>
                            <select id="editContactGroupSelect">
                                <option value="">No group</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Tags (comma separated):</label>
                            <input type="text" id="editContactTags" value="${
                              contact.tags || ""
                            }" placeholder="customer, vip, discount">
                        </div>
                        <div class="modal-footer">
                            <button type="button" onclick="closeEditContactModal()" class="btn btn-secondary">Cancel</button>
                            <button type="submit" class="btn btn-primary">Save Alterações</button>
                        </div>
                    </form>
                </div>
            </div>
        `;

    document.getElementById("modalContainer").innerHTML = modalHtml;

    // Carregar grupos no select
    await loadGroupsInSelect("editContactGroupSelect", contact.group_id);

    // Configure validação do campo de telefone com + fixo
    const phoneInput = document.getElementById("editContactPhone");

    // Se o valor não começar com +, adiciona
    if (phoneInput.value && !phoneInput.value.startsWith("+")) {
      phoneInput.value = "+" + phoneInput.value.replace(/[^0-9]/g, "");
    } else if (!phoneInput.value) {
      phoneInput.value = "+";
    }

    phoneInput.addEventListener("input", (e) => {
      let value = e.target.value;

      // Se o usuário apagar o +, restaura
      if (!value.startsWith("+")) {
        value = "+" + value.replace(/\+/g, "");
      }

      // Remove tudo que não é número após o +
      const numbers = value.substring(1).replace(/[^0-9]/g, "");
      e.target.value = "+" + numbers;
    });

    phoneInput.addEventListener("keydown", (e) => {
      // Impede apagar o + se o cursor estiver no início
      if (
        (e.key === "Backspace" || e.key === "Delete") &&
        e.target.selectionStart <= 1
      ) {
        e.preventDefault();
      }
    });

    phoneInput.addEventListener("click", (e) => {
      // Se clicar antes do +, move o cursor para depois do +
      if (e.target.selectionStart < 1) {
        e.target.setSelectionRange(1, 1);
      }
    });

    // Configure formulário
    document.getElementById("editContactForm").onsubmit = async (e) => {
      e.preventDefault();
      await updateContact(contactId);
    };
  } catch (error) {
    console.error("Error loading contact:", error);
    showNotification("Error loading contact", "error");
  }
}

function closeEditContactModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

// Função para atualizar contato
async function updateContact(contactId) {
  const name = document.getElementById("editContactName").value;
  const phoneRaw = document.getElementById("editContactPhone").value;
  // Limpar telefone: remover espaços, traços e o sinal de +
  const phone = phoneRaw.replace(/[\s\-\+]/g, "");
  const email = document.getElementById("editContactEmail").value;
  const groupId = document.getElementById("editContactGroupSelect").value;
  const tags = document.getElementById("editContactTags").value;

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/contacts/${contactId}`,
      {
        method: "PUT",
        body: JSON.stringify({
          name,
          phone,
          email,
          group_id: groupId || null,
          tags,
        }),
      }
    );

    if (response.ok) {
      showNotification("Contact updated successfully", "success");
      closeEditContactModal();
      await loadContactsList();
    } else {
      const error = await response.json();
      showNotification(error.error || "Error updating contact", "error");
    }
  } catch (error) {
    console.error("Error updating contact:", error);
    showNotification("Error updating contact", "error");
  }
}

// Função para excluir contato
async function deleteContact(contactId) {
  if (!confirm("Are you sure you want to delete this contact?")) {
    return;
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/contacts/${contactId}`,
      {
        method: "DELETE",
      }
    );

    if (response.ok) {
      showNotification("Contact deleted successfully", "success");
      await loadContactsList();
    } else {
      const error = await response.json();
      showNotification(error.error || "Error deleting contact", "error");
    }
  } catch (error) {
    console.error("Error deleting contact:", error);
    showNotification("Error deleting contact", "error");
  }
}

// Função para editar grupo de contatos
async function editContactGroup(groupId) {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/contact-groups/${groupId}`
    );
    if (!response.ok) {
      throw new Error("Grupo not found");
    }

    const group = await response.json();

    const modalHtml = `
            <div id="editGroupModal" class="modal" style="display: block;">
                <div class="modal-content">
                    <div class="modal-header">
                        <h3>Edit Group</h3>
                        <span class="close" onclick="closeEditGroupModal()">&times;</span>
                    </div>
                    <form id="editGroupForm">
                        <div class="form-group">
                            <label>Group Name:</label>
                            <input type="text" id="editGroupName" value="${
                              group.name || ""
                            }" required>
                        </div>
                        <div class="form-group">
                            <label>Description:</label>
                            <textarea id="editGroupDescription" rows="3">${
                              group.description || ""
                            }</textarea>
                        </div>
                        <div class="modal-footer">
                            <button type="button" onclick="closeEditGroupModal()" class="btn btn-secondary">Cancel</button>
                            <button type="submit" class="btn btn-primary">Save Alterações</button>
                        </div>
                    </form>
                </div>
            </div>
        `;

    document.getElementById("modalContainer").innerHTML = modalHtml;

    // Configure formulário
    document.getElementById("editGroupForm").onsubmit = async (e) => {
      e.preventDefault();
      await updateContactGroup(groupId);
    };
  } catch (error) {
    console.error("Error loading group:", error);
    showNotification("Error loading group", "error");
  }
}

function closeEditGroupModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

// Função para atualizar grupo de contatos
async function updateContactGroup(groupId) {
  const name = document.getElementById("editGroupName").value;
  const description = document.getElementById("editGroupDescription").value;

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/contact-groups/${groupId}`,
      {
        method: "PUT",
        body: JSON.stringify({ name, description }),
      }
    );

    if (response.ok) {
      showNotification("Group updated successfully", "success");
      closeEditGroupModal();
      await loadContactGroups();
      await loadContactGroupsFilter();
    } else {
      const error = await response.json();
      showNotification(error.error || "Error updating group", "error");
    }
  } catch (error) {
    console.error("Error updating group:", error);
    showNotification("Error updating group", "error");
  }
}

// Função para excluir grupo de contatos
async function deleteContactGroup(groupId) {
  if (
    !confirm(
      "Are you sure you want to delete this group? Contacts will not be deleted, only unlinked from the group."
    )
  ) {
    return;
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/contact-groups/${groupId}`,
      {
        method: "DELETE",
      }
    );

    if (response.ok) {
      showNotification("Group deleted successfully", "success");
      await loadContactGroups();
      await loadContactGroupsFilter();
    } else {
      const error = await response.json();
      showNotification(error.error || "Error deleting group", "error");
    }
  } catch (error) {
    console.error("Error deleting group:", error);
    showNotification("Error deleting group", "error");
  }
}

// Função auxiliar para carregar grupos em um select
async function loadGroupsInSelect(selectId, selectedValue = null) {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/contact-groups`
    );
    if (!response.ok) return;

    const groups = await response.json();
    const select = document.getElementById(selectId);

    select.innerHTML =
      '<option value="">No group</option>' +
      groups
        .map(
          (group) =>
            `<option value="${group.id}" ${
              group.id == selectedValue ? "selected" : ""
            }>${group.name}</option>`
        )
        .join("");
  } catch (error) {
    console.error("Error loading groups:", error);

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

console.log("Funções de gerenciamento de contatos adicionadas ao admin.js");

// Função para atualizar todos os selects que usam grupos
async function updateGroupSelects() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/contact-groups`
    );
    if (!response.ok) return;

    const groups = await response.json();

    // Lista de IDs dos selects que precisam ser atualizados
    const selectIds = [
      "contactGroupFilter",
      "campaignGroup",
      "importGroup",
      "contactGroupSelect",
      "editContactGroupSelect",
    ];

    selectIds.forEach((selectId) => {
      const select = document.getElementById(selectId);
      if (select) {
        const currentValue = select.value;

        if (selectId === "contactGroupFilter") {
          select.innerHTML = `<option value="" data-i18n="contacts.all_groups">${i18n.t(
            "contacts.all_groups"
          )}</option>`;
        } else if (selectId === "campaignGroup") {
          select.innerHTML = '<option value="">Select a group</option>';
        } else if (selectId === "importGroup") {
          select.innerHTML = '<option value="">Select a group</option>';
        } else {
          select.innerHTML = '<option value="">No group</option>';
        }

        // Add opções dos grupos
        groups.forEach((group) => {
          const option = document.createElement("option");
          option.value = group.id;
          option.textContent = group.name;
          if (group.id == currentValue) {
            option.selected = true;
          }
          select.appendChild(option);
        });

        console.log(
          `Select ${selectId} atualizado com ${groups.length} grupos`
        );
      }
    });
  } catch (error) {
    console.error("Error updating group selects:", error);
  }
}

// Função melhorada para loadCampaigns com tratamento de erro
async function loadCampaigns() {
  try {
    console.log("Trying to load campaigns...");

    // Primeiro verificar se o endpoint existe
    const testResponse = await fetch(`${API_BASE}/api/admin/campaigns`, {
      method: "HEAD",
      headers: {
        Authorization: `Bearer ${localStorage.getItem("authToken")}`,
      },
    });

    if (!testResponse.ok) {
      throw new Error(`Endpoint not available: HTTP ${testResponse.status}`);
    }

    // Se chegou até aqui, o endpoint existe, fazer a requisição real
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/campaigns`
    );

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }

    const campaigns = await response.json();
    console.log("Campanhas carregadas:", campaigns);

    const tbody = document.getElementById("campaignsTableBody");
    if (!tbody) {
      console.error("Elemento campaignsTableBody not found");
      return;
    }

    if (!campaigns || campaigns.length === 0) {
      tbody.innerHTML =
        '<tr><td colspan="7" class="text-center">No campaigns found</td></tr>';
      return;
    }

    tbody.innerHTML = campaigns
      .map(
        (campaign) => `
            <tr>
                <td>${campaign.name || "N/A"}</td>
                <td>${campaign.group_name || "N/A"}</td>
                <td>
                    <span class="badge badge-${getStatusClass(
                      campaign.status
                    )}">
                        ${getStatusText(campaign.status)}
                    </span>
                </td>
                <td>${campaign.sent_count || 0}</td>
                <td>${campaign.success_count || 0}</td>
                <td>${
                  campaign.created_at
                    ? new Date(campaign.created_at).toLocaleDateString("pt-BR")
                    : "N/A"
                }</td>
                <td>
                    <button onclick="viewCampaign(${
                      campaign.id
                    })" class="btn btn-sm btn-info">Ver</button>
                    <button onclick="editCampaign(${
                      campaign.id
                    })" class="btn btn-sm btn-secondary">Edit</button>
                    <button onclick="deleteCampaign(${
                      campaign.id
                    })" class="btn btn-sm btn-danger">Delete</button>
                </td>
            </tr>
        `
      )
      .join("");
  } catch (error) {
    console.error("Error loading campaigns:", error);
    const tbody = document.getElementById("campaignsTableBody");
    if (tbody) {
      if (
        error.message.includes("404") ||
        error.message.includes("não disponível")
      ) {
        tbody.innerHTML = `
                    <tr>
                        <td colspan="7" class="text-center">
                            <div style="padding: 20px; color: #666;">
                                <p><strong>Funcionalidade em Desenvolvimento</strong></p>
                                <p>O sistema de campanhas será implementado em breve.</p>
                                <p>Por enquanto, você pode gerenciar grupos de contatos na aba "Grupos de Contacts".</p>
                            </div>
                        </td>
                    </tr>
                `;
      } else {
        tbody.innerHTML = `
                    <tr>
                        <td colspan="7" class="text-center text-error">
                            Error loading campaigns: ${error.message}
                        </td>
                    </tr>
                `;
      }
    }

    // No mostrar notificação de erro se for apenas endpoint não implementado
    if (
      !error.message.includes("404") &&
      !error.message.includes("não disponível")
    ) {
      showNotification("Error loading campaigns: " + error.message, "error");
    }

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Função para garantir que os grupos sejam carregados quando necessário
async function ensureGroupsLoaded() {
  try {
    // Verificar se já temos grupos carregados
    const testSelect = document.getElementById("massGroupSelect");
    if (testSelect && testSelect.children.length <= 1) {
      // No temos grupos carregados, vamos carregar
      await loadMassGroupSelects();
    }
  } catch (error) {
    console.error("Error garantir carregamento dos grupos:", error);
  }
}

// Função para carregar grupos no select de envio em massa
async function loadMassGroupSelects() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/mass-messaging/groups`
    );

    if (!response.ok) {
      throw new Error("Error loading groups");
    }

    const groups = await response.json();

    const select = document.getElementById("massGroupSelect");
    if (select) {
      select.innerHTML = '<option value="">Select a group</option>';

      groups.forEach((group) => {
        const option = document.createElement("option");
        option.value = group.id;
        option.textContent = `${group.group_name} (${
          group.contact_count || 0
        } contatos)`;
        select.appendChild(option);
      });

      // Add evento onchange
      select.onchange = loadMassGroupContacts;
    }
  } catch (error) {
    console.error("Error loading groups para envio em massa:", error);
    const select = document.getElementById("massGroupSelect");
    if (select) {
      select.innerHTML = '<option value="">Error loading groups</option>';
    }

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Modificar a função showPage para garantir carregamento correto
const originalShowPage = showPage;
showPage = async function (page) {
  await originalShowPage(page);

  // Carregar grupos quando necessário
  if (page === "mass-messaging") {
    setTimeout(async () => {
      await loadMassGroupSelects();
    }, 500);
  } else if (page === "contacts") {
    setTimeout(async () => {
      await ensureGroupsLoaded();
    }, 500);
  }
};

// Função para debug - verificar estado dos grupos
function debugGroups() {
  console.log("=== DEBUG GRUPOS ===");

  // Verificar selects
  const selects = ["contactGroupFilter", "campaignGroup", "importGroup"];
  selects.forEach((selectId) => {
    const select = document.getElementById(selectId);
    if (select) {
      console.log(`Select ${selectId}:`, select.children.length, "opções");
    } else {
      console.log(`Select ${selectId}: NOT FOUND`);
    }
  });

  // Verificar tabelas
  const tables = ["contactGroupsTableBody", "campaignsTableBody"];
  tables.forEach((tableId) => {
    const table = document.getElementById(tableId);
    if (table) {
      console.log(`Tabela ${tableId}:`, table.children.length, "linhas");
    } else {
      console.log(`Tabela ${tableId}: NOT FOUND`);
    }
  });
}

// Função para forçar reload dos grupos
async function forceReloadGroups() {
  try {
    console.log("Forçando reload dos grupos...");

    // Recarregar grupos
    await loadContactGroups();

    // Atualizar selects
    await updateGroupSelects();

    // Se estivermos na página de contatos, recarregar filtros
    if (currentPage === "contacts") {
      await loadContactGroupsFilter();
    }

    showNotification("Grupos recarregados successfully", "success");
  } catch (error) {
    console.error("Error forçar reload dos grupos:", error);
    showNotification("Error reloading groups", "error");
  }
}

// Add ao objeto global
window.forceReloadGroups = forceReloadGroups;

// Funções auxiliares para status das campanhas
function getStatusClass(status) {
  const statusClasses = {
    draft: "secondary",
    pending: "warning",
    sending: "primary",
    sent: "success",
    completed: "success",
    failed: "danger",
    cancelled: "dark",
  };
  return statusClasses[status] || "secondary";
}

function getStatusText(status) {
  const statusTexts = {
    draft: "Rascunho",
    pending: "Pending",
    sending: "Enviando",
    sent: "Enviada",
    completed: "Completa",
    failed: "Failed",
    cancelled: "Cancelada",
  };
  return statusTexts[status] || status;
}

// Função corrigida para showCreateCampaignModal - só mostrar se campanhas estiverem disponíveis
function showCreateCampaignModal() {
  // Verificar se o sistema de campanhas está disponível
  const tbody = document.getElementById("campaignsTableBody");
  if (tbody && tbody.innerHTML.includes("Funcionalidade em Desenvolvimento")) {
    showNotification(
      "Campaign system is not yet available",
      "warning"
    );
    return;
  }

  const modalHtml = `
        <div id="campaignModal" class="modal" style="display: block;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3>New Campaign</h3>
                    <span class="close" onclick="closeCampaignModal()">&times;</span>
                </div>
                <form id="campaignForm">
                    <div class="form-group">
                        <label>Campaign Name:</label>
                        <input type="text" id="campaignName" required placeholder="Ex: Summer Promotion">
                    </div>
                    <div class="form-group">
                        <label>Message:</label>
                        <textarea id="campaignMessage" rows="4" required placeholder="Type your message here..."></textarea>
                    </div>
                    <div class="form-group">
                        <label>Contact Group:</label>
                        <select id="campaignGroup" required>
                            <option value="">Loading grupos...</option>
                        </select>
                        <small>If no groups are available, <a href="#contacts" onclick="showPage('contacts')">create a group first</a></small>
                    </div>
                    <div class="form-group">
                        <label>Schedule for (optional):</label>
                        <input type="datetime-local" id="campaignSchedule">
                        <small>Deixe vazio para enviar imediatamente</small>
                    </div>
                    <div class="modal-footer">
                        <button type="button" onclick="closeCampaignModal()" class="btn btn-secondary">Cancel</button>
                        <button type="submit" class="btn btn-primary">Criar Campanha</button>
                    </div>
                </form>
            </div>
        </div>
    `;

  document.getElementById("modalContainer").innerHTML = modalHtml;

  // Carregar grupos no select após criar o modal
  setTimeout(async () => {
    await updateGroupSelects();

    // Verificar se há grupos disponíveis
    const select = document.getElementById("campaignGroup");
    if (select && select.children.length <= 1) {
      select.innerHTML = `
                <option value="">No groups found</option>
                <option value="" disabled>Crie um grupo primeiro na aba Grupos</option>
            `;
    }

    // Add bibliotecas ao campo de mensagem
    if (typeof addLibrariesToField === "function") {
      addLibrariesToField("campaignMessage", "#campaignModal .form-group");
    }
  }, 100);

  // Configure formulário
  document.getElementById("campaignForm").onsubmit = async (e) => {
    e.preventDefault();
    await saveCampaign();
  };
}

// Função corrigida para saveCampaign - com melhor tratamento de erro
async function saveCampaign() {
  const name = document.getElementById("campaignName").value.trim();
  const message = document.getElementById("campaignMessage").value.trim();
  const groupId = document.getElementById("campaignGroup").value;
  const schedule = document.getElementById("campaignSchedule").value;

  if (!name || !message) {
    showNotification("Campaign name and message are required", "error");
    return;
  }

  if (!groupId) {
    showNotification("Select a contact group", "error");
    return;
  }

  const submitBtn = document.querySelector(
    '#campaignForm button[type="submit"]'
  );
  if (submitBtn) {
    const originalText = submitBtn.textContent;
    submitBtn.disabled = true;
    submitBtn.textContent = "Creating...";

    try {
      console.log("Salvando campanha:", { name, message, groupId, schedule });

      const response = await authenticatedFetch(
        `${API_BASE}/api/admin/campaigns`,
        {
          method: "POST",
          body: JSON.stringify({
            name,
            message,
            group_id: groupId,
            schedule: schedule || null,
          }),
        }
      );

      if (!response.ok) {
        const errorText = await response.text();
        let errorMessage = "Error creating campaign";

        try {
          const errorJson = JSON.parse(errorText);
          errorMessage = errorJson.error || errorMessage;
        } catch {
          if (response.status === 404) {
            errorMessage =
              "Sistema de campanhas ainda não implementado no backend";
          } else {
            errorMessage = `Erro HTTP ${response.status}: ${response.statusText}`;
          }
        }

        throw new Error(errorMessage);
      }

      const result = await response.json();

      showNotification("Campaign created successfully", "success");
      closeCampaignModal();
      await loadCampaigns();
    } catch (error) {
      console.error("Error creating campaign:", error);

      if (
        error.message.includes("ainda não implementado") ||
        error.message.includes("404")
      ) {
        showNotification(
          "Campaign system is not yet available no backend",
          "warning"
        );
      } else {
        showNotification(error.message, "error");
      }
    } finally {
      if (submitBtn) {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
      }
    }
  }
}

// Funções auxiliares para campanhas (mesmo que não implementadas ainda)
function viewCampaign(campaignId) {
  showNotification(
    "Campaign preview will be implemented soon",
    "info"
  );
}

function editCampaign(campaignId) {
  showNotification("Campaign editing will be implemented soon", "info");
}

function deleteCampaign(campaignId) {
  if (confirm("Are you sure you want to delete this campaign?")) {
    showNotification("Campaign deletion will be implemented soon", "info");
  }
}

function closeCampaignModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

console.log("Funções corrigidas para campanhas adicionadas ao admin.js");
// ===== FUNÇÕES PARA GERENCIAR SETORES (DEPARTMENTS) =====

// Carregar lista de setores
async function loadDepartments() {
  try {
    console.log("Loading setores...");
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/departments`
    );

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }

    const departments = await response.json();
    console.log("Departments carregados:", departments);

    const container = document.getElementById("departmentsTableBody");

    if (!container) {
      console.error("Container departmentsTableBody not found");
      return;
    }

    container.innerHTML = "";

    if (departments.length === 0) {
      container.innerHTML =
        '<tr><td colspan="5">No departments found</td></tr>';
      return;
    }

    // Search contagem de usuários por setor
    const usersResponse = await authenticatedFetch(
      `${API_BASE}/api/admin/users`
    );
    const users = await usersResponse.json();

    departments.forEach((dept) => {
      const userCount = users.filter((u) => u.department === dept.name).length;

      const row = document.createElement("tr");
      row.innerHTML = `
                <td>${dept.name}</td>
                <td>${userCount}</td>
                <td>${formatDate(dept.created_at)}</td>
                <td>
                    <button onclick="editDepartment(${
                      dept.id
                    })" class="btn btn-secondary btn-sm">
                        <span data-i18n="btn.edit">${i18n.t("btn.edit")}</span>
                    </button>
                    <button onclick="deleteDepartment(${
                      dept.id
                    })" class="btn btn-danger btn-sm">
                        <span data-i18n="btn.delete">${i18n.t(
                          "btn.delete"
                        )}</span>
                    </button>
                </td>
            `;
      container.appendChild(row);
    });
  } catch (error) {
    console.error("Error loading departments:", error);
    showNotification(
      i18n.t("notification.departments_load_error") + ": " + error.message,
      "error"
    );

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }
}

// Mostrar modal para adicionar setor
function showAddDepartmentModal() {
  const modalContainer = document.getElementById("modalContainer");
  if (!modalContainer) {
    console.error("Elemento modalContainer not found no DOM");
    showNotification("Error: Modal container not found", "error");
    return;
  }

  const modalHtml = `
        <div id="departmentModal" class="modal" style="display: block;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3>Add Department</h3>
                    <span class="close" onclick="closeDepartmentModal()">&times;</span>
                </div>
                <form id="departmentForm">
                    <div class="form-group">
                        <label>Department Name:</label>
                        <input type="text" id="departmentName" required placeholder="Ex: Finance, Support, HR">
                    </div>
                    <div class="form-group">
                        <label>Description:</label>
                        <textarea id="departmentDescription" rows="3" placeholder="Optional department description"></textarea>
                    </div>
                    <div class="modal-footer">
                        <button type="button" onclick="closeDepartmentModal()" class="btn btn-secondary">Cancel</button>
                        <button type="submit" class="btn btn-primary">Save</button>
                    </div>
                </form>
            </div>
        </div>
    `;

  modalContainer.innerHTML = modalHtml;

  // Configure formulário
  setTimeout(() => {
    const form = document.getElementById("departmentForm");
    if (form) {
      form.onsubmit = async (e) => {
        e.preventDefault();
        await saveDepartment();
      };
    }
  }, 10);
}

// Fechar modal de setor
function closeDepartmentModal() {
  document.getElementById("modalContainer").innerHTML = "";
}

// Save novo setor
async function saveDepartment() {
  const name = document.getElementById("departmentName").value.trim();
  const description = document
    .getElementById("departmentDescription")
    .value.trim();

  if (!name) {
    showNotification("Department name is required", "error");
    return;
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/departments`,
      {
        method: "POST",
        body: JSON.stringify({ name, description }),
      }
    );

    if (response.ok) {
      showNotification(i18n.t("notification.department_created"), "success");
      closeDepartmentModal();
      await loadDepartments();
    } else {
      const error = await response.json();
      showNotification(
        error.error || i18n.t("notification.department_create_error"),
        "error"
      );
    }
  } catch (error) {
    console.error("Error creating department:", error);
    showNotification(i18n.t("notification.department_create_error"), "error");
  }
}

// Edit setor
async function editDepartment(id) {
  try {
    // Search dados do setor
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/departments`
    );
    const departments = await response.json();
    const department = departments.find((d) => d.id === id);

    if (!department) {
      showNotification(i18n.t("notification.error"), "error");
      return;
    }

    const modalContainer = document.getElementById("modalContainer");
    const modalHtml = `
            <div id="departmentModal" class="modal" style="display: block;">
                <div class="modal-content">
                    <div class="modal-header">
                        <h3 data-i18n="departments.edit">Edit Department</h3>
                        <span class="close" onclick="closeDepartmentModal()">&times;</span>
                    </div>
                    <form id="departmentEditForm">
                        <div class="form-group">
                            <label>Department Name:</label>
                            <input type="text" id="editDepartmentName" value="${
                              department.name
                            }" required>
                        </div>
                        <div class="form-group">
                            <label>Description:</label>
                            <textarea id="editDepartmentDescription" rows="3">${
                              department.description || ""
                            }</textarea>
                        </div>
                        <div class="modal-footer">
                            <button type="button" onclick="closeDepartmentModal()" class="btn btn-secondary">
                                <span data-i18n="btn.cancel">Cancel</span>
                            </button>
                            <button type="submit" class="btn btn-primary">
                                <span data-i18n="btn.save">Save</span>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        `;

    modalContainer.innerHTML = modalHtml;

    // Atualizar traduções
    i18n.updatePageTranslations();

    setTimeout(() => {
      const form = document.getElementById("departmentEditForm");
      if (form) {
        form.onsubmit = async (e) => {
          e.preventDefault();
          await updateDepartment(id);
        };
      }
    }, 10);
  } catch (error) {
    console.error("Error editar setor:", error);
    showNotification(i18n.t("notification.department_create_error"), "error");
  }
}

// Atualizar setor
async function updateDepartment(id) {
  const name = document.getElementById("editDepartmentName").value.trim();
  const description = document
    .getElementById("editDepartmentDescription")
    .value.trim();

  if (!name) {
    showNotification(i18n.t("notification.username_required"), "error");
    return;
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/departments/${id}`,
      {
        method: "PUT",
        body: JSON.stringify({ name, description }),
      }
    );

    if (response.ok) {
      showNotification(i18n.t("notification.updated"), "success");
      closeDepartmentModal();
      await loadDepartments();
    } else {
      const error = await response.json();
      showNotification(
        error.error || i18n.t("notification.department_create_error"),
        "error"
      );
    }
  } catch (error) {
    console.error("Error updating department:", error);
    showNotification(i18n.t("notification.department_create_error"), "error");
  }
}

// Deletar setor
async function deleteDepartment(id) {
  if (
    !confirm(
      "Are you sure you want to delete this department? This action cannot be undone."
    )
  ) {
    return;
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/departments/${id}`,
      {
        method: "DELETE",
      }
    );

    if (response.ok) {
      showNotification(i18n.t("notification.department_deleted"), "success");
      await loadDepartments();
    } else {
      const error = await response.json();
      showNotification(
        error.error || i18n.t("notification.department_delete_error"),
        "error"
      );
    }
  } catch (error) {
    console.error("Error deleting department:", error);
    showNotification("Error deleting department", "error");
  }
}

// ===================================================================
//  FUNÇÕES DE ENVIO EM MASSA - IMPLEMENTAÇÃO COMPLETA
// ===================================================================

// Variáveis globais para envio em massa
let massMediaFile = null;
let massMediaType = null;
let selectedMassContacts = [];

// Função para inserir placeholder no textarea de envio em massa
function insertMassMessagePlaceholder(placeholder) {
  const textarea = document.getElementById("massMessageText");
  if (!textarea) {
    console.warn("Campo massMessageText not found");
    return;
  }

  const cursorPos = textarea.selectionStart || 0;
  const textBefore = textarea.value.substring(0, cursorPos);
  const textAfter = textarea.value.substring(
    textarea.selectionEnd || cursorPos
  );

  textarea.value = textBefore + placeholder + " " + textAfter;

  // Posicionar cursor após o placeholder
  const newPos = cursorPos + placeholder.length + 1;
  textarea.setSelectionRange(newPos, newPos);
  textarea.focus();

  updateMassPreview();
}

// Função para atualizar preview da mensagem
function updateMassPreview() {
  const textarea = document.getElementById("massMessageText");
  const previewText = document.getElementById("previewMessageText");
  const charCount = document.getElementById("charCount");
  const previewTime = document.getElementById("previewTime");

  if (!textarea || !previewText) return;

  // Atualizar contador de caracteres
  if (charCount) {
    charCount.textContent = textarea.value.length;
  }

  // Atualizar hora no preview
  if (previewTime) {
    const now = new Date();
    previewTime.textContent = now.toLocaleTimeString("pt-BR", {
      hour: "2-digit",
      minute: "2-digit",
    });
  }

  // Substituir placeholders por exemplos
  let previewMessage = textarea.value;
  previewMessage = previewMessage.replace(/\{\{nome\}\}/g, "João Silva");
  previewMessage = previewMessage.replace(
    /\{\{telefone\}\}/g,
    "(11) 98765-4321"
  );
  previewMessage = previewMessage.replace(/\{\{email\}\}/g, "joao@email.com");
  previewMessage = previewMessage.replace(/\{\{grupo\}\}/g, "Clientes VIP");

  // Atualizar texto do preview
  if (previewMessage.trim() === "") {
    previewText.textContent = "Type a message to see the preview...";
    previewText.style.color = "#999";
  } else {
    previewText.textContent = previewMessage;
    previewText.style.color = "#111";
  }

  // Atualizar preview de mídia
  updateMediaPreview();
}

// Função para lidar com upload de mídia
function handleMassMediaUpload(input, type) {
  const file = input.files[0];
  if (!file) return;

  // Validar tamanho do arquivo (máximo 16MB para WhatsApp)
  const maxSize = 16 * 1024 * 1024; // 16MB
  if (file.size > maxSize) {
    showNotification("File too large. Maximum: 16MB", "error");
    input.value = "";
    return;
  }

  massMediaFile = file;
  massMediaType = type;

  // Mostrar preview do arquivo
  const previewContainer = document.getElementById("massMediaPreviewContainer");
  const previewContent = document.getElementById("massMediaPreviewContent");

  if (!previewContainer || !previewContent) return;

  previewContainer.style.display = "block";

  const reader = new FileReader();
  reader.onload = function (e) {
    if (type === "image") {
      previewContent.innerHTML = `<img src="${e.target.result}" style="max-width: 200px; border-radius: 8px;">`;
    } else if (type === "video") {
      previewContent.innerHTML = `<video src="${e.target.result}" controls style="max-width: 200px; border-radius: 8px;"></video>`;
    } else if (type === "audio") {
      previewContent.innerHTML = `<audio src="${e.target.result}" controls style="width: 100%;"></audio>`;
    }
    previewContent.innerHTML += `<p style="margin-top: 10px; font-size: 14px;"><strong>${
      file.name
    }</strong> (${(file.size / 1024 / 1024).toFixed(2)} MB)</p>`;

    updateMediaPreview();
  };
  reader.readAsDataURL(file);
}

// Função para remover mídia
function removeMassMedia() {
  massMediaFile = null;
  massMediaType = null;

  const previewContainer = document.getElementById("massMediaPreviewContainer");
  const previewContent = document.getElementById("massMediaPreviewContent");

  if (previewContainer) previewContainer.style.display = "none";
  if (previewContent) previewContent.innerHTML = "";

  // Limpar inputs de arquivo
  document.getElementById("massImageUpload").value = "";
  document.getElementById("massVideoUpload").value = "";
  document.getElementById("massAudioUpload").value = "";

  updateMediaPreview();
}

// Função para atualizar preview de mídia no mockup do celular
function updateMediaPreview() {
  const previewMediaContent = document.getElementById("previewMediaContent");
  if (!previewMediaContent) return;

  if (massMediaFile && massMediaType) {
    const reader = new FileReader();
    reader.onload = function (e) {
      if (massMediaType === "image") {
        previewMediaContent.innerHTML = `<img src="${e.target.result}" style="max-width: 100%; border-radius: 8px;">`;
      } else if (massMediaType === "video") {
        previewMediaContent.innerHTML = `<video src="${e.target.result}" controls style="max-width: 100%; border-radius: 8px;"></video>`;
      } else if (massMediaType === "audio") {
        previewMediaContent.innerHTML = `<audio src="${e.target.result}" controls style="width: 100%;"></audio>`;
      }
    };
    reader.readAsDataURL(massMediaFile);
  } else {
    previewMediaContent.innerHTML = "";
  }
}

// Função para alternar entre abas de destinatários
function switchMassRecipientTab(tab) {
  // Remover classe active de todos os botões
  document.querySelectorAll(".recipient-tab").forEach((btn) => {
    btn.classList.remove("active");
  });

  // Ocultar todos os conteúdos
  document.querySelectorAll(".recipient-tab-content").forEach((content) => {
    content.style.display = "none";
  });

  // Activer aba selecionada
  const activeBtn = document.querySelector(`[data-tab="${tab}"]`);
  if (activeBtn) activeBtn.classList.add("active");

  const activeContent = document.getElementById(
    `mass${tab.charAt(0).toUpperCase() + tab.slice(1)}RecipientTab`
  );
  if (activeContent) activeContent.style.display = "block";
}

// Função para carregar contatos do grupo selecionado
async function loadMassGroupContacts() {
  const groupSelect = document.getElementById("massGroupSelect");
  const infoDiv = document.getElementById("massGroupContactsInfo");

  if (!groupSelect || !infoDiv) return;

  const groupId = groupSelect.value;

  if (!groupId) {
    infoDiv.innerHTML = "";
    return;

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/mass-messaging/contacts?group_id=${groupId}`
    );

    if (!response.ok) {
      throw new Error("Error loading group contacts");
    }

    const data = await response.json();
    const contacts = data.contacts || [];

    if (contacts.length === 0) {
      infoDiv.innerHTML =
        '<p style="color: #dc3545;">⚠️ Este grupo não possui contatos.</p>';
    } else {
      infoDiv.innerHTML = `<p style="color: #28a745;">✅ ${contacts.length} contato(s) selecionado(s)</p>`;
    }
  } catch (error) {
    console.error("Error loading group contacts:", error);
    infoDiv.innerHTML =
      '<p style="color: #dc3545;">Error loading contacts</p>';
  }
}

// Função para buscar contatos individuais
async function searchMassContacts() {
  const searchInput = document.getElementById("massContactSearch");
  const contactsList = document.getElementById("massContactsList");

  if (!searchInput || !contactsList) return;

  const search = searchInput.value.trim();

  if (search.length < 2) {
    contactsList.innerHTML =
      '<p style="padding: 15px; text-align: center; color: #999;">Digite pelo menos 2 caracteres para buscar</p>';
    return;
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/mass-messaging/contacts?search=${encodeURIComponent(
        search
      )}`
    );

    if (!response.ok) {
      throw new Error("Error fetching contacts");
    }

    const data = await response.json();
    const contacts = data.contacts || [];

    if (contacts.length === 0) {
      contactsList.innerHTML =
        '<p style="padding: 15px; text-align: center; color: #999;">No contacts found</p>';
      return;
    }

    contactsList.innerHTML = contacts
      .map(
        (contact) => `
            <div class="contact-item">
                <input type="checkbox" 
                       id="contact_${contact.id}" 
                       value="${contact.id}"
                       onchange="toggleMassContact(${contact.id}, '${
          contact.name
        }', '${contact.phone_number}')">
                <div class="contact-item-info">
                    <div class="contact-item-name">${
                      contact.name || "Sem nome"
                    }</div>
                    <div class="contact-item-phone">${
                      contact.phone_number
                    }</div>
                </div>
            </div>
        `
      )
      .join("");

    // Marcar contatos já selecionados
    selectedMassContacts.forEach((contactId) => {
      const checkbox = document.getElementById(`contact_${contactId}`);
      if (checkbox) checkbox.checked = true;
    });
  } catch (error) {
    console.error("Error fetching contacts:", error);
    contactsList.innerHTML =
      '<p style="padding: 15px; text-align: center; color: #dc3545;">Error fetching contacts</p>';
  }
}

// Função para alternar seleção de contato individual
function toggleMassContact(contactId, name, phone) {
  const index = selectedMassContacts.findIndex((c) => c.id === contactId);

  if (index > -1) {
    selectedMassContacts.splice(index, 1);
  } else {
    selectedMassContacts.push({ id: contactId, name, phone });
  }

  updateSelectedContactsInfo();
}

// Função para atualizar informações de contatos selecionados
function updateSelectedContactsInfo() {
  const infoDiv = document.getElementById("massSelectedContactsInfo");
  if (!infoDiv) return;

  if (selectedMassContacts.length === 0) {
    infoDiv.innerHTML = "";
  } else {
    infoDiv.innerHTML = `<p style="color: #28a745;">✅ ${selectedMassContacts.length} contato(s) selecionado(s)</p>`;
  }
}

// Função para enviar mensagem em massa
async function sendMassCampaign() {
  const messageText = document.getElementById("massMessageText").value.trim();
  const groupSelect = document.getElementById("massGroupSelect");
  const intervalSelect = document.getElementById("massSendInterval");
  const sendBtn = document.getElementById("sendMassCampaignBtn");

  // Validar mensagem
  if (!messageText && !massMediaFile) {
    showNotification("Type a message or attach media", "error");
    return;
  }

  // Determinar destinatários
  const activeTab = document.querySelector(".recipient-tab.active");
  let recipients = [];
  let recipientType = "group";

  if (activeTab && activeTab.dataset.tab === "individual") {
    recipientType = "individual";
    if (selectedMassContacts.length === 0) {
      showNotification("Select at least one contact", "error");
      return;
    }
    recipients = selectedMassContacts.map((c) => c.id);
  } else {
    if (!groupSelect || !groupSelect.value) {
      showNotification("Select a group", "error");
      return;
    }
    recipients = [groupSelect.value];
  }

  // Obter intervalo
  const interval = intervalSelect ? parseInt(intervalSelect.value) : 21;

  // Confirmar envio
  const confirmMsg =
    recipientType === "group"
      ? "Deseja iniciar o envio em massa para este grupo?"
      : `Deseja iniciar o envio em massa para ${recipients.length} contato(s)?`;

  if (!confirm(confirmMsg)) {
    return;
  }

  // Verificar agendamento
  const scheduleEnabled = document.getElementById("scheduleEnabled");
  const scheduleDateTime = document.getElementById("scheduleDateTime");
  let scheduledAt = null;

  if (
    scheduleEnabled &&
    scheduleEnabled.checked &&
    scheduleDateTime &&
    scheduleDateTime.value
  ) {
    scheduledAt = scheduleDateTime.value;
  }

  // Preparar dados
  const formData = new FormData();
  formData.append("message_text", messageText);

  if (recipientType === "group") {
    formData.append("group_id", recipients[0]);
  } else {
    formData.append("contact_ids", JSON.stringify(recipients));
  }

  formData.append("send_interval", interval);

  if (scheduledAt) {
    formData.append("scheduled_at", scheduledAt);
  }

  if (massMediaFile) {
    formData.append("mediaFile", massMediaFile);
    formData.append("message_type", massMediaType);
  }

  // Desabilitar botão
  const originalText = sendBtn.textContent;
  sendBtn.disabled = true;
  sendBtn.textContent = "⏳ Sending...";

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/mass-messaging/campaigns`,
      {
        method: "POST",
        body: formData,
        headers: {}, // Remover Content-Type para FormData
      }
    );

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.error || "Error sending message em massa");
    }

    const result = await response.json();

    if (scheduledAt) {
      showNotification(
        `Scheduled sending! ${
          result.contactCount || 0
        } messages will be sent on ${new Date(scheduledAt).toLocaleString(
          "pt-BR"
        )}.`,
        "success"
      );
    } else {
      showNotification(
        `Sending started! ${result.contactCount || 0} messages will be sent.`,
        "success"
      );
    }

    // Limpar formulário
    document.getElementById("massMessageText").value = "";
    removeMassMedia();
    selectedMassContacts = [];
    updateSelectedContactsInfo();
    updateMassPreview();

    // Limpar agendamento
    const scheduleCheckbox = document.getElementById("scheduleEnabled");
    const scheduleDateInput = document.getElementById("scheduleDateTime");
    if (scheduleCheckbox) scheduleCheckbox.checked = false;
    if (scheduleDateInput) scheduleDateInput.value = "";
    toggleSchedule();
  } catch (error) {
    console.error("Error sending message em massa:", error);
    showNotification(
      error.message || "Error sending message em massa",
      "error"
    );
  } finally {
    sendBtn.disabled = false;
    sendBtn.textContent = originalText;
  }
}

// Função para alternar agendamento
function toggleSchedule() {
  const checkbox = document.getElementById("scheduleEnabled");
  const inputs = document.getElementById("scheduleInputs");

  if (checkbox && inputs) {
    inputs.style.display = checkbox.checked ? "block" : "none";

    if (!checkbox.checked) {
      const dateInput = document.getElementById("scheduleDateTime");
      if (dateInput) dateInput.value = "";
    }
  }
}

// Inicializar preview ao carregar a página
document.addEventListener("DOMContentLoaded", function () {
  const massMessageText = document.getElementById("massMessageText");
  if (massMessageText) {
    updateMassPreview();
  }
});

console.log("Funções de envio em massa carregadas successfully");

// ===================================================================
//  SEÇÃO DE EDITAR PERFIL
// ===================================================================

async function loadProfile() {
  console.log("Loading página de perfil...");

  // Carregar configurações salvas do localStorage
  const systemName = localStorage.getItem("systemName") || "Misayan Admin";
  const systemNameInput = document.getElementById("systemName");
  if (systemNameInput) {
    systemNameInput.value = systemName;

    // Atualizar traduções
    if (typeof i18n !== "undefined") {
      i18n.updatePageTranslations();
    }
  }

  // Carregar cores salvas
  loadSavedColors();
}

function loadSavedColors() {
  // Cores do Menu
  const menuBackground = localStorage.getItem("menuBackground") || "#8B4513";
  const menuActive = localStorage.getItem("menuActive") || "#A0522D";
  const menuHover = localStorage.getItem("menuHover") || "#9B5A3C";

  // Cores dos Cards
  const cardBackground = localStorage.getItem("cardBackground") || "#FFFFFF";
  const cardBorder = localStorage.getItem("cardBorder") || "#EEEEEE";

  // Cores dos Textos
  const primaryColor = localStorage.getItem("primaryColor") || "#8B4513";
  const titleColor = localStorage.getItem("titleColor") || "#333333";

  // Aplicar aos inputs de cor
  setColorInput("menuBackground", menuBackground);
  setColorInput("menuActive", menuActive);
  setColorInput("menuHover", menuHover);
  setColorInput("cardBackground", cardBackground);
  setColorInput("cardBorder", cardBorder);
  setColorInput("primaryColor", primaryColor);
  setColorInput("titleColor", titleColor);
}

function setColorInput(id, color) {
  const colorInput = document.getElementById(id);
  const textInput = document.getElementById(id + "Text");

  if (colorInput) colorInput.value = color;
  if (textInput) textInput.value = color;
}

function updateSystemName() {
  const systemNameInput = document.getElementById("systemName");
  if (!systemNameInput) return;

  const systemName = systemNameInput.value.trim();
  if (!systemName) {
    showNotification("Please enter a name for the system", "warning");
    return;
  }

  localStorage.setItem("systemName", systemName);

  // Atualizar o título na sidebar
  const logoTitle = document.querySelector(".logo h2");
  if (logoTitle) {
    logoTitle.textContent = systemName;
  }

  // Atualizar o título da página
  document.title = systemName;

  showNotification("System name updated successfully!", "success");
}

function updatePreviewRealtime() {
  // Sincronizar inputs de cor com inputs de texto
  syncColorInputs();
}

function syncColorInputs() {
  const colorIds = [
    "menuBackground",
    "menuActive",
    "menuHover",
    "cardBackground",
    "cardBorder",
    "primaryColor",
    "titleColor",
  ];

  colorIds.forEach((id) => {
    const colorInput = document.getElementById(id);
    const textInput = document.getElementById(id + "Text");

    if (colorInput && textInput) {
      // Se o input de cor mudou, atualizar o texto
      if (document.activeElement === colorInput) {
        textInput.value = colorInput.value;
      }
      // Se o input de texto mudou, atualizar a cor
      else if (document.activeElement === textInput) {
        colorInput.value = textInput.value;
      }
    }
  });
}

function openPreviewModal(type) {
  showNotification("Preview function in development", "info");
}

function saveColorChanges() {
  const colors = {
    menuBackground:
      document.getElementById("menuBackground")?.value || "#8B4513",
    menuActive: document.getElementById("menuActive")?.value || "#A0522D",
    menuHover: document.getElementById("menuHover")?.value || "#9B5A3C",
    cardBackground:
      document.getElementById("cardBackground")?.value || "#FFFFFF",
    cardBorder: document.getElementById("cardBorder")?.value || "#EEEEEE",
    primaryColor: document.getElementById("primaryColor")?.value || "#8B4513",
    titleColor: document.getElementById("titleColor")?.value || "#333333",
  };

  // Save no localStorage
  Object.keys(colors).forEach((key) => {
    localStorage.setItem(key, colors[key]);
  });

  // Aplicar as cores ao sistema
  applyColorsToSystem(colors);

  showNotification("Cores salvas e aplicadas successfully!", "success");
}

function applyColorsToSystem(colors) {
  const root = document.documentElement;

  // Aplicar cores do menu
  root.style.setProperty("--menu-bg", colors.menuBackground);
  root.style.setProperty("--menu-active", colors.menuActive);
  root.style.setProperty("--menu-hover", colors.menuHover);

  // Aplicar cores dos cards
  root.style.setProperty("--card-bg", colors.cardBackground);
  root.style.setProperty("--card-border", colors.cardBorder);

  // Aplicar cores dos textos
  root.style.setProperty("--primary-color", colors.primaryColor);
  root.style.setProperty("--title-color", colors.titleColor);
}

function resetColors() {
  const defaultColors = {
    menuBackground: "#8B4513",
    menuActive: "#A0522D",
    menuHover: "#9B5A3C",
    cardBackground: "#FFFFFF",
    cardBorder: "#EEEEEE",
    primaryColor: "#8B4513",
    titleColor: "#333333",
  };

  // Remover do localStorage
  Object.keys(defaultColors).forEach((key) => {
    localStorage.removeItem(key);
  });

  // Aplicar cores padrão
  loadSavedColors();
  applyColorsToSystem(defaultColors);

  showNotification("Colors restored to default!", "success");
}

console.log("Funções de perfil carregadas successfully");

// ===================================================================
//  GERENCIAMENTO DE MENSAGENS DE BOAS-VINDAS
// ===================================================================

let welcomeMediaFile = null; // Apenas mensagem 3 tem mídia

// Atualizar preview em tempo real
function updateWelcomePreview() {
  const previewContainer = document.getElementById("previewMessages");
  if (!previewContainer) return;

  previewContainer.innerHTML = "";

  for (let i = 1; i <= 3; i++) {
    const textarea = document.getElementById(`welcomeMsg${i}`);
    const text = textarea ? textarea.value.trim() : "";
    const mediaFile = i === 3 ? welcomeMediaFile : null;

    if (text || mediaFile) {
      const messageDiv = document.createElement("div");
      messageDiv.style.cssText =
        "background: white; padding: 8px 12px; border-radius: 8px; max-width: 80%; align-self: flex-start; box-shadow: 0 1px 2px rgba(0,0,0,0.1); margin-bottom: 8px; animation: slideIn 0.3s ease;";

      if (mediaFile) {
        const mediaPreview = document.createElement("div");
        mediaPreview.style.cssText =
          "margin-bottom: 5px; border-radius: 4px; overflow: hidden;";

        if (mediaFile.type.startsWith("image/")) {
          const img = document.createElement("img");
          img.src = URL.createObjectURL(mediaFile);
          img.style.cssText = "max-width: 100%; height: auto; display: block;";
          mediaPreview.appendChild(img);
        } else if (mediaFile.type.startsWith("video/")) {
          const video = document.createElement("video");
          video.src = URL.createObjectURL(mediaFile);
          video.style.cssText =
            "max-width: 100%; height: auto; display: block;";
          video.controls = true;
          mediaPreview.appendChild(video);
        } else if (mediaFile.type.startsWith("audio/")) {
          const audio = document.createElement("audio");
          audio.src = URL.createObjectURL(mediaFile);
          audio.controls = true;
          audio.style.cssText = "width: 100%;";
          mediaPreview.appendChild(audio);
        }

        messageDiv.appendChild(mediaPreview);
      }

      if (text) {
        const textDiv = document.createElement("div");
        textDiv.textContent = text;
        textDiv.style.cssText =
          "font-size: 14px; color: #333; word-wrap: break-word;";
        messageDiv.appendChild(textDiv);
      }

      const timeDiv = document.createElement("div");
      timeDiv.textContent = new Date().toLocaleTimeString("pt-BR", {
        hour: "2-digit",
        minute: "2-digit",
      });
      timeDiv.style.cssText =
        "font-size: 10px; color: #999; text-align: right; margin-top: 4px;";
      messageDiv.appendChild(timeDiv);

      previewContainer.appendChild(messageDiv);
    }
  }

  if (previewContainer.children.length === 0) {
    previewContainer.innerHTML =
      '<div style="text-align: center; color: #666; font-size: 12px; padding: 10px;">Messages will appear here as you type</div>';
  }
}

// Inserir placeholder diretamente (sem modal)
function insertPlaceholderDirect(lineNumber, placeholder) {
  const textarea = document.getElementById(`welcomeMsg${lineNumber}`);
  if (!textarea) return;

  const cursorPos = textarea.selectionStart;
  const textBefore = textarea.value.substring(0, cursorPos);
  const textAfter = textarea.value.substring(cursorPos);
  textarea.value = textBefore + placeholder + textAfter;
  textarea.focus();
  textarea.setSelectionRange(
    cursorPos + placeholder.length,
    cursorPos + placeholder.length
  );
  updateWelcomePreview();
}

// Copiar placeholder
function copyPlaceholder(placeholder) {
  navigator.clipboard.writeText(placeholder);
  showNotification("Placeholder copiado: " + placeholder, "success");
}

// Save mensagens de boas-vindas
async function saveWelcomeMessages() {
  try {
    const messages = [];

    for (let i = 1; i <= 3; i++) {
      const textarea = document.getElementById(`welcomeMsg${i}`);
      const text = textarea ? textarea.value.trim() : "";

      if (text) {
        messages.push({
          line: i,
          text: text,
        });
      }
    }

    const response = await authenticatedFetch(
      `${API_BASE}/api/faqs/welcome-messages`,
      {
        method: "POST",
        body: JSON.stringify({ messages }),
      }
    );

    if (response.ok) {
      showNotification(
        "Welcome messages saved successfully!",
        "success"
      );
    } else {
      throw new Error("Error salvar mensagens");
    }
  } catch (error) {
    console.error("Error saving welcome messages:", error);
    showNotification("Error saving welcome messages", "error");
  }
}

// Carregar mensagens de boas-vindas existentes
async function loadWelcomeMessages() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/faqs/welcome-messages`
    );
    const messages = await response.json();

    messages.forEach((msg) => {
      const textarea = document.getElementById(
        `welcomeMsg${msg.order_position}`
      );
      if (textarea) {
        textarea.value = msg.message_text || "";
      }

      if (msg.media_url) {
        const previewDiv = document.getElementById(
          `welcomeMediaPreview${msg.order_position}`
        );
        if (previewDiv) {
          previewDiv.innerHTML = `<small style="color: #666;">📎 Mídia anexada</small>`;
        }
      }
    });

    updateWelcomePreview();
  } catch (error) {
    console.error("Error loading messages de boas-vindas:", error);
  }
}

// Add listeners para atualização em tempo real
document.addEventListener("DOMContentLoaded", () => {
  for (let i = 1; i <= 3; i++) {
    const textarea = document.getElementById(`welcomeMsg${i}`);
    if (textarea) {
      textarea.addEventListener("input", updateWelcomePreview);
    }
  }
});

console.log("Funções de mensagens de boas-vindas carregadas successfully");

// ===================================================================
//  FUNÇÕES DO MODAL DE SELEÇÃO DE HORÁRIO
// ===================================================================

let currentTimeInputField = null;

function openTimePickerModal(fieldId) {
  currentTimeInputField = fieldId;
  const currentValue = document.getElementById(fieldId)?.value || "08:00-18:00";
  const [start, end] = currentValue.split("-");

  // Parse horário de início
  const [startHour, startMinute] = start.trim().split(":");
  document.getElementById("startHour").value = startHour.padStart(2, "0");
  document.getElementById("startMinute").value = startMinute.padStart(2, "0");

  // Parse horário de término
  const [endHour, endMinute] = end.trim().split(":");
  document.getElementById("endHour").value = endHour.padStart(2, "0");
  document.getElementById("endMinute").value = endMinute.padStart(2, "0");

  // Mostrar modal
  document.getElementById("timePickerModal").style.display = "block";

  // Adicionar listeners para validação em tempo real
  setupTimeInputValidation();
}

function closeTimePickerModal() {
  document.getElementById("timePickerModal").style.display = "none";
  currentTimeInputField = null;
}

function confirmTimePicker() {
  if (!currentTimeInputField) return;

  const startHour = document.getElementById("startHour").value.padStart(2, "0");
  const startMinute = document
    .getElementById("startMinute")
    .value.padStart(2, "0");
  const endHour = document.getElementById("endHour").value.padStart(2, "0");
  const endMinute = document.getElementById("endMinute").value.padStart(2, "0");

  // Validar valores
  if (
    !isValidTimeValue(startHour, 0, 23) ||
    !isValidTimeValue(startMinute, 0, 59) ||
    !isValidTimeValue(endHour, 0, 23) ||
    !isValidTimeValue(endMinute, 0, 59)
  ) {
    showNotification(i18n.t("notification.invalid_time_values"), "error");
    return;
  }

  // Formatar e atualizar o campo
  const timeRange = `${startHour}:${startMinute}-${endHour}:${endMinute}`;
  document.getElementById(currentTimeInputField).value = timeRange;

  closeTimePickerModal();
}

function isValidTimeValue(value, min, max) {
  const num = parseInt(value, 10);
  return !isNaN(num) && num >= min && num <= max;
}

function setupTimeInputValidation() {
  const inputs = ["startHour", "startMinute", "endHour", "endMinute"];

  inputs.forEach((inputId) => {
    const input = document.getElementById(inputId);
    if (!input) return;

    // Remover listeners anteriores
    input.onblur = null;
    input.oninput = null;

    input.addEventListener("input", function () {
      let value = parseInt(this.value, 10);
      const max = inputId.includes("Hour") ? 23 : 59;

      if (isNaN(value)) {
        this.value = "00";
      } else if (value > max) {
        this.value = max.toString().padStart(2, "0");
      } else if (value < 0) {
        this.value = "00";
      }
    });

    input.addEventListener("blur", function () {
      if (this.value === "" || isNaN(parseInt(this.value, 10))) {
        this.value = "00";
      } else {
        this.value = parseInt(this.value, 10).toString().padStart(2, "0");
      }
    });
  });
}

// Fechar modal ao clicar fora
window.addEventListener("click", function (event) {
  const modal = document.getElementById("timePickerModal");
  if (event.target === modal) {
    closeTimePickerModal();
  }
});

console.log("Time selection modal functions loaded successfully");

// ===================================================================
//  FUNÇÕES DO MODAL DE ADICIONAR STRING DE TRADUÇÃO
// ===================================================================

function addNewString() {
  const modal = document.getElementById("addStringModal");
  if (!modal) {
    console.error("Add string modal not found");
    return;
  }

  // Limpar campos
  document.getElementById("newStringCategory").value = "general";
  document.getElementById("newStringKey").value = "";
  document.getElementById("newStringValue").value = "";
  updateFullKeyPreview();

  // Mostrar modal
  modal.style.display = "block";

  // Adicionar listener para atualizar preview da chave
  const categorySelect = document.getElementById("newStringCategory");
  const keyInput = document.getElementById("newStringKey");

  if (categorySelect && keyInput) {
    categorySelect.removeEventListener("change", updateFullKeyPreview);
    keyInput.removeEventListener("input", updateFullKeyPreview);

    categorySelect.addEventListener("change", updateFullKeyPreview);
    keyInput.addEventListener("input", updateFullKeyPreview);
  }
}

function closeAddStringModal() {
  const modal = document.getElementById("addStringModal");
  if (modal) {
    modal.style.display = "none";
  }
}

function updateFullKeyPreview() {
  const category =
    document.getElementById("newStringCategory")?.value || "general";
  const key = document.getElementById("newStringKey")?.value || "chave";
  const preview = document.getElementById("fullKeyPreview");

  if (preview) {
    preview.textContent = `${category}.${key}`;
  }
}

async function saveNewString() {
  const category = document.getElementById("newStringCategory")?.value;
  const key = document.getElementById("newStringKey")?.value?.trim();
  const value = document.getElementById("newStringValue")?.value?.trim();

  // Validações
  if (!category) {
    showNotification(i18n.t("notification.select_category"), "error");
    return;
  }

  if (!key) {
    showNotification(i18n.t("notification.key_required"), "error");
    return;
  }

  if (!value) {
    showNotification(i18n.t("notification.value_required"), "error");
    return;
  }

  // Validar formato da chave (apenas letras, números e underscore)
  if (!/^[a-z0-9_]+$/i.test(key)) {
    showNotification(i18n.t("notification.invalid_key_format"), "error");
    return;
  }

  // Montar chave completa
  const fullKey = `${category}.${key}`;

  try {
    // Adicionar a string em todos os idiomas
    const response = await authenticatedFetch(
      `${API_BASE}/api/translations/strings/en`,
      {
        method: "POST",
        body: JSON.stringify({
          translation_key: fullKey,
          translation_value: value,
        }),
      }
    );

    if (response.ok) {
      showNotification(i18n.t("notification.string_added_success"), "success");
      closeAddStringModal();

      // Recarregar a página de traduções
      if (typeof loadTranslations === "function") {
        await loadTranslations();
      }
    } else {
      const error = await response.json();
      showNotification(
        error.error || i18n.t("notification.string_add_error"),
        "error"
      );
    }
  } catch (error) {
    console.error("Error adding string:", error);
    showNotification(i18n.t("notification.string_add_error"), "error");
  }
}

// Fechar modal ao clicar fora
window.addEventListener("click", function (event) {
  const addStringModal = document.getElementById("addStringModal");
  if (event.target === addStringModal) {
    closeAddStringModal();
  }
});

console.log(
  "Add translation string modal functions loaded successfully"
);
