/**
 * LICENSE VALIDATION MIDDLEWARE
 * 
 * This file should be placed in the MISAYAN system that will be sold.
 * It validates the license remotely on the server https://licenca.misayan.cloud
 * 
 * Installation:
 * 1. Copy this file to the /middleware/ folder of your project
 * 2. Configure the LICENSE_KEY variable in the client's .env file
 * 3. Use the middleware on routes that need validation
 */

const axios = require('axios');
const fs = require('fs');
const path = require('path');

// Validation server URL
const LICENSE_SERVER = 'https://licenca.misayan.cloud';

// File to cache the last validation
const CACHE_FILE = path.join(__dirname, '../.license_cache');

// Validation interval (24 hours)
const VALIDATION_INTERVAL = 24 * 60 * 60 * 1000;

/**
 * Validates the license remotely
 */
async function validateLicenseRemote(licenseKey) {
  try {
    const response = await axios.post(
      `${LICENSE_SERVER}/api/license/verify`,
      { license_key: licenseKey },
      {
        timeout: 10000,
        headers: { 'Content-Type': 'application/json' }
      }
    );

    return response.data;
  } catch (error) {
    console.error('Error validating license remotely:', error.message);
    
    // In case of connection error, use cache if available
    if (error.code === 'ECONNREFUSED' || error.code === 'ETIMEDOUT') {
      return { success: false, error: 'connection_error', useCache: true };
    }
    
    return { success: false, valid: false, error: 'Error validating license' };
  }
}

/**
 * Saves validation result to cache
 */
function saveCache(data) {
  try {
    const cacheData = {
      ...data,
      cached_at: new Date().toISOString()
    };
    fs.writeFileSync(CACHE_FILE, JSON.stringify(cacheData), 'utf8');
  } catch (error) {
    console.error('Error saving license cache:', error);
  }
}

/**
 * Reads cache from last validation
 */
function readCache() {
  try {
    if (fs.existsSync(CACHE_FILE)) {
      const data = fs.readFileSync(CACHE_FILE, 'utf8');
      return JSON.parse(data);
    }
  } catch (error) {
    console.error('Error reading license cache:', error);
  }
  return null;
}

/**
 * Checks if the cache is still valid
 */
function isCacheValid(cache) {
  if (!cache || !cache.cached_at) return false;
  
  const cachedAt = new Date(cache.cached_at);
  const now = new Date();
  const diff = now - cachedAt;
  
  return diff < VALIDATION_INTERVAL;
}

/**
 * License validation middleware
 * 
 * Usage:
 * const { validateLicense } = require('./middleware/licenseValidator');
 * app.use('/api/admin', validateLicense, require('./routes/admin'));
 */
async function validateLicense(req, res, next) {
  try {
    const licenseKey = process.env.LICENSE_KEY;
    
    if (!licenseKey) {
      return res.status(403).json({
        error: 'License not configured. Configure the LICENSE_KEY variable in the .env file'
      });
    }
    
    // Check cache first
    const cache = readCache();
    
    if (cache && isCacheValid(cache)) {
      if (cache.valid) {
        // Valid cache, allow access
        req.license = cache;
        return next();
      } else {
        // Cache indicates invalid license
        return res.status(403).json({
          error: cache.error || 'Invalid license',
          revoked: cache.revoked,
          expired: cache.expired
        });
      }
    }
    
    // Cache expired or doesn't exist, validate remotely
    const validation = await validateLicenseRemote(licenseKey);
    
    // If there's a connection error and cache exists, use cache
    if (validation.useCache && cache) {
      console.warn('Using license cache due to connection error');
      if (cache.valid) {
        req.license = cache;
        return next();
      }
    }
    
    // Save result to cache
    if (validation.success && validation.valid) {
      saveCache(validation);
      req.license = validation;
      return next();
    }
    
    // Invalid license
    if (validation.valid === false) {
      saveCache(validation);
    }
    
    return res.status(403).json({
      error: validation.error || 'Invalid license',
      revoked: validation.revoked,
      expired: validation.expired
    });
    
  } catch (error) {
    console.error('Error in license validation middleware:', error);
    
    // In case of error, check if valid cache exists
    const cache = readCache();
    if (cache && cache.valid) {
      console.warn('Using license cache due to middleware error');
      req.license = cache;
      return next();
    }
    
    return res.status(500).json({
      error: 'Error validating license. Please contact support.'
    });
  }
}

/**
 * Function to validate license manually (for use in scripts)
 */
async function checkLicense(licenseKey) {
  const validation = await validateLicenseRemote(licenseKey);
  
  if (validation.success && validation.valid) {
    console.log('✅ Valid license!');
    console.log('Buyer:', validation.buyer);
    console.log('Domain:', validation.domain);
    console.log('Expires at:', new Date(validation.expires_at).toLocaleString('en-US'));
    return true;
  } else {
    console.log('❌ Invalid license!');
    console.log('Error:', validation.error);
    if (validation.revoked) console.log('Status: REVOKED');
    if (validation.expired) console.log('Status: EXPIRED');
    return false;
  }
}

module.exports = {
  validateLicense,
  checkLicense,
  validateLicenseRemote
};

