/**
 * Centralized Error Handler Middleware
 * 
 * Handles all errors in a consistent way across the application
 * Provides proper error responses and logging
 */

/**
 * Custom Error Classes
 */
class AppError extends Error {
  constructor(message, statusCode = 500, isOperational = true) {
    super(message);
    this.statusCode = statusCode;
    this.isOperational = isOperational;
    this.status = `${statusCode}`.startsWith('4') ? 'fail' : 'error';
    
    Error.captureStackTrace(this, this.constructor);
  }
}

class ValidationError extends AppError {
  constructor(message, details = []) {
    super(message, 400);
    this.details = details;
    this.name = 'ValidationError';
  }
}

class AuthenticationError extends AppError {
  constructor(message = 'Authentication failed') {
    super(message, 401);
    this.name = 'AuthenticationError';
  }
}

class AuthorizationError extends AppError {
  constructor(message = 'Access denied') {
    super(message, 403);
    this.name = 'AuthorizationError';
  }
}

class NotFoundError extends AppError {
  constructor(resource = 'Resource') {
    super(`${resource} not found`, 404);
    this.name = 'NotFoundError';
  }
}

class DatabaseError extends AppError {
  constructor(message = 'Database operation failed') {
    super(message, 500);
    this.name = 'DatabaseError';
  }
}

/**
 * Error Handler Middleware
 * Must be the last middleware in the chain
 */
function errorHandler(err, req, res, next) {
  // Set default error
  err.statusCode = err.statusCode || 500;
  err.status = err.status || 'error';

  // Ignore 404 errors for common browser requests
  const ignored404Paths = [
    '/.well-known',
    '/favicon.ico',
    '/robots.txt',
    '/sitemap.xml'
  ];
  
  const isIgnored404 = err.statusCode === 404 && 
    ignored404Paths.some(path => req.path.startsWith(path));
  
  if (isIgnored404) {
    return res.status(404).end();
  }

  // Don't log sensitive authentication errors with full details
  const isAuthError = err.statusCode === 401 || err.statusCode === 403;
  const isLoginPath = req.path === '/api/auth/login';
  
  if (isAuthError && isLoginPath) {
    // For login errors, log minimal info without sensitive data
    if (process.env.NODE_ENV !== 'production') {
      console.error('Authentication Error:', {
        message: err.message,
        statusCode: err.statusCode,
        path: req.path,
        username: req.body?.username // Only log username, not password
      });
    }
    // In production, don't log authentication errors at all
  } else {
    // Log error details for other errors
    if (process.env.NODE_ENV === 'production') {
      // In production, log minimal info
      console.error('Error:', {
        message: err.message,
        statusCode: err.statusCode,
        path: req.path,
        method: req.method
      });
    } else {
      // In development, log full stack but sanitize sensitive data
      const sanitizedBody = { ...req.body };
      if (sanitizedBody.password) {
        sanitizedBody.password = '***REDACTED***';
      }
      if (sanitizedBody.token) {
        sanitizedBody.token = '***REDACTED***';
      }
      
      console.error('Error Details:', {
        message: err.message,
        statusCode: err.statusCode,
        stack: err.stack,
        path: req.path,
        method: req.method,
        body: sanitizedBody,
        query: req.query
      });
    }
  }

  // Handle specific error types
  if (err.name === 'ValidationError') {
    return res.status(400).json({
      error: 'Validation Error',
      message: err.message,
      details: err.details || []
    });
  }

  if (err.name === 'JsonWebTokenError') {
    return res.status(401).json({
      error: 'Invalid Token',
      message: 'The authentication token is invalid'
    });
  }

  if (err.name === 'TokenExpiredError') {
    return res.status(401).json({
      error: 'Token Expired',
      message: 'The authentication token has expired'
    });
  }

  if (err.name === 'MulterError') {
    if (err.code === 'LIMIT_FILE_SIZE') {
      return res.status(400).json({
        error: 'File Too Large',
        message: 'The uploaded file exceeds the maximum allowed size'
      });
    }
    return res.status(400).json({
      error: 'Upload Error',
      message: err.message
    });
  }

  // Handle database errors
  if (err.code === 'ER_DUP_ENTRY') {
    return res.status(409).json({
      error: 'Duplicate Entry',
      message: 'A record with this information already exists'
    });
  }

  if (err.code === 'ER_NO_REFERENCED_ROW_2') {
    return res.status(400).json({
      error: 'Invalid Reference',
      message: 'Referenced record does not exist'
    });
  }

  // Handle MySQL connection errors
  if (err.code === 'ECONNREFUSED' || err.code === 'PROTOCOL_CONNECTION_LOST') {
    return res.status(503).json({
      error: 'Database Unavailable',
      message: 'Unable to connect to database. Please try again later.'
    });
  }

  // Operational errors: send message to client
  if (err.isOperational) {
    return res.status(err.statusCode).json({
      error: err.status,
      message: err.message,
      ...(err.details && { details: err.details })
    });
  }

  // Programming or unknown errors: don't leak error details
  // In production, send generic message
  if (process.env.NODE_ENV === 'production') {
    return res.status(500).json({
      error: 'Internal Server Error',
      message: 'Something went wrong. Please try again later.'
    });
  }

  // In development, send full error details
  return res.status(err.statusCode).json({
    error: err.status,
    message: err.message,
    stack: err.stack,
    ...(err.details && { details: err.details })
  });
}

/**
 * Async Handler Wrapper
 * Wraps async route handlers to catch errors automatically
 */
function asyncHandler(fn) {
  return (req, res, next) => {
    Promise.resolve(fn(req, res, next)).catch(next);
  };
}

/**
 * 404 Handler
 * Must be placed before error handler
 */
function notFoundHandler(req, res, next) {
  // Ignore common browser/development tool requests (don't log as errors)
  const ignoredPaths = [
    '/.well-known',
    '/favicon.ico',
    '/robots.txt',
    '/sitemap.xml'
  ];
  
  const shouldIgnore = ignoredPaths.some(path => req.path.startsWith(path));
  
  if (shouldIgnore) {
    return res.status(404).end();
  }
  
  const err = new NotFoundError(`Route ${req.method} ${req.path}`);
  next(err);
}

module.exports = {
  AppError,
  ValidationError,
  AuthenticationError,
  AuthorizationError,
  NotFoundError,
  DatabaseError,
  errorHandler,
  asyncHandler,
  notFoundHandler
};

