const mysql = require('mysql2/promise');
const bcrypt = require('bcryptjs');
require('dotenv').config();

const dbConfig = {
  host: process.env.DB_HOST,
  port: process.env.DB_PORT,
  user: process.env.DB_USER,
  password: process.env.DB_PASS,
  database: process.env.DB_NAME,
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0
};

const pool = mysql.createPool(dbConfig);

// Function to initialize the database
async function initDatabase() {
  let connection;
  try {
    connection = await pool.getConnection();
    console.log('📄 Initializing database...');
    
    // Create admins table
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS admins (
        id INT AUTO_INCREMENT PRIMARY KEY,
        username VARCHAR(50) UNIQUE NOT NULL,
        password VARCHAR(255) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_username (username)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Admins table created/verified');

    // Create tables if they don't exist
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS users (
        id INT AUTO_INCREMENT PRIMARY KEY,
        username VARCHAR(50) UNIQUE NOT NULL,
        password VARCHAR(255) NOT NULL,
        role ENUM('admin', 'user') DEFAULT 'user',
        store VARCHAR(100),
        department VARCHAR(100),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_username (username),
        INDEX idx_role (role),
        INDEX idx_store (store),
        INDEX idx_department (department)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Users table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS stores (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL UNIQUE,
        address TEXT,
        hours TEXT,
        promotions TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_name (name)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Stores table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS departments (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL UNIQUE,
        description TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_name (name)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Departments table created/verified');

    try {
      await connection.execute(
        `ALTER TABLE stores ADD UNIQUE KEY \`name\` (\`name\`)`
      );
      console.log('🔧 UNIQUE constraint applied/verified on stores table.');
    } catch (error) {
      if (error.code === 'ER_DUP_KEYNAME') {
        console.log('ℹ️ UNIQUE constraint on stores table is already configured.');
      } else {
        if (error.code === 'ER_DUP_ENTRY') {
            console.error('❌ ERROR: Could not apply UNIQUE rule because there are already stores with duplicate names in your database.');
            console.error('   -> Please remove duplicate stores manually and restart the application.');
        }
        throw error;
      }
    }

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS conversations (
        id INT AUTO_INCREMENT PRIMARY KEY,
        phone_number VARCHAR(20) NOT NULL,
        contact_name VARCHAR(100),
        last_message TEXT,
        last_message_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        status ENUM('waiting', 'attended', 'closed') DEFAULT 'waiting',
        assigned_user_id INT,
        assigned_store VARCHAR(100),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (assigned_user_id) REFERENCES users(id) ON DELETE SET NULL,
        INDEX idx_phone (phone_number),
        INDEX idx_status (status),
        INDEX idx_last_message_time (last_message_time)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Conversations table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS messages (
        id INT AUTO_INCREMENT PRIMARY KEY,
        conversation_id INT,
        phone_number VARCHAR(20) NOT NULL,
        message_text TEXT,
        message_type ENUM('text', 'image', 'audio', 'video', 'document', 'location') DEFAULT 'text',
        media_url VARCHAR(500),
        is_from_bot BOOLEAN DEFAULT FALSE,
        sender_user_id INT,
        timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (conversation_id) REFERENCES conversations(id) ON DELETE CASCADE,
        FOREIGN KEY (sender_user_id) REFERENCES users(id) ON DELETE SET NULL,
        INDEX idx_phone (phone_number),
        INDEX idx_timestamp (timestamp),
        INDEX idx_is_from_bot (is_from_bot)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Messages table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS faqs (
        id INT AUTO_INCREMENT PRIMARY KEY,
        question VARCHAR(500) NOT NULL,
        answer TEXT NOT NULL,
        emoji VARCHAR(10),
        placeholder_key VARCHAR(100),
        active BOOLEAN DEFAULT TRUE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_active (active),
        INDEX idx_question (question(100)),
        UNIQUE KEY unique_question (question(255))
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ FAQs table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS bot_settings (
        id INT AUTO_INCREMENT PRIMARY KEY,
        setting_key VARCHAR(100) UNIQUE NOT NULL,
        setting_value TEXT,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_setting_key (setting_key)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Bot_settings table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS mass_messages (
        id INT AUTO_INCREMENT PRIMARY KEY,
        message_text TEXT NOT NULL,
        media_url VARCHAR(500),
        message_type ENUM('text', 'image', 'audio', 'video') DEFAULT 'text',
        total_contacts INT DEFAULT 0,
        sent_count INT DEFAULT 0,
        status ENUM('pending', 'sending', 'completed', 'failed') DEFAULT 'pending',
        created_by INT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
        INDEX idx_status (status),
        INDEX idx_created_at (created_at)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Mass_messages table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS contact_groups (
        id INT AUTO_INCREMENT PRIMARY KEY,
        group_name VARCHAR(100) NOT NULL,
        description TEXT,
        created_by INT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
        INDEX idx_name (group_name),
        INDEX idx_created_by (created_by),
        UNIQUE KEY unique_group_name (group_name)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Contact_groups table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS contacts (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        phone VARCHAR(20) NOT NULL,
        email VARCHAR(100),
        group_id INT,
        tags VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (group_id) REFERENCES contact_groups(id) ON DELETE SET NULL,
        INDEX idx_phone (phone),
        INDEX idx_name (name),
        INDEX idx_group (group_id),
        UNIQUE KEY unique_phone (phone)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Contacts table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS push_subscriptions (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        subscription TEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        INDEX idx_user_id (user_id)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Push_subscriptions table created/verified');

    // New tables for improved FAQ
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS welcome_messages (
        id INT AUTO_INCREMENT PRIMARY KEY,
        message_text TEXT NOT NULL,
        order_position INT NOT NULL DEFAULT 1,
        active BOOLEAN DEFAULT TRUE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_order (order_position),
        INDEX idx_active (active)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Welcome_messages table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS welcome_sent (
        id INT AUTO_INCREMENT PRIMARY KEY,
        phone_number VARCHAR(20) NOT NULL,
        sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_phone_date (phone_number, sent_at)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Welcome_sent table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS message_placeholders (
        id INT AUTO_INCREMENT PRIMARY KEY,
        placeholder_key VARCHAR(100) UNIQUE NOT NULL,
        placeholder_value TEXT NOT NULL,
        description VARCHAR(255),
        active BOOLEAN DEFAULT TRUE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_key (placeholder_key),
        INDEX idx_active (active)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Message_placeholders table created/verified');



    await connection.execute(`
      CREATE TABLE IF NOT EXISTS mass_message_logs (
        id INT AUTO_INCREMENT PRIMARY KEY,
        mass_message_id INT,
        contact_phone VARCHAR(20),
        contact_name VARCHAR(100),
        status ENUM('pending', 'sent', 'failed', 'delivered') DEFAULT 'pending',
        error_message TEXT,
        sent_at TIMESTAMP NULL,
        delivered_at TIMESTAMP NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (mass_message_id) REFERENCES mass_messages(id) ON DELETE CASCADE,
        INDEX idx_mass_message (mass_message_id),
        INDEX idx_status (status),
        INDEX idx_sent_at (sent_at)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Mass_message_logs table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS payment_methods (
        id INT AUTO_INCREMENT PRIMARY KEY,
        method_name ENUM('paypal', 'pagseguro') NOT NULL,
        api_key VARCHAR(255),
        api_secret VARCHAR(255),
        sandbox_mode BOOLEAN DEFAULT TRUE,
        active BOOLEAN DEFAULT FALSE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_method (method_name),
        INDEX idx_active (active)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Payment_methods table created/verified');

    // Fix column size to support encrypted data
    try {
      await connection.execute(`
        ALTER TABLE payment_methods 
        MODIFY COLUMN api_key TEXT,
        MODIFY COLUMN api_secret TEXT
      `);
      console.log('✅ api_key and api_secret columns adjusted to TEXT');
    } catch (error) {
      if (error.code === 'ER_DUP_FIELDNAME') {
        console.log('ℹ️ api_key and api_secret columns are already in the correct format');
      } else {
        console.log('ℹ️ Payment_methods columns adjustment:', error.message);
      }
    }

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS payment_links (
        id INT AUTO_INCREMENT PRIMARY KEY,
        payment_method ENUM('paypal', 'pagseguro') NOT NULL,
        amount DECIMAL(10,2) NOT NULL,
        description VARCHAR(255),
        customer_phone VARCHAR(20),
        customer_name VARCHAR(100),
        payment_url VARCHAR(500),
        payment_id VARCHAR(100),
        status ENUM('pending', 'paid', 'cancelled', 'expired') DEFAULT 'pending',
        created_by INT,
        expires_at TIMESTAMP NULL,
        paid_at TIMESTAMP NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
        INDEX idx_status (status),
        INDEX idx_payment_id (payment_id),
        INDEX idx_customer_phone (customer_phone)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Payment_links table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS translations (
        id INT AUTO_INCREMENT PRIMARY KEY,
        language_code VARCHAR(5) NOT NULL,
        language_name VARCHAR(50) NOT NULL,
        translation_key VARCHAR(255) NOT NULL,
        translation_value TEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        UNIQUE KEY unique_translation (language_code, translation_key),
        INDEX idx_language (language_code),
        INDEX idx_key (translation_key)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Translations table created/verified');

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS ai_settings (
        id INT AUTO_INCREMENT PRIMARY KEY,
        provider ENUM('deepseek', 'gpt') NOT NULL,
        api_key VARCHAR(255) NOT NULL,
        model_name VARCHAR(100),
        persona_name VARCHAR(100) DEFAULT 'Assistente',
        persona_behavior TEXT,
        working_hours VARCHAR(255) DEFAULT '08:00-18:00',
        working_days VARCHAR(50) DEFAULT 'monday,tuesday,wednesday,thursday,friday,saturday',
        active BOOLEAN DEFAULT FALSE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_provider (provider),
        INDEX idx_active (active)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
    console.log('✅ Ai_settings table created/verified');


    await connection.execute(`
  CREATE TABLE IF NOT EXISTS ai_configurations (
    id INT AUTO_INCREMENT PRIMARY KEY,
    persona_name VARCHAR(100) NOT NULL COMMENT 'Persona/assistant name',
    persona_description TEXT COMMENT 'Persona description',
    provider ENUM('openai', 'deepseek') NOT NULL DEFAULT 'deepseek' COMMENT 'AI provider',
    model_name VARCHAR(50) NOT NULL COMMENT 'Model name (e.g.: deepseek-chat)',
    api_key TEXT NOT NULL COMMENT 'Encrypted API key',
    system_prompt TEXT NOT NULL COMMENT 'System prompt/instructions',
    temperature DECIMAL(3,2) DEFAULT 0.70 COMMENT 'Temperature (0.0 to 1.0)',
    max_tokens INT DEFAULT 1000 COMMENT 'Maximum tokens in response',
    business_hours_start TIME DEFAULT '08:00:00' COMMENT 'Start time',
    business_hours_end TIME DEFAULT '18:00:00' COMMENT 'End time',
    business_days VARCHAR(100) DEFAULT 'monday,tuesday,wednesday,thursday,friday' COMMENT 'Business days',
    out_of_hours_message TEXT COMMENT 'Out of hours message',
    auto_response_enabled BOOLEAN DEFAULT TRUE COMMENT 'Auto response enabled',
    response_delay INT DEFAULT 0 COMMENT 'Response delay (seconds)',
    active BOOLEAN DEFAULT TRUE COMMENT 'Configuration active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_provider (provider),
    INDEX idx_active (active)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
`);
console.log('✅ Ai_configurations table created/verified');

// Create AI usage logs table
await connection.execute(`
  CREATE TABLE IF NOT EXISTS ai_usage_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    phone_number VARCHAR(20) NOT NULL,
    user_message TEXT NOT NULL,
    ai_response TEXT,
    response_time DECIMAL(10,2) COMMENT 'Response time in seconds',
    success BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_phone (phone_number),
    INDEX idx_created_at (created_at),
    INDEX idx_success (success)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
`);
console.log('✅ Ai_usage_logs table created/verified');

// Insert default DeepSeek configuration (if it doesn't exist)
const [existingConfigs] = await connection.execute('SELECT id FROM ai_configurations LIMIT 1');
if (existingConfigs.length === 0) {
  console.log('📝 Creating default DeepSeek configuration...');
  console.log('⚠️  IMPORTANT: Configure your DeepSeek API Key in the admin panel!');
  
  await connection.execute(`
    INSERT INTO ai_configurations (
      persona_name, 
      provider, 
      model_name, 
      api_key, 
      system_prompt,
      temperature,
      max_tokens,
      business_hours_start,
      business_hours_end,
      business_days,
      out_of_hours_message,
      active
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
  `, [
    'Virtual Assistant',
    'deepseek',
    'deepseek-chat',
    'YOUR_API_KEY_HERE', // You will need to change this in the admin panel
    'You are a helpful and friendly virtual assistant. Respond clearly, objectively and politely. Use emojis when appropriate to make the conversation lighter and more pleasant.',
    0.70,
    1000,
    '08:00:00',
    '18:00:00',
    'monday,tuesday,wednesday,thursday,friday,saturday',
    'Hello! Our virtual assistant is available Monday through Saturday, from 8am to 6pm. Leave your message and we will respond soon! 😊',
    true
  ]);
  
  console.log('✅ Default DeepSeek configuration created!');
  console.log('🔑 Don\'t forget to configure your API Key in the panel!');
} else {
  console.log('✅ AI configuration already exists');
}
    
    // Insert initial data
    console.log('📄 Inserting initial data...');
    
    // Insert admin into admins table
    const [adminExists] = await connection.execute('SELECT id FROM admins WHERE username = ?', ['admin']);
    if (adminExists.length === 0) {
      const hashedPassword = await bcrypt.hash('password', 10);
      await connection.execute(
        `INSERT INTO admins (username, password) VALUES (?, ?)`,
        ['admin', hashedPassword]
      );
      console.log('✅ Admin created in admins table (username: admin, password: password)');
    } else {
      console.log('✅ Admin already exists in admins table');
    }

    // Insert default admin user into users table
    const [userAdminExists] = await connection.execute('SELECT id FROM users WHERE username = ?', ['admin']);
    if (userAdminExists.length === 0) {
      await connection.execute(
        `INSERT INTO users (username, password, role, store) VALUES (?, ?, ?, ?)`,
        ['admin', '$2a$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin', '']
      );
      console.log('✅ Admin user created');
    } else {
      console.log('✅ Admin user already exists');
    }

    // Stores must be created manually by admin

    // Check and add department column to users table if it doesn't exist
    try {
      const [columns] = await connection.execute(`
        SELECT COLUMN_NAME 
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
        AND TABLE_NAME = 'users' 
        AND COLUMN_NAME = 'department'
      `);
      
      if (columns.length === 0) {
        await connection.execute(`
          ALTER TABLE users 
          ADD COLUMN department VARCHAR(100) AFTER store,
          ADD INDEX idx_department (department)
        `);
        console.log('✅ Department column added to users table');
      } else {
        console.log('✅ Department column already exists in users table');
      }
    } catch (error) {
      console.log('ℹ️ Error checking/adding department column:', error.message);
    }

    // Update FAQs table with new fields if they don't exist
    try {
      await connection.execute(`
        ALTER TABLE faqs 
        ADD COLUMN IF NOT EXISTS reaction_time INT DEFAULT 3,
        ADD COLUMN IF NOT EXISTS response_time INT DEFAULT 7,
        ADD COLUMN IF NOT EXISTS schedule_hours VARCHAR(255) DEFAULT '08:00-18:00',
        ADD COLUMN IF NOT EXISTS schedule_days VARCHAR(50) DEFAULT 'monday,tuesday,wednesday,thursday,friday,saturday',
        ADD COLUMN IF NOT EXISTS order_position INT DEFAULT 1
      `);
      
      await connection.execute(`
        ALTER TABLE faqs 
        ADD INDEX IF NOT EXISTS idx_order (order_position)
      `);
      
      console.log('✅ FAQs table updated with new fields');
    } catch (error) {
      console.log('ℹ️ FAQs table fields already exist or update error:', error.message);
    }

    // Update contacts table with new fields if they don't exist
    try {
      await connection.execute(`
        ALTER TABLE contacts 
        ADD COLUMN IF NOT EXISTS group_id INT,
        ADD COLUMN IF NOT EXISTS tags VARCHAR(255),
        ADD COLUMN IF NOT EXISTS notes TEXT
      `);
      
      await connection.execute(`
        ALTER TABLE contacts 
        ADD INDEX IF NOT EXISTS idx_group (group_id)
      `);
      
      console.log('✅ Contacts table updated with new fields');
    } catch (error) {
      console.log('ℹ️ Contacts table fields already exist or update error:', error.message);
    }

    // Update mass_messages table with new fields if they don't exist
    try {
      await connection.execute(`
        ALTER TABLE mass_messages 
        ADD COLUMN IF NOT EXISTS send_interval INT DEFAULT 5,
        ADD COLUMN IF NOT EXISTS group_id INT,
        ADD COLUMN IF NOT EXISTS scheduled_at TIMESTAMP NULL
      `);
      
      await connection.execute(`
        ALTER TABLE mass_messages 
        ADD INDEX IF NOT EXISTS idx_scheduled (scheduled_at)
      `);
      
      console.log('✅ Mass_messages table updated with new fields');
    } catch (error) {
      console.log('ℹ️ Mass_messages table fields already exist or update error:', error.message);
    }

    // Insert default bot settings
    const defaultSettings = [
      ['bot_active', 'true'],
      ['auto_reply_groups', 'true'],
      ['bot_name', 'BeZap'],
      ['bot_persona', 'BeZap virtual assistant'],
      ['default_message', 'Hello! I am the BeZap virtual assistant. How can I help you today?'],
      ['business_hours', '08:00-18:00'],
      ['after_hours_message', 'Hello! Our service hours are from 8am to 6pm. Leave your message and we will respond as soon as possible.'],
      ['welcome_message', 'Welcome to BeZap! How can I help you?']
    ];

    for (const [key, value] of defaultSettings) {
      await connection.execute(
        `INSERT IGNORE INTO bot_settings (setting_key, setting_value) VALUES (?, ?)`,
        [key, value]
      );
    }
    console.log('✅ Default bot settings verified/created');

    // FAQs, welcome messages and placeholders must be created manually by admin
    console.log('✅ FAQs, welcome messages and placeholders must be configured by the administrator in the panel');

    // Insert default contact group
    await connection.execute(
      `INSERT IGNORE INTO contact_groups (group_name, description) VALUES (?, ?)`,
      ['General Customers', 'Default group for all customers']
    );
    console.log('✅ Default contact group verified/created');
    
    console.log('✅ Database initialized successfully!');
    
  } catch (error) {
    console.error('❌ Error initializing database:', error);
    throw error;
  } finally {
    if (connection) {
      connection.release();
    }
  }
}

// Function to check database health
async function checkDatabaseHealth() {
  let connection;
  try {
    connection = await pool.getConnection();
    await connection.execute('SELECT 1');
    const tables = ['users', 'stores', 'conversations', 'messages', 'faqs', 'bot_settings'];
    const tableChecks = [];
    for (const table of tables) {
      try {
        const [result] = await connection.execute(`SELECT COUNT(*) as count FROM ${table}`);
        tableChecks.push({ table, count: result[0].count, status: 'OK' });
      } catch (error) {
        tableChecks.push({ table, count: 0, status: 'ERROR', error: error.message });
      }
    }
    console.log('🏥 Database status:');
    console.table(tableChecks);
    return {
      healthy: tableChecks.every(check => check.status === 'OK'),
      tables: tableChecks
    };
  } catch (error) {
    console.error('❌ Error checking database health:', error);
    return { healthy: false, error: error.message };
  } finally {
    if (connection) {
      connection.release();
    }
  }
}

// Users must have stores assigned manually by admin

// Function to backup settings
async function backupSettings() {
  let connection;
  try {
    connection = await pool.getConnection();
    const [botSettings] = await connection.execute('SELECT * FROM bot_settings');
    const [faqs] = await connection.execute('SELECT * FROM faqs');
    const [stores] = await connection.execute('SELECT * FROM stores');
    const backup = {
      timestamp: new Date().toISOString(),
      version: '1.0',
      bot_settings: botSettings,
      faqs: faqs,
      stores: stores
    };
    return backup;
  } catch (error) {
    console.error('❌ Error creating backup:', error);
    throw error;
  } finally {
    if (connection) {
      connection.release();
    }
  }
}

// Function to restore settings
async function restoreSettings(backupData) {
  let connection;
  try {
    connection = await pool.getConnection();
    await connection.beginTransaction();
    
    // Restore bot settings
    if (backupData.bot_settings) {
      for (const setting of backupData.bot_settings) {
        await connection.execute(
          'INSERT INTO bot_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)',
          [setting.setting_key, setting.setting_value]
        );
      }
    }
    
    // Restore FAQs
    if (backupData.faqs) {
      for (const faq of backupData.faqs) {
        await connection.execute(
          'INSERT INTO faqs (question, answer, emoji, placeholder_key, active) VALUES (?, ?, ?, ?, ?)',
          [faq.question, faq.answer, faq.emoji, faq.placeholder_key, faq.active]
        );
      }
    }
    
    await connection.commit();
    console.log('✅ Settings restored successfully!');
    
  } catch (error) {
    if (connection) {
      await connection.rollback();
    }
    console.error('❌ Error restoring settings:', error);
    throw error;
  } finally {
    if (connection) {
      connection.release();
    }
  }
}

module.exports = { 
  pool, 
  initDatabase, 
  checkDatabaseHealth, 
  backupSettings, 
  restoreSettings
};